#include "dailymotionvideolisttab.h"
#include "dailymotionvideoinfowindow.h"
#include "dailymotionplaylistdialog.h"
#include "videolistdelegate.h"
#include "dailymotionvideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/dailymotion.h"
#include "../base/settings.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/urlgrabber.h"
#include "windowstack.h"
#include "../base/notifications.h"
#include "../base/dailymotionvideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>
#include <QTimer>
#include <QMaemo5Style>

DailymotionVideoListTab::DailymotionVideoListTab(QWidget *parent) :
    AbstractTab(parent),
    m_model(new DailymotionVideoListModel(this)),
    m_listView(new ListView(this)),
    m_deleteFromUploadsAction(new QAction(tr("Delete from uploads"), this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_deleteFromFavouritesAction(new QAction(tr("Delete from favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_deleteFromFavouritesContextAction(m_contextMenu->addAction(tr("Delete from favourites"), this, SLOT(deleteVideoFromFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_deleteFromUploadsAction);
    this->addAction(m_addToFavouritesAction);
    this->addAction(m_deleteFromFavouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_addToQueueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_deleteFromUploadsAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromUploads()));
    this->connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    this->connect(m_deleteFromFavouritesAction, SIGNAL(triggered()), this, SLOT(deleteVideosFromFavourites()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    this->connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    this->onOrientationChanged();
}

void DailymotionVideoListTab::showEvent(QShowEvent *event) {
    AbstractTab::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void DailymotionVideoListTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void DailymotionVideoListTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void DailymotionVideoListTab::setVideoFeed(const QUrl &feed) {
    m_model->getVideos(feed);
    this->setMenuActions();
}

void DailymotionVideoListTab::setMenuActions() {
    bool signedIn = Dailymotion::instance()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->feed() == DAILYMOTION_UPLOADS_FEED));
    m_addToFavouritesAction->setEnabled((signedIn) && (m_model->feed() != DAILYMOTION_FAVOURITES_FEED));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->feed() == DAILYMOTION_FAVOURITES_FEED));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->feed() == DAILYMOTION_UPLOADS_FEED));
    m_addToFavouritesContextAction->setVisible((signedIn) && (m_model->feed() != DAILYMOTION_FAVOURITES_FEED));
    m_deleteFromFavouritesContextAction->setVisible((signedIn) && (m_model->feed() == DAILYMOTION_FAVOURITES_FEED));
    m_addToPlaylistContextAction->setVisible(signedIn);
}

void DailymotionVideoListTab::onLoadingChanged(bool loading) {
    Notifications::toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void DailymotionVideoListTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void DailymotionVideoListTab::deleteVideosFromUploads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::addVideosToFavourites() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::deleteVideosFromFavourites() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::addVideosToPlaylist() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::addVideoDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Download"), SLOT(addVideoDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::addAudioDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Download"), SLOT(addAudioDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::queueVideos() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_model, tr("Add to queue"), SLOT(queueVideos()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void DailymotionVideoListTab::deleteVideoFromUploads() {
    Dailymotion::instance()->deleteFromUploads(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListTab::addVideoToFavourites() {
    Dailymotion::instance()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListTab::deleteVideoFromFavourites() {
    Dailymotion::instance()->deleteFromFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListTab::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void DailymotionVideoListTab::addVideoToPlaylist(const QString &id) {
    Dailymotion::instance()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void DailymotionVideoListTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    Dailymotion::instance()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListTab::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListTab::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void DailymotionVideoListTab::queueVideo() {
    PlaybackListModel::playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void DailymotionVideoListTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(video, WindowStack::instance()->currentWindow());
    window->show();
}

void DailymotionVideoListTab::playVideo(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), WindowStack::instance()->currentWindow());
        window->show();
    }
    else {
        UrlGrabber::instance()->getDailymotionVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
