#include "thumbnaillabel.h"
#include "../base/utils.h"
#include <QPainter>
#include <QImage>
#include <QtCore/qmath.h>
#include <QMaemo5Style>

ThumbnailLabel::ThumbnailLabel(QWidget *parent) :
    QWidget(parent),
    m_duration(0),
    m_interactive(true),
    m_pressed(false),
    m_allowClick(true)
{
}

ThumbnailLabel::ThumbnailLabel(int duration, QWidget *parent) :
    QWidget(parent),
    m_duration(duration),
    m_interactive(true),
    m_pressed(false)
{
}

void ThumbnailLabel::setImage(const QImage &image) {
    m_image = image;
    this->repaint();
}

void ThumbnailLabel::setDuration(int duration) {
    m_duration = duration;
    this->repaint();
}

void ThumbnailLabel::mousePressEvent(QMouseEvent *event) {
    if (this->interactive()) {
        event->accept();
        m_pressed = true;
        this->repaint();
    }
    else {
        event->ignore();
    }
}

void ThumbnailLabel::mouseReleaseEvent(QMouseEvent *event) {
    if ((this->interactive()) && (this->rect().contains(event->pos()))) {
        event->accept();
        m_pressed = false;
        this->repaint();

        if (m_allowClick) {
            emit clicked();
            m_allowClick = false;
            this->startTimer(500);
        }
    }
    else {
        event->ignore();
    }
}

void ThumbnailLabel::leaveEvent(QEvent *event) {
    event->accept();
    m_pressed = false;
    this->repaint();
}

void ThumbnailLabel::paintEvent(QPaintEvent *event) {
    Q_UNUSED(event);

    QPainter painter(this);

    if (m_pressed) {
        painter.setOpacity(0.5);
    }

    if (!m_image.isNull()) {
        painter.drawImage(this->rect(), m_image);
    }

    QRect backgroundRect = this->rect();
    backgroundRect.setBottomRight(QPoint(backgroundRect.right() - 1, backgroundRect.bottom() - 1));

    painter.setPen(QMaemo5Style::standardColor("SecondaryTextColor"));
    painter.drawRect(backgroundRect);

    if (m_duration) {
        int backgroundWidth = m_duration >= 3600 ? qFloor(this->width() / 2.5) : qFloor(this->width() / 4);

        backgroundRect.setLeft(backgroundRect.right() - backgroundWidth);
        backgroundRect.setTop(backgroundRect.bottom() - qFloor(this->height() / 6));

        QFont font;
        font.setPixelSize(qFloor(this->height() / 8));

        painter.setOpacity(0.5);
        painter.fillRect(backgroundRect, Qt::black);
        painter.setPen(Qt::white);
        painter.setFont(font);
        painter.setOpacity(1);
        painter.drawText(backgroundRect, Qt::AlignCenter, Utils::durationFromSecs(m_duration));
    }
}

void ThumbnailLabel::timerEvent(QTimerEvent *event) {
    m_allowClick = true;
    event->accept();
    this->killTimer(event->timerId());
}
