# -*- coding: utf-8 -*-

import os
import re
import dbus
from PyQt4.QtGui import *
from PyQt4.QtCore import *
from PyQt4.phonon import Phonon
from PyQt4.QtMaemo5 import QMaemo5InformationBox
from youtubeservice import YouTubeDataService, YouTubeVideoDownloader
from cutetubeconfig import VideoPlaybackSettings, VideoDownloadSettings

class VideoPlaybackWindow(Phonon.VideoPlayer):
    """
    Provides playback of YouTube videos.
    """
    def __init__(self, parent, video = None, playlist = None):
        Phonon.VideoPlayer.__init__(self, parent)
        self.setWindowFlags(Qt.Window)
        self.resize(800, 480)
        self.setAttribute(Qt.WA_Maemo5StackedWindow)
        self.video = video
        self.playlist = playlist
        self.toolBar = QToolBar(self)        
        self.toolBar.setVisible(False)
        self.muteButton = QAction(self.toolBar)
        self.muteButton.setIcon(QIcon("/opt/cutetube/ui-images/vol.png"))
        self.playPauseButton = QAction(self.toolBar)
        self.playPauseButton.setIcon(QIcon("/opt/cutetube/ui-images/pause.png"))
        self.stopButton = QAction(self.toolBar)
        self.stopButton.setIcon(QIcon("/opt/cutetube/ui-images/stop.png"))
        self.previousButton = QAction(self.toolBar)
        self.previousButton.setIcon(QIcon("/opt/cutetube/ui-images/rew.png"))
        self.nextButton = QAction(self.toolBar)
        self.nextButton.setIcon(QIcon("/opt/cutetube/ui-images/ff.png"))
        self.seekBar = Phonon.SeekSlider(self.mediaObject(), self.toolBar)
        self.toolBar.addAction(self.previousButton)
        self.toolBar.addAction(self.playPauseButton)
        self.toolBar.addAction(self.stopButton)
        self.toolBar.addAction(self.nextButton)
        self.toolBar.addWidget(self.seekBar)
        self.toolBar.addAction(self.muteButton)
        self.titleLabel = QLabel(self)
        self.titleLabel.setVisible(False)
        self.titleLabel.setFixedHeight(30)        
        self.layout().addWidget(self.titleLabel)
        self.layout().addWidget(self.toolBar)
        self.mousePressEvent = self.toggleControls
        self.keyPressEvent = self.keyControl
        self.connect(self.muteButton, SIGNAL("triggered()"), self.toggleMute)
        self.connect(self.stopButton, SIGNAL("triggered()"), self.stopAndClose)
        self.connect(self.playPauseButton, SIGNAL("triggered()"), self.playPause)
        self.connect(self.previousButton, SIGNAL("triggered()"), self.playPrevious)
        self.connect(self.nextButton, SIGNAL("triggered()"), self.playNext)
        self.mediaObject().setPrefinishMark(5000)
        self.connect(self.mediaObject(), SIGNAL("prefinishMarkReached(qint32)"), self.queueNextVideo)
        self.connect(self.mediaObject(), SIGNAL("currentSourceChanged(const Phonon::MediaSource&)"), self.displayCurrentVideoInfo)
        self.connect(self.mediaObject(), SIGNAL("finished()"), self.stopAndClose)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.currentVideo = 0
        self.showFullScreen()
        if self.video != None:
            mediasource = self.getMediaSource(self.video)
        else:
            mediasource = self.getMediaSource(self.playlist[0])
        self.play(mediasource)
        self.keepDisplayOn()
        
    def orientationChanged(self):
        """
        This method is used to prevent rotation of the widgets.
        """
        pass
            
        
    def keepDisplayOn(self):
        """
        Makes a request to MCE via dbus to keep the display on.
        """
        bus = dbus.SystemBus()
        mce = bus.get_object("com.nokia.mce", "/com/nokia/mce/request")
        mce.req_display_blanking_pause()
        QTimer.singleShot(9000, self.keepDisplayOn)  #Display timeout can be as low as 10 seconds.
        
    def displayCurrentVideoInfo(self):
        """
        Informs the user when the current video changes. 
        Also sets the title label to the current video title.
        """
        try:
            videoTitle = unicode(self.video.media.title.text, "utf-8")
        except:
            videoTitle = unicode(self.playlist[self.currentVideo].media.title.text, "utf-8")
        self.titleLabel.setText("           %s" % videoTitle)
        QMaemo5InformationBox.information(self, "Playing \'%s\'" % videoTitle)        
        
    def toggleControls(self, event = None):
        """
        Toggles the visibility of the video controls when the user 
        clicks on the screen.
        """
        self.titleLabel.setVisible(not self.titleLabel.isVisible())
        self.toolBar.setVisible(not self.toolBar.isVisible())

    def toggleMute(self):
        
        if self.volume() > 0.0:
            self.setVolume(0.0)
            self.muteButton.setIcon(QIcon("/opt/cutetube/ui-images/volmuted.png"))
        else:
            self.setVolume(1.0)
            self.muteButton.setIcon(QIcon("/opt/cutetube/ui-images/vol.png"))
        
    def keyControl(self, event):
        """
        Control the video playback using the keyboard as follows:
        
        'I': Toggle visibility of controls
        'P' or 'Space': Toggle Play/Pause
        'S' or 'Backspace': Stop playback and close video playback window.
        'Left': Play previous video/seek to beginning as appropriate.
        'Right': Play next video.
        """
        if event.key() in (Qt.Key_P, Qt.Key_Space):
            self.playPause()
        elif event.key() in (Qt.Key_S, Qt.Key_Backspace):
            self.stopAndClose()
        elif event.key() == Qt.Key_Left:
            self.playPrevious()
        elif event.key() == Qt.Key_Right:
            self.playNext()
        elif event.key() == Qt.Key_I:
            self.toggleControls()
            
    def getMediaSource(self, video):
        """
        Returns the filename if the local file exists, 
        otherwise the video URL is obtained and returned.
        """
        filename = "%s/%s.mp4" % (VideoDownloadSettings.downloadFolder, re.sub('[<>:"/|?*\\\\]', "-", video.media.title.text).decode("utf-8"))
        if os.path.exists(filename):
            mediasource = Phonon.MediaSource(filename)
        else:
            videoUrl = YouTubeDataService.getVideoUrl(video, VideoPlaybackSettings.playbackFormat)
            mediasource = Phonon.MediaSource(videoUrl)
        return mediasource
            
    def queueNextVideo(self):
        """
        Queues the next video in the playlist.
        """
        if self.playlist != None:
            try:
                video = self.playlist[self.currentVideo + 1]
                mediasource = self.getMediaSource(video)
                self.mediaObject().enqueue(mediasource)
                self.currentVideo += 1
            except:
                pass # Ignore if this is the last video in the playlist.

    def playPause(self):
        """
        Toggles play/pause of the video.
        """
        if self.isPlaying():
            self.pause()
            self.playPauseButton.setIcon(QIcon("/opt/cutetube/ui-images/play.png"))
        else:
            self.play()
            self.playPauseButton.setIcon(QIcon("/opt/cutetube/ui-images/pause.png"))
            
    def playPrevious(self):
        """
        Plays the previous video in the playlist if the current time < 5 secs, 
        otherwise we seek to the beginning of the current video.
        """
        if self.playlist != None and self.currentTime() <= 5000:
            video = self.playlist[self.currentVideo - 1]
            mediasource = self.getMediaSource(video)
            if self.currentVideo > 0:
                self.currentVideo -= 1
            else:
                self.currentVideo = len(self.playlist) - 1
            self.play(mediasource)
        else:
            self.seek(0)
            
    def playNext(self):
        """
        Plays the next video in the playlist.
        """
        if self.playlist != None:
            try:
                video = self.playlist[self.currentVideo + 1]
                self.currentVideo += 1
            except:
                video = self.playlist[0]
                self.currentVideo = 0
            mediasource = self.getMediaSource(video)
            self.play(mediasource)
            
    def stopAndClose(self):
        """
        Stops the video and closes the window.
        """
        self.stop()
        self.deleteLater()
        self.close()
