/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "tuneinpairlistmodel.h"
#include "tunein.h"
#include "tuneingenrelist.h"
#include "tuneinlanguagelist.h"
#include "tuneinnamelist.h"

TuneInPairListModel::TuneInPairListModel(QObject *parent) :
    QSortFilterProxyModel(parent),
    m_model(new QStandardItemModel(this)),
    m_loading(false),
    m_queryType(Queries::Unknown)
{
    m_roleNames[NameRole] = "name";
    m_roleNames[ValueRole] = "value";
    m_roleNames[QueryMatchRole] = "queryMatch";
    m_roleNames[SectionRole] = "section";
    this->setSourceModel(m_model);
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->setFilterRole(NameRole);
    this->setSortRole(NameRole);
    this->setDynamicSortFilter(true);
}

TuneInPairListModel::~TuneInPairListModel() {}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> TuneInPairListModel::roleNames() const {
    return m_roleNames;
}
#endif

QVariant TuneInPairListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case QueryMatchRole:
        return this->queryMatch(QSortFilterProxyModel::data(index, this->filterRole()).toString());
    case SectionRole:
        return QSortFilterProxyModel::data(index, NameRole).toString().left(1).toUpper();
    default:
        return QSortFilterProxyModel::data(index, role);
    }
}

QVariant TuneInPairListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

bool TuneInPairListModel::loading() const {
    return m_loading;
}

void TuneInPairListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

Queries::QueryType TuneInPairListModel::queryType() const {
    return m_queryType;
}

void TuneInPairListModel::setQueryType(Queries::QueryType queryType) {
    if (queryType != this->queryType()) {
        m_queryType = queryType;
        emit queryTypeChanged(queryType);
    }
}

QString TuneInPairListModel::query() const {
    return m_query;
}

void TuneInPairListModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit queryChanged(query);
    }
}

QString TuneInPairListModel::queryMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#else
        str.replace(re, "<b>" + re.cap() + "</b>");
#endif
    }

    return str;
}

#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
QString TuneInPairListModel::highlightColor() const {
    return m_highlight;
}

void TuneInPairListModel::setHighlightColor(const QString &color) {
    if (color != this->highlightColor()) {
        m_highlight = color;
        emit highlightColorChanged(color);
    }
}
#endif

void TuneInPairListModel::showGenres() {
    this->setLoading(true);
    this->setQueryType(Queries::Genres);

    TuneInGenreList *list = TuneIn::getGenres();
    this->connect(list, SIGNAL(finished(TuneInGenreList*)), this, SLOT(onGenreListFinished(TuneInGenreList*)));
    this->connect(list, SIGNAL(canceled(TuneInGenreList*)), this, SLOT(onGenreListCanceled(TuneInGenreList*)));
}

void TuneInPairListModel::showPodcastGenres() {
    this->setLoading(true);
    this->setQueryType(Queries::PodcastGenres);

    TuneInGenreList *list = TuneIn::getGenres();
    this->connect(list, SIGNAL(finished(TuneInGenreList*)), this, SLOT(onGenreListFinished(TuneInGenreList*)));
    this->connect(list, SIGNAL(canceled(TuneInGenreList*)), this, SLOT(onGenreListCanceled(TuneInGenreList*)));
}

void TuneInPairListModel::showCountries() {
    this->setLoading(true);
    this->setQueryType(Queries::Countries);
    this->addPairs(TuneIn::getCountries());
}

void TuneInPairListModel::showLanguages() {
    this->setLoading(true);
    this->setQueryType(Queries::Languages);

    TuneInLanguageList *list = TuneIn::getLanguages();
    this->connect(list, SIGNAL(finished(TuneInLanguageList*)), this, SLOT(onLanguageListFinished(TuneInLanguageList*)));
    this->connect(list, SIGNAL(canceled(TuneInLanguageList*)), this, SLOT(onLanguageListCanceled(TuneInLanguageList*)));
}

void TuneInPairListModel::showNames(const QString &id, const QString &filter) {
    this->setLoading(true);
    this->setQueryType(Queries::Names);

    TuneInNameList *list = TuneIn::getNames(id, filter);
    this->connect(list, SIGNAL(finished(TuneInNameList*)), this, SLOT(onNameListFinished(TuneInNameList*)));
    this->connect(list, SIGNAL(canceled(TuneInNameList*)), this, SLOT(onNameListCanceled(TuneInNameList*)));
}

void TuneInPairListModel::showPodcastNames(const QString &id, const QString &filter) {
    this->setLoading(true);
    this->setQueryType(Queries::PodcastNames);

    TuneInNameList *list = TuneIn::getNames(id, filter);
    this->connect(list, SIGNAL(finished(TuneInNameList*)), this, SLOT(onNameListFinished(TuneInNameList*)));
    this->connect(list, SIGNAL(canceled(TuneInNameList*)), this, SLOT(onNameListCanceled(TuneInNameList*)));
}

void TuneInPairListModel::clear() {
    m_model->clear();
}

void TuneInPairListModel::reload() {
    this->clear();

    switch (this->queryType()) {
    case Queries::Genres:
        this->showGenres();
        return;
    case Queries::PodcastGenres:
        this->showPodcastGenres();
        return;
    case Queries::Countries:
        this->showCountries();
        return;
    case Queries::Languages:
        this->showLanguages();
        return;
    default:
        return;
    }
}

void TuneInPairListModel::onGenreListFinished(TuneInGenreList *list) {
    switch (list->error()) {
    case QNetworkReply::NoError:
        break;
    default:
        this->setLoading(false);
        emit error(list->errorString());
        list->deleteLater();
        return;
    }

    this->addPairs(list->results());
    list->deleteLater();
}

void TuneInPairListModel::onGenreListCanceled(TuneInGenreList *list) {
    this->setLoading(false);
    list->deleteLater();
}

void TuneInPairListModel::onLanguageListFinished(TuneInLanguageList *list) {
    switch (list->error()) {
    case QNetworkReply::NoError:
        break;
    default:
        this->setLoading(false);
        emit error(list->errorString());
        list->deleteLater();
        return;
    }

    this->addPairs(list->results());
    list->deleteLater();
}

void TuneInPairListModel::onLanguageListCanceled(TuneInLanguageList *list) {
    this->setLoading(false);
    list->deleteLater();
}

void TuneInPairListModel::onNameListFinished(TuneInNameList *list) {
    switch (list->error()) {
    case QNetworkReply::NoError:
        break;
    default:
        this->setLoading(false);
        emit error(list->errorString());
        list->deleteLater();
        return;
    }

    this->addPairs(list->results());
    list->deleteLater();
}

void TuneInPairListModel::onNameListCanceled(TuneInNameList *list) {
    this->setLoading(false);
    list->deleteLater();
}

void TuneInPairListModel::addPairs(QList<QPair<QString, QString> > pairs) {
    for (int i = 0; i < pairs.size(); i++) {
        QStandardItem *item = new QStandardItem(pairs.at(i).first);
        item->setData(pairs.at(i).second);
        m_model->appendRow(item);
    }

    this->setLoading(false);
    emit countChanged(this->rowCount());
}

void TuneInPairListModel::addPair(QPair<QString, QString> pair) {
    QStandardItem *item = new QStandardItem(pair.first);
    item->setData(pair.second);
    m_model->appendRow(item);

    emit countChanged(this->rowCount());
}

void TuneInPairListModel::removePair(const QModelIndex &index) {
    this->removePair(index.row());
}

void TuneInPairListModel::removePair(int row) {
    m_model->removeRow(row);

    emit countChanged(this->rowCount());
}

void TuneInPairListModel::onItemAdded(const QString &name, const QString &value) {
    this->addPair(QPair<QString, QString>(name, value));
}

void TuneInPairListModel::onItemUpdated(const QString &name, const QString &value) {
    QModelIndexList indexes = m_model->match(m_model->index(0, 0), NameRole, name, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        m_model->setData(indexes.first(), value, ValueRole);
    }
}

void TuneInPairListModel::onItemDeleted(const QString &name) {
    QModelIndexList indexes = m_model->match(m_model->index(0, 0), NameRole, name, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removePair(indexes.first().row());
    }
}
