/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QTimerEvent>
#include <QNetworkRequest>
#include "mediaplayer.h"
#include "../../base/streamextractor.h"
#include "../../base/networkaccessmanager.h"
#include "../../base/utils.h"
#include "../../base/database.h"
#include "../../base/settings.h"

#define SLEEP_TIMER_INTERVAL 60000

MediaPlayer* MediaPlayer::self = 0;

MediaPlayer::MediaPlayer(QObject *parent) :
    QObject(parent),
    m_mediaPlayer(new QMediaPlayer(this)),
    m_extractor(new StreamExtractor(this)),
    m_station(0),
    m_state(MediaState::Stopped),
    m_playbackMode(PlaybackMode::Radio),
    m_sleepTimerEnabled(false),
    m_sleepTimerRemaining(60000),
    m_sleepTimerId(0)
{
    if (!self) {
        self = this;
    }

    this->connect(m_mediaPlayer, SIGNAL(seekableChanged(bool)), this, SIGNAL(seekableChanged(bool)));
    this->connect(m_mediaPlayer, SIGNAL(positionChanged(qint64)), this, SIGNAL(positionChanged(qint64)));
    this->connect(m_mediaPlayer, SIGNAL(durationChanged(qint64)), this, SIGNAL(durationChanged(qint64)));
    this->connect(m_mediaPlayer, SIGNAL(bufferStatusChanged(int)), this, SIGNAL(bufferProgressChanged(int)));
    this->connect(m_mediaPlayer, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onStateChanged(QMediaPlayer::State)));
    this->connect(m_mediaPlayer, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged(QMediaPlayer::MediaStatus)));
    this->connect(m_mediaPlayer, SIGNAL(error(QMediaPlayer::Error)), this, SLOT(onMediaError(QMediaPlayer::Error)));
    this->connect(m_mediaPlayer, SIGNAL(metaDataChanged()), this, SIGNAL(metaDataChanged()));
    this->connect(m_mediaPlayer, SIGNAL(volumeChanged(int)), this, SIGNAL(volumeChanged(int)));
    this->connect(m_extractor, SIGNAL(gotStreamUrl(QUrl)), this, SLOT(play(QUrl)));
    this->connect(m_extractor, SIGNAL(error(QString)), this, SLOT(onExtractorError(QString)));
    this->connect(Settings::instance(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
}

MediaPlayer::~MediaPlayer() {}

MediaPlayer* MediaPlayer::instance() {
    return !self ? new MediaPlayer : self;
}

Station* MediaPlayer::currentStation() const {
    return m_station;
}

void MediaPlayer::setCurrentStation(Station *station) {
    if (m_station) {
        m_station->deleteLater();
        m_station = 0;
    }

    if (station) {
        m_station = new Station(station, this);
    }

    emit currentStationChanged(this->currentStation());
}

PlaybackMode::Mode MediaPlayer::playbackMode() const {
    return m_playbackMode;
}

void MediaPlayer::setPlaybackMode(PlaybackMode::Mode playbackMode) {
    if (playbackMode != this->playbackMode()) {
        m_playbackMode = playbackMode;
        emit playbackModeChanged(playbackMode);
    }
}

MediaState::State MediaPlayer::state() const {
    return m_state;
}

void MediaPlayer::setState(MediaState::State state) {
    if (state != this->state()) {
        m_state = state;
        emit stateChanged(state);
    }
}

QString MediaPlayer::stateString() const {
    switch (this->state()) {
    case MediaState::Stopped:
        return tr("Stopped");
    case MediaState::Loading:
        return tr("Loading");
    case MediaState::Buffering:
        return tr("Buffering");
    case MediaState::Playing:
        return tr("Playing");
    case MediaState::Paused:
        return tr("Paused");
    case MediaState::Error:
        return tr("Error");
    default:
        return tr("Unknown");
    }
}

bool MediaPlayer::playing() const {
    return this->state() == MediaState::Playing;
}

bool MediaPlayer::paused() const {
    return this->state() == MediaState::Paused;
}

bool MediaPlayer::stopped() const {
    switch (this->state()) {
    case MediaState::Stopped:
    case MediaState::Error:
        return true;
    default:
        return false;
    }
}

bool MediaPlayer::loading() const {
    return this->state() == MediaState::Loading;
}

bool MediaPlayer::buffering() const {
    return this->state() == MediaState::Buffering;
}

bool MediaPlayer::seekable() const {
    return m_mediaPlayer->isSeekable();
}

QString MediaPlayer::errorString() const {
    return m_errorString;
}

void MediaPlayer::setErrorString(const QString &errorString) {
    m_errorString = errorString;
}

qint64 MediaPlayer::position() const {
    return m_mediaPlayer->position();
}

void MediaPlayer::setPosition(qint64 position) {
    m_mediaPlayer->setPosition(position);
}

qint64 MediaPlayer::duration() const {
    return m_mediaPlayer->duration();
}

int MediaPlayer::bufferProgress() const {
    return m_mediaPlayer->bufferStatus();
}

int MediaPlayer::volume() const {
    return m_mediaPlayer->volume();
}

void MediaPlayer::setVolume(int volume) {
    m_mediaPlayer->setVolume(volume);
}

QString MediaPlayer::title() const {
    return  m_mediaPlayer->metaData(QtMultimediaKit::Title).toString();
}

QString MediaPlayer::artist() const {
    return m_mediaPlayer->metaData(QtMultimediaKit::AlbumArtist).toString();
}

QString MediaPlayer::album() const {
    return m_mediaPlayer->metaData(QtMultimediaKit::AlbumTitle).toString();
}

QString MediaPlayer::genre() const {
    return m_mediaPlayer->metaData(QtMultimediaKit::Genre).toString();
}

QString MediaPlayer::description() const {
    return m_mediaPlayer->metaData(QtMultimediaKit::Description).toString();
}

qint64 MediaPlayer::bitRate() const {
    return m_mediaPlayer->metaData(QtMultimediaKit::AudioBitRate).toLongLong();
}

bool MediaPlayer::sleepTimerEnabled() const {
    return m_sleepTimerEnabled;
}

void MediaPlayer::setSleepTimerEnabled(bool enabled) {
    if (enabled != this->sleepTimerEnabled()) {
        m_sleepTimerEnabled = enabled;
        emit sleepTimerEnabledChanged(enabled);

        if (enabled) {
            int duration = Settings::instance()->sleepTimerDuration();
            this->setSleepTimerRemaining(!duration ? SLEEP_TIMER_INTERVAL : duration * SLEEP_TIMER_INTERVAL);
            this->setSleepTimerId(this->startTimer(SLEEP_TIMER_INTERVAL));
        }
        else {
            this->killTimer(this->sleepTimerId());
        }
    }
}

qint64 MediaPlayer::sleepTimerRemaining() const {
    return m_sleepTimerRemaining;
}

void MediaPlayer::setSleepTimerRemaining(qint64 remaining) {
    if (remaining != this->sleepTimerRemaining()) {
        m_sleepTimerRemaining = remaining;
        emit sleepTimerEnabledChanged(remaining);
    }
}

int MediaPlayer::tickInterval() const {
    return m_mediaPlayer->notifyInterval();
}

void MediaPlayer::setTickInterval(int interval) {
    m_mediaPlayer->setNotifyInterval(interval);
}

void MediaPlayer::play() {
    if ((m_mediaPlayer->media().isNull()) && (this->currentStation())) {
        this->play(this->currentStation()->source());
    }
    else {
        m_mediaPlayer->setNetworkConfigurations(QList<QNetworkConfiguration>()
                                                << NetworkAccessManager::instance()->activeConfiguration());
        m_mediaPlayer->play();
    }
}

void MediaPlayer::play(const QUrl &url) {
    if (url.scheme() == "file") {
        // url is a local file
        this->setPlaybackMode(PlaybackMode::Sequential);
        m_mediaPlayer->setMedia(url);
        m_mediaPlayer->play();
    }
    else {
        this->setPlaybackMode(PlaybackMode::Radio);

        if (Utils::urlIsPlaylist(url)) {
            this->setState(MediaState::Loading);
            m_extractor->getStreamUrl(url);
        }
        else {
            m_mediaPlayer->setNetworkConfigurations(QList<QNetworkConfiguration>()
                                                    << NetworkAccessManager::instance()->activeConfiguration());

            QNetworkRequest request(url);
            request.setRawHeader("Icy-MetaData", "1");
            m_mediaPlayer->setMedia(QMediaContent(request));
            m_mediaPlayer->play();
        }
    }
}

void MediaPlayer::play(Station *station) {
    this->setCurrentStation(station);
    this->play(station->source());
}

void MediaPlayer::pause() {
    m_mediaPlayer->pause();
}

void MediaPlayer::stop() {
    m_mediaPlayer->stop();
    m_extractor->cancelCurrentOperation();
}

void MediaPlayer::clear() {
    this->stop();
    this->setCurrentStation(0);
}

void MediaPlayer::updateSleepTimer() {
    this->setSleepTimerRemaining(this->sleepTimerRemaining() - SLEEP_TIMER_INTERVAL);

    if (!this->sleepTimerRemaining()) {
        this->stop();
    }
}

int MediaPlayer::sleepTimerId() const {
    return m_sleepTimerId;
}

void MediaPlayer::setSleepTimerId(int timerId) {
    m_sleepTimerId = timerId;
}

void MediaPlayer::timerEvent(QTimerEvent *event) {
    if (!this->sleepTimerEnabled()) {
        this->killTimer(event->timerId());
    }
    else {
        this->updateSleepTimer();
    }
}

void MediaPlayer::onStateChanged(QMediaPlayer::State state) {
    switch (state) {
    case QMediaPlayer::PlayingState:
        this->setState(MediaState::Playing);
        this->onStationPlayed();
        break;
    case QMediaPlayer::PausedState:
        this->setState(MediaState::Paused);
        break;
    case QMediaPlayer::StoppedState:
        this->setState(MediaState::Stopped);
        this->setSleepTimerEnabled(false);
        break;
    default:
        break;
    }
}

void MediaPlayer::onMediaStatusChanged(QMediaPlayer::MediaStatus status) {
    switch (status) {
    case QMediaPlayer::LoadingMedia:
        this->setState(MediaState::Loading);
        break;
    case QMediaPlayer::BufferingMedia:
        this->setState(MediaState::Buffering);
        break;
    case QMediaPlayer::BufferedMedia:
        this->setState(MediaState::Playing);
        break;
    default:
        break;
    }
}

void MediaPlayer::onMediaError(QMediaPlayer::Error mediaError) {
    if (m_mediaPlayer->media().isNull()) {
        return;
    }
#ifdef Q_OS_SYMBIAN
    switch (mediaError) {
    case QMediaPlayer::NoError:
        return;
    case QMediaPlayer::ResourceError:
        m_mediaPlayer->setMedia(QMediaContent());
        return;
    case QMediaPlayer::FormatError:
        this->setErrorString(tr("Format not supported"));
        break;
    case QMediaPlayer::NetworkError:
        this->setErrorString(tr("Network error"));
        break;
    default:
        this->setErrorString(m_mediaPlayer->errorString());
        break;
    }

    m_mediaPlayer->setMedia(QMediaContent());
    this->setState(MediaState::Error);
    this->setSleepTimerEnabled(false);
    emit error(this->errorString());
#else
    switch (mediaError) {
    case QMediaPlayer::NoError:
        return;
    case QMediaPlayer::ResourceError:
        this->setErrorString(tr("Cannot find media resource"));
        break;
    case QMediaPlayer::FormatError:
        this->setErrorString(tr("Format not supported"));
        break;
    case QMediaPlayer::NetworkError:
        this->setErrorString(tr("Network error"));
        break;
    default:
        this->setErrorString(m_mediaPlayer->errorString());
        break;
    }

    m_mediaPlayer->setMedia(QMediaContent());
    this->setState(MediaState::Error);
    this->setSleepTimerEnabled(false);
    emit error(this->errorString());
#endif
}

void MediaPlayer::onExtractorError(const QString &errorString) {
    this->setErrorString(errorString);
    this->setState(MediaState::Error);
    emit error(errorString);
}

void MediaPlayer::onStationPlayed() {
    if ((this->currentStation()) && (this->currentStation()->service() == Services::NoService)) {
        Database::asyncUpdateStation(this->currentStation()->id(), "lastPlayed", QDateTime::currentMSecsSinceEpoch(), false);
    }
}

void MediaPlayer::onMediaPlayerChanged(const QString &mediaPlayer) {
    if (mediaPlayer != "cuteradio") {
        this->clear();
    }
}
