/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLineEdit>
#include <QTextEdit>
#include <QCheckBox>
#include <QLabel>
#include <QRegExpValidator>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QMaemo5InformationBox>
#include "tuneinaddstationdialog.h"
#include "scrollarea.h"
#include "../tunein/tunein.h"
#include "../tunein/tuneinstation.h"
#include "../base/database.h"
#include "../base/logodownloader.h"
#include "../base/station.h"
#include "../base/utils.h"

TuneInAddStationDialog::TuneInAddStationDialog(const QString &id, QWidget *parent) :
    AbstractDialog(parent),
    m_station(0),
    m_titleEdit(new QLineEdit(this)),
    m_genreEdit(new QLineEdit(this)),
    m_countryEdit(new QLineEdit(this)),
    m_languageEdit(new QLineEdit(this)),
    m_sourceEdit(new QLineEdit(this)),
    m_descriptionEdit(new QTextEdit(this)),
    m_favouriteCheckBox(new QCheckBox(tr("Favourite?"), this)),
    m_scrollArea(new ScrollArea(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Ok, Qt::Vertical, this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Add station"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(new QLabel(tr("Title"), this));
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(new QLabel(tr("Description"), this));
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(new QLabel(tr("Genre"), this));
    vbox->addWidget(m_genreEdit);
    vbox->addWidget(new QLabel(tr("Country"), this));
    vbox->addWidget(m_countryEdit);
    vbox->addWidget(new QLabel(tr("Language"), this));
    vbox->addWidget(m_languageEdit);
    vbox->addWidget(new QLabel(tr("Source"), this));
    vbox->addWidget(m_sourceEdit);
    vbox->addWidget(m_favouriteCheckBox);

    m_titleEdit->setValidator(new QRegExpValidator(QRegExp("^\\S.+"), m_titleEdit));
    m_sourceEdit->setValidator(new QRegExpValidator(QRegExp("^http(s|)://\\S+"), m_sourceEdit));

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

    this->connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleChanged()));
    this->connect(m_sourceEdit, SIGNAL(textChanged(QString)), this, SLOT(onSourceChanged()));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(addStation()));

    this->onOrientationChanged();
    this->onTitleChanged();
    this->toggleBusy(true);

    TuneInStation *station = TuneIn::getStation(id);
    station->setParent(this);
    this->connect(station, SIGNAL(finished(TuneInStation*)), this, SLOT(onStationFinished(TuneInStation*)));
}

void TuneInAddStationDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void TuneInAddStationDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
}

void TuneInAddStationDialog::toggleBusy(bool busy) {
    AbstractDialog::toggleBusy(busy);
    m_scrollArea->setEnabled(!busy);
    m_buttonBox->setEnabled(!busy);
}

void TuneInAddStationDialog::onStationFinished(TuneInStation *station) {
    this->toggleBusy(false);

    m_station = station->result();

    if (m_station) {
        m_station->setParent(this);
        m_titleEdit->setText(m_station->title());
        m_descriptionEdit->setText(m_station->description());
        m_genreEdit->setText(m_station->genre());
        m_countryEdit->setText(m_station->country());
        m_languageEdit->setText(m_station->language());
        m_sourceEdit->setText(m_station->source().toString());
    }
    else {
        switch (station->error()) {
        case QNetworkReply::NoError:
            QMaemo5InformationBox::information(0, tr("No station information found"));
            break;
        default:
            QMaemo5InformationBox::information(0, station->errorString());
            break;
        }
    }

    station->deleteLater();
}

void TuneInAddStationDialog::onTitleChanged() {
    m_buttonBox->setEnabled((m_titleEdit->hasAcceptableInput()) && (m_sourceEdit->hasAcceptableInput()));
}

void TuneInAddStationDialog::onSourceChanged() {
    m_buttonBox->setEnabled((m_titleEdit->hasAcceptableInput()) && (m_sourceEdit->hasAcceptableInput()));
}

void TuneInAddStationDialog::addStation() {
    this->toggleBusy(true);

    LogoDownloader *downloader = new LogoDownloader(m_station->logo(), QString("%1%2.%3").arg(Utils::logoPath()).arg(m_station->id()).arg(m_station->logo().toString().section('.', -1)), this);
    this->connect(downloader, SIGNAL(finished(LogoDownloader*)), this, SLOT(onLogoDownloaderFinished(LogoDownloader*)));
    downloader->start();
}

void TuneInAddStationDialog::onLogoDownloaderFinished(LogoDownloader *downloader) {
    Database::asyncAddStation(m_titleEdit->text(),
                              m_descriptionEdit->toPlainText(),
                              QUrl(downloader->localPath()),
                              m_genreEdit->text(),
                              m_countryEdit->text(),
                              m_languageEdit->text(),
                              m_sourceEdit->text(),
                              m_favouriteCheckBox->isChecked());

    this->toggleBusy(false);
    this->accept();
}
