/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLabel>
#include <QDialogButtonBox>
#include <QSpinBox>
#include <QLineEdit>
#include <QScrollArea>
#include <QGridLayout>
#include "settingsdialog.h"
#include "valueselector.h"
#include "networkproxydialog.h"
#include "../base/settings.h"
#include "../base/selectionmodels.h"

SettingsDialog::SettingsDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_orientationSelector(new ValueSelector(tr("Screen orientation"), this)),
    m_languageSelector(new ValueSelector(tr("Language"), this)),
    m_mediaPlayerSelector(new ValueSelector(tr("Media player"), this)),
    m_mediaPlayerCommandEdit(new QLineEdit(this)),
    m_proxyButton(new QPushButton(tr("Network proxy"), this)),
    m_sleepTimerSpinBox(new QSpinBox(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Save, Qt::Vertical, this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this)),
    m_proxyDialog(0)
{
    this->setWindowTitle(tr("Settings"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_languageSelector);
    vbox->addWidget(m_orientationSelector);
    vbox->addWidget(m_mediaPlayerSelector);
    vbox->addWidget(m_mediaPlayerCommandEdit);
    vbox->addWidget(m_proxyButton);
    vbox->addWidget(new QLabel(tr("Sleep timer duration (mins)")));
    vbox->addWidget(m_sleepTimerSpinBox);

    m_orientationSelector->setModel(new ScreenOrientationModel(m_orientationSelector));
    m_languageSelector->setModel(new LanguageModel(m_languageSelector));
    m_mediaPlayerSelector->setModel(new MediaPlayerModel(m_mediaPlayerSelector));

    m_mediaPlayerCommandEdit->setPlaceholderText(tr("Custom command"));

    m_sleepTimerSpinBox->setMaximum(999);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

    this->connect(m_mediaPlayerSelector, SIGNAL(valueChanged(QVariant)), this, SLOT(onMediaPlayerChanged(QVariant)));
    this->connect(m_proxyButton, SIGNAL(clicked()), this, SLOT(showNetworkProxyDialog()));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));

    this->loadSettings();
    this->onOrientationChanged();
}

void SettingsDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void SettingsDialog::setPortraitLayout() {
    this->setFixedHeight(440);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
}

void SettingsDialog::loadSettings() {
    m_languageSelector->setValue(Settings::instance()->language());
    m_orientationSelector->setValue(Settings::instance()->screenOrientation());
    m_mediaPlayerSelector->setValue(Settings::instance()->mediaPlayer());
    m_mediaPlayerCommandEdit->setText(Settings::instance()->mediaPlayerCommand());
    m_mediaPlayerCommandEdit->setEnabled(Settings::instance()->mediaPlayer() == "other");
    m_sleepTimerSpinBox->setValue(Settings::instance()->sleepTimerDuration());
}

void SettingsDialog::saveSettings() {
    Settings::instance()->setLanguage(m_languageSelector->currentValue().toString());
    Settings::instance()->setScreenOrientation(static_cast<ScreenOrientation::Orientation>(m_orientationSelector->currentValue().toInt()));
    Settings::instance()->setMediaPlayer((m_mediaPlayerSelector->currentValue().toString() == "other")
                                         && (m_mediaPlayerCommandEdit->text().isEmpty())
                                         ? "cuteradio"
                                         : m_mediaPlayerSelector->currentValue().toString());
    Settings::instance()->setMediaPlayerCommand(m_mediaPlayerCommandEdit->text());
    Settings::instance()->setSleepTimerDuration(m_sleepTimerSpinBox->value());

    this->accept();
}

void SettingsDialog::onMediaPlayerChanged(const QVariant &player) {
    m_mediaPlayerCommandEdit->setEnabled(player.toString() == "other");
}

void SettingsDialog::showNetworkProxyDialog() {
    if (!m_proxyDialog) {
        m_proxyDialog = new NetworkProxyDialog(this);
    }

    m_proxyDialog->open();
}
