/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.models 1.0
import com.cuteradio.items 1.0
import com.cuteradio.enums 1.0

MyPage {
    id: root

    function searchPodcasts(query) {
        podcastModel.searchPodcasts(query);
    }

    function showPodcasts() {
        podcastModel.showPodcasts();
    }

    tools: ToolBarLayout {

        BackToolButton {}

        NowPlayingButton {}
    }

    MyListView {
        id: view

        property int selectedIndex: -1

        anchors.fill: parent
        model: PodcastListModel {
            id: podcastModel
        }
        delegate: PodcastDelegate {
            onClicked: {
                switch (service) {
                case Services.TuneIn:
                    appWindow.pageStack.push(Qt.resolvedUrl("TuneInPodcastEpisodeListPage.qml"), { title: title }).showPodcastEpisodes(source);
                    return;
                default:
                    appWindow.pageStack.push(Qt.resolvedUrl("PodcastEpisodeListPage.qml"), { title: title }).showPodcastEpisodes(source);
                    return;
                }
            }

            onPressAndHold: {
                view.selectedIndex = -1;
                view.selectedIndex = index;
                contextMenu.open();
            }
        }
        section.delegate: SectionDelegate {
            text: section
        }
        section.property: "section"
        section.criteria: ViewSection.FirstCharacter
    }

    MySectionScroller {
        listView: view
    }

    Label {
        anchors {
            fill: parent
            margins: platformStyle.paddingLarge
        }
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        wrapMode: Text.WordWrap
        color: platformStyle.colorNormalMid
        font.bold: true
        font.pixelSize: 32
        text: qsTr("No podcasts found")
        visible: (!podcastModel.count) && (!podcastModel.loading)
    }

    BusyIndicator {
        anchors.centerIn: parent
        width: 64
        height: 64
        visible: podcastModel.loading
        running: visible
    }

    ContextMenu {
        id: contextMenu

        MenuLayout {

            MenuItem {
                text: qsTr("Edit")
                onClicked: {
                    dialogLoader.sourceComponent = editPodcastDialog;
                    dialogLoader.item.loadPodcast(podcastModel.get(view.selectedIndex));
                    dialogLoader.item.open();
                }
            }

            MenuItem {
                text: qsTr("Delete")
                onClicked: {
                    dialogLoader.sourceComponent = confirmDeleteDialog;
                    dialogLoader.item.open();
                }
            }
        }

        onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
    }

    Loader {
        id: dialogLoader
    }

    Component {
        id: editPodcastDialog

        EditPodcastDialog {
            onStatusChanged: {
                switch (status) {
                case DialogStatus.Closed:
                    if (root.status === PageStatus.Active) view.forceActiveFocus();
                    return;
                default:
                    return;
                }
            }
        }
    }

    Component {
        id: confirmDeleteDialog

        MyQueryDialog {
            titleText: qsTr("Delete podcast?")
            icon: "images/info.png"
            message: qsTr("Do you wish to delete the podcast") + " '" + podcastModel.data(view.selectedIndex, "title") + "' " + qsTr("from the database?")
            acceptButtonText: qsTr("Yes")
            rejectButtonText: qsTr("No")
            onAccepted: Database.asyncDeletePodcast(podcastModel.data(view.selectedIndex, "id"))
            onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) view.forceActiveFocus();
        }
    }
}
