/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "tuneinstationlistmodel.h"
#include "../tunein/tunein.h"
#include "../tunein/tuneinstationlist.h"
#ifndef QML_USER_INTERFACE
#include "../thumbnailcache/thumbnailcache.h"
#endif

TuneInStationListModel::TuneInStationListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    #endif
    m_loading(false),
    m_queryType(Queries::Unknown)
{
    m_roleNames[ServiceRole] = "service";
    m_roleNames[IdRole] = "id";
    m_roleNames[TitleRole] = "title";
    m_roleNames[DescriptionRole] = "description";
    m_roleNames[LogoRole] = "logo";
    m_roleNames[GenreRole] = "genre";
    m_roleNames[CountryRole] = "country";
    m_roleNames[LanguageRole] = "language";
    m_roleNames[SourceRole] = "source";
    m_roleNames[FavouriteRole] = "favourite";
    m_roleNames[LastPlayedRole] = "lastPlayed";
    m_roleNames[SectionRole] = "section";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

TuneInStationListModel::~TuneInStationListModel() {
    this->clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> TuneInStationListModel::roleNames() const {
    return m_roleNames;
}
#endif

int TuneInStationListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant TuneInStationListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case ServiceRole:
        return m_list.at(index.row())->service();
    case IdRole:
        return m_list.at(index.row())->id();
    case TitleRole:
        return m_list.at(index.row())->title();
    case DescriptionRole:
        return m_list.at(index.row())->description();
    case LogoRole:
#ifndef QML_USER_INTERFACE
        return m_cache->thumbnail(m_list.at(index.row())->logo(), QSize(64, 64));
#else
        return m_list.at(index.row())->logo();
#endif
    case GenreRole:
        return m_list.at(index.row())->genre();
    case CountryRole:
        return m_list.at(index.row())->country();
    case LanguageRole:
        return m_list.at(index.row())->language();
    case SourceRole:
        return m_list.at(index.row())->source();
    case FavouriteRole:
        return m_list.at(index.row())->favourite();
    case LastPlayedRole:
        return m_list.at(index.row())->lastPlayed();
    case SectionRole:
        return m_list.at(index.row())->title().left(1).toUpper();
    default:
        return QVariant();
    }
}

QVariant TuneInStationListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool TuneInStationListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if ((!this->rowCount()) || (!index.isValid())) {
        return false;
    }

    switch (role) {
    case IdRole:
    case LastPlayedRole:
    case SectionRole:
        break;
    default:
        if (Station *station = this->get(index.row())) {
            return station->setProperty(this->roleNames().value(role), value);
        }
    }

    return false;
}

bool TuneInStationListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return this->setData(this->index(row), value, this->roleNames().key(role));
}

bool TuneInStationListModel::moreResults() const {
    return !m_nextUrl.isEmpty();
}

QUrl TuneInStationListModel::nextUrl() const {
    return m_nextUrl;
}

void TuneInStationListModel::setNextUrl(const QUrl &url) {
    m_nextUrl = url;
}

bool TuneInStationListModel::loading() const {
    return m_loading;
}

void TuneInStationListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QString TuneInStationListModel::searchQuery() const {
    return m_searchQuery;
}

void TuneInStationListModel::setSearchQuery(const QString &query) {
    if (query != this->searchQuery()) {
        m_searchQuery = query;
        emit searchQueryChanged(query);
    }
}

Queries::QueryType TuneInStationListModel::queryType() const {
    return m_queryType;
}

void TuneInStationListModel::setQueryType(Queries::QueryType type) {
    if (type != this->queryType()) {
        m_queryType = type;
        emit queryTypeChanged(type);
    }
}

Station* TuneInStationListModel::get(const QModelIndex &index) const {
    return this->get(index.row());
}

Station* TuneInStationListModel::get(int row) const {
    return (row >= 0) && (row < m_list.size()) ? m_list.at(row) : 0;
}

void TuneInStationListModel::searchStations(const QString &query) {
    this->setLoading(true);
    this->setSearchQuery(query);
    this->setQueryType(Queries::StationSearch);

    TuneInStationList *list = TuneIn::searchStations(query);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInStationList*)), this, SLOT(onStationListFinished(TuneInStationList*)));
    this->connect(list, SIGNAL(canceled(TuneInStationList*)), this, SLOT(onStationListCanceled(TuneInStationList*)));
}

void TuneInStationListModel::showStationsByGenre(const QString &id) {
    this->setLoading(true);
    this->setSearchQuery(id);
    this->setQueryType(Queries::Genres);

    TuneInStationList *list = TuneIn::getStationsByGenre(id);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInStationList*)), this, SLOT(onStationListFinished(TuneInStationList*)));
    this->connect(list, SIGNAL(canceled(TuneInStationList*)), this, SLOT(onStationListCanceled(TuneInStationList*)));
}

void TuneInStationListModel::showStationsByCountry(const QString &id) {
    this->setLoading(true);
    this->setSearchQuery(id);
    this->setQueryType(Queries::Countries);

    TuneInStationList *list = TuneIn::getStationsByCountry(id);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInStationList*)), this, SLOT(onStationListFinished(TuneInStationList*)));
    this->connect(list, SIGNAL(canceled(TuneInStationList*)), this, SLOT(onStationListCanceled(TuneInStationList*)));
}

void TuneInStationListModel::showStations(const QString &url) {
    this->setLoading(true);
    this->setSearchQuery(url);
    this->setQueryType(Queries::Unknown);

    TuneInStationList *list = TuneIn::getStations(url);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInStationList*)), this, SLOT(onStationListFinished(TuneInStationList*)));
    this->connect(list, SIGNAL(canceled(TuneInStationList*)), this, SLOT(onStationListCanceled(TuneInStationList*)));
}

void TuneInStationListModel::getMoreResults() {
    if ((this->moreResults()) && (!this->loading())) {
        this->setLoading(true);

        TuneInStationList *list = TuneIn::getStations(this->nextUrl());
        list->setParent(this);
        this->connect(list, SIGNAL(finished(TuneInStationList*)), this, SLOT(onStationListFinished(TuneInStationList*)));
        this->connect(list, SIGNAL(canceled(TuneInStationList*)), this, SLOT(onStationListCanceled(TuneInStationList*)));
    }
}

void TuneInStationListModel::clear() {
    this->beginResetModel();
    qDeleteAll(m_list);
    m_list.clear();
    this->endResetModel();
}

void TuneInStationListModel::reload() {
    this->clear();

    switch (this->queryType()) {
    case Queries::StationSearch:
        this->searchStations(this->searchQuery());
        return;
    case Queries::Genres:
        this->showStationsByGenre(this->searchQuery());
        return;
    case Queries::Countries:
        this->showStationsByCountry(this->searchQuery());
        return;
    default:
        this->showStations(this->searchQuery());
        return;
    }
}

void TuneInStationListModel::onStationListFinished(TuneInStationList *list) {
    switch (list->error()) {
    case QNetworkReply::NoError:
        break;
    default:
        this->setLoading(false);
        emit error(list->errorString());
        list->deleteLater();
        return;
    }

    this->setNextUrl(list->nextUrl());
    this->addStations(list->results());
    list->deleteLater();
}

void TuneInStationListModel::onStationListCanceled(TuneInStationList *list) {
    this->setLoading(false);
    list->deleteLater();
}

void TuneInStationListModel::addStations(QList<Station *> stations) {
    if (stations.isEmpty()) {
        this->setLoading(false);
        return;
    }

    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount() + stations.size() - 1);
    m_list << stations;
    this->endInsertRows();
    emit countChanged(this->rowCount());

    foreach (Station *station, stations) {
        this->connect(station, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
        this->connect(station, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
    }

    this->setLoading(false);
}

void TuneInStationListModel::addStation(Station *station) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(station);
    this->endInsertRows();
    emit countChanged(this->rowCount());

    this->connect(station, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
    this->connect(station, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
}

void TuneInStationListModel::insertStation(int row, Station *station) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginInsertRows(QModelIndex(), row, row);
        m_list.insert(row, station);
        this->endInsertRows();
        emit countChanged(this->rowCount());

        this->connect(station, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onStationUpdated(QString,QVariantMap)));
        this->connect(station, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onStationUpdated(QString,QString,QVariant)));
    }
}

void TuneInStationListModel::removeStation(int row) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row)->deleteLater();
        this->endRemoveRows();
        emit countChanged(this->rowCount());
    }
}

void TuneInStationListModel::removeStation(int role, const QVariant &value) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removeStation(indexes.first().row());
    }
}

void TuneInStationListModel::onStationUpdated(const QString &id, const QVariantMap &properties) {
    Q_UNUSED(properties)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        switch (this->queryType()) {
        case Queries::Genres:
            if (indexes.first().data(GenreRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Countries:
            if (indexes.first().data(CountryRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        default:
            break;
        }

        emit dataChanged(indexes.first(), indexes.first());
    }
}

void TuneInStationListModel::onStationUpdated(const QString &id, const QString &property, const QVariant &value) {
    Q_UNUSED(property)
    Q_UNUSED(value)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        switch (this->queryType()) {
        case Queries::Genres:
            if (indexes.first().data(GenreRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        case Queries::Countries:
            if (indexes.first().data(CountryRole).toString() != this->searchQuery()) {
                this->removeStation(indexes.first().row());
                return;
            }

            break;
        default:
            break;
        }

        emit dataChanged(indexes.first(), indexes.first());
    }
}

void TuneInStationListModel::onStationDeleted(const QString &id) {
    this->removeStation(IdRole, id);
}

#ifndef QML_USER_INTERFACE
void TuneInStationListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif
