/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLineEdit>
#include <QTextEdit>
#include <QCheckBox>
#include <QLabel>
#include <QDialogButtonBox>
#include <QGridLayout>
#include "tuneinaddpodcastdialog.h"
#include "scrollarea.h"
#include "../base/station.h"
#include "../base/database.h"
#include "../base/logodownloader.h"
#include "../base/utils.h"

TuneInAddPodcastDialog::TuneInAddPodcastDialog(Station *podcast, QWidget *parent) :
    AbstractDialog(parent),
    m_podcast(podcast),
    m_titleEdit(new QLineEdit(podcast->title(), this)),
    m_sourceEdit(new QLineEdit(podcast->source().toString(), this)),
    m_descriptionEdit(new QTextEdit(podcast->description(), this)),
    m_scrollArea(new ScrollArea(this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Ok, Qt::Vertical, this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Add podcast"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(new QLabel(tr("Title"), this));
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(new QLabel(tr("Description"), this));
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(new QLabel(tr("Source"), this));
    vbox->addWidget(m_sourceEdit);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Preferred);

    m_buttonBox->setEnabled((!m_titleEdit->text().isEmpty()) && (!m_sourceEdit->text().isEmpty()));

    this->connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleChanged(QString)));
    this->connect(m_sourceEdit, SIGNAL(textChanged(QString)), this, SLOT(onSourceChanged(QString)));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(addPodcast()));

    this->onOrientationChanged();
}

void TuneInAddPodcastDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void TuneInAddPodcastDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0);
    }
}

void TuneInAddPodcastDialog::onTitleChanged(const QString &title) {
    m_buttonBox->setEnabled((!title.isEmpty()) && (!m_sourceEdit->text().isEmpty()));
}

void TuneInAddPodcastDialog::onSourceChanged(const QString &source) {
    m_buttonBox->setEnabled((!m_titleEdit->text().isEmpty()) && (!source.isEmpty()));
}

void TuneInAddPodcastDialog::addPodcast() {
    if (m_podcast->logo().isEmpty()) {
        Database::asyncAddPodcast(m_titleEdit->text(),
                                  m_descriptionEdit->toPlainText(),
                                  QUrl(),
                                  m_sourceEdit->text(),
                                  Services::TuneIn);

        this->accept();
    }
    else {
        this->toggleBusy(true);

        LogoDownloader *downloader = new LogoDownloader(m_podcast->logo(), QString("%1%2.%3").arg(Utils::logoPath()).arg(m_podcast->id()).arg(m_podcast->logo().toString().section('.', -1)), this);
        this->connect(downloader, SIGNAL(finished(LogoDownloader*)), this, SLOT(onLogoDownloaderFinished(LogoDownloader*)));
        downloader->start();
    }
}

void TuneInAddPodcastDialog::onLogoDownloaderFinished(LogoDownloader *downloader) {
    Database::asyncAddPodcast(m_titleEdit->text(),
                              m_descriptionEdit->toPlainText(),
                              QUrl(downloader->localPath()),
                              m_sourceEdit->text(),
                              Services::TuneIn);

    this->toggleBusy(false);
    this->accept();
}
