/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */


#include <QDomDocument>
#include <QDomElement>
#include <QDateTime>
#include "podcastepisodelist.h"

PodcastEpisodeList::PodcastEpisodeList(QNetworkReply *reply, QObject *parent) :
    QObject(parent),
    m_reply(reply)
{
    this->connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

PodcastEpisodeList::~PodcastEpisodeList() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

int PodcastEpisodeList::count() const {
    return m_results.size();
}

QList<Station*> PodcastEpisodeList::results() const {
    return m_results;
}

QNetworkReply::NetworkError PodcastEpisodeList::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString PodcastEpisodeList::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

void PodcastEpisodeList::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void PodcastEpisodeList::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    QDomDocument doc;
    doc.setContent(m_reply->readAll());

    QDomElement channel = doc.documentElement().firstChildElement("channel");
    QString logo = channel.firstChildElement("itunes:image").attribute("href");
    QString genre = channel.firstChildElement("itunes:category").attribute("text");
    QDomNodeList results = channel.elementsByTagName("item");

    qsrand(QDateTime::currentMSecsSinceEpoch());

    for (int i = 0; i < results.size(); i++) {
        QDomElement result = results.at(i).toElement();

        Station *podcast = new Station;
        podcast->setStationType(Stations::Podcast);
        podcast->setId(QString("p%1").arg(qrand()));
        podcast->setTitle(result.firstChildElement("title").text());
        podcast->setDescription(result.firstChildElement("description").text());
        podcast->setLogo(logo);
        podcast->setGenre(genre);
        podcast->setSource(result.firstChildElement("enclosure").attribute("url"));
        m_results.append(podcast);
    }

    emit finished(this);
}
