/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */


#include <QDomDocument>
#include <QDomElement>
#include <QDateTime>
#include "podcast.h"
#include "networkaccessmanager.h"

Podcast::Podcast(QObject *parent) :
    QObject(parent),
    m_reply(0),
    m_result(0)
{
}

Podcast::Podcast(QNetworkReply *reply, QObject *parent) :
    QObject(parent),
    m_reply(reply),
    m_result(0)
{
    this->connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

Podcast::~Podcast() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

Station* Podcast::result() const {
    return m_result;
}

void Podcast::setResult(Station *result) {
    m_result = result;
}

QNetworkReply::NetworkError Podcast::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString Podcast::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

void Podcast::load(const QUrl &url) {
    if (m_reply) {
        m_reply->deleteLater();
        m_reply = 0;
    }

    m_reply = NetworkAccessManager::instance()->get(QNetworkRequest(url));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

void Podcast::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void Podcast::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    QDomDocument doc;
    doc.setContent(m_reply->readAll());

    QDomElement result = doc.documentElement().firstChildElement("channel");

    if (!result.isNull()) {
        Station *podcast = new Station;
        qsrand(QDateTime::currentMSecsSinceEpoch());
        podcast->setStationType(Stations::Podcast);
        podcast->setId(QString("p%1").arg(qrand()));
        podcast->setTitle(result.firstChildElement("title").text());
        podcast->setDescription(result.firstChildElement("description").text());
        podcast->setLogo(result.firstChildElement("itunes:image").attribute("href"));
        podcast->setGenre(result.firstChildElement("itunes:category").attribute("text"));
        podcast->setSource(m_reply->url());
        this->setResult(podcast);
    }

    emit finished(this);
}
