/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "pairlistmodel.h"
#include "database.h"

PairListModel::PairListModel(QObject *parent) :
    QSortFilterProxyModel(parent),
    m_model(new QStandardItemModel(this)),
    m_loading(false),
    m_queryType(Queries::Unknown)
{
    m_roleNames[NameRole] = "name";
    m_roleNames[CountRole] = "count";
    m_roleNames[QueryMatchRole] = "queryMatch";
    m_roleNames[SectionRole] = "section";
    this->setSourceModel(m_model);
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->setFilterRole(NameRole);
    this->setSortRole(NameRole);
    this->setDynamicSortFilter(true);
}

PairListModel::~PairListModel() {}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> PairListModel::roleNames() const {
    return m_roleNames;
}
#endif

QVariant PairListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case QueryMatchRole:
        return this->queryMatch(QSortFilterProxyModel::data(index, this->filterRole()).toString());
    case SectionRole:
        return QSortFilterProxyModel::data(index, NameRole).toString().left(1).toUpper();
    default:
        return QSortFilterProxyModel::data(index, role);
    }
}

QVariant PairListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

bool PairListModel::loading() const {
    return m_loading;
}

void PairListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

Queries::QueryType PairListModel::queryType() const {
    return m_queryType;
}

void PairListModel::setQueryType(Queries::QueryType queryType) {
    if (queryType != this->queryType()) {
        m_queryType = queryType;
        emit queryTypeChanged(queryType);

        this->disconnect(Database::instance(), 0, this, 0);

        switch (this->queryType()) {
            case Queries::Genres:
            this->connect(Database::instance(), SIGNAL(genreAdded(QString,int)), this, SLOT(onItemAdded(QString,int)));
            this->connect(Database::instance(), SIGNAL(genreUpdated(QString,int)), this, SLOT(onItemUpdated(QString,int)));
            this->connect(Database::instance(), SIGNAL(genreDeleted(QString)), this, SLOT(onItemDeleted(QString)));
            return;
        case Queries::Countries:
            this->connect(Database::instance(), SIGNAL(countryAdded(QString,int)), this, SLOT(onItemAdded(QString,int)));
            this->connect(Database::instance(), SIGNAL(countryUpdated(QString,int)), this, SLOT(onItemUpdated(QString,int)));
            this->connect(Database::instance(), SIGNAL(countryDeleted(QString)), this, SLOT(onItemDeleted(QString)));
            return;
        case Queries::Languages:
            this->connect(Database::instance(), SIGNAL(languageAdded(QString,int)), this, SLOT(onItemAdded(QString,int)));
            this->connect(Database::instance(), SIGNAL(languageUpdated(QString,int)), this, SLOT(onItemUpdated(QString,int)));
            this->connect(Database::instance(), SIGNAL(languageDeleted(QString)), this, SLOT(onItemDeleted(QString)));
            return;
        default:
            return;
        }
    }
}

QString PairListModel::query() const {
    return m_query;
}

void PairListModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit queryChanged(query);
    }
}

QString PairListModel::queryMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#else
        str.replace(re, "<b>" + re.cap() + "</b>");
#endif
    }

    return str;
}

#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
QString PairListModel::highlightColor() const {
    return m_highlight;
}

void PairListModel::setHighlightColor(const QString &color) {
    if (color != this->highlightColor()) {
        m_highlight = color;
        emit highlightColorChanged(color);
    }
}
#endif

void PairListModel::showGenres() {
    this->setLoading(true);
    this->setQueryType(Queries::Genres);
    this->connect(Database::instance(), SIGNAL(gotGenres(QList<QPair<QString,int> >)), this, SLOT(addPairs(QList<QPair<QString,int> >)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetGenres", Qt::QueuedConnection);
}

void PairListModel::showCountries() {
    this->setLoading(true);
    this->setQueryType(Queries::Countries);
    this->connect(Database::instance(), SIGNAL(gotCountries(QList<QPair<QString,int> >)), this, SLOT(addPairs(QList<QPair<QString,int> >)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetCountries", Qt::QueuedConnection);
}

void PairListModel::showLanguages() {
    this->setLoading(true);
    this->setQueryType(Queries::Languages);
    this->connect(Database::instance(), SIGNAL(gotLanguages(QList<QPair<QString,int> >)), this, SLOT(addPairs(QList<QPair<QString,int> >)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetLanguages", Qt::QueuedConnection);
}

void PairListModel::clear() {
    m_model->clear();
}

void PairListModel::reload() {
    this->clear();

    switch (this->queryType()) {
    case Queries::Genres:
        this->showGenres();
        return;
    case Queries::Countries:
        this->showCountries();
        return;
    case Queries::Languages:
        this->showLanguages();
        return;
    default:
        return;
    }
}

void PairListModel::addPairs(QList<QPair<QString, int> > pairs) {
    for (int i = 0; i < pairs.size(); i++) {
        QStandardItem *item = new QStandardItem(pairs.at(i).first);
        item->setData(pairs.at(i).second);
        m_model->appendRow(item);
    }

    this->setLoading(false);
    emit countChanged(this->rowCount());
}

void PairListModel::addPair(QPair<QString, int> pair) {
    QStandardItem *item = new QStandardItem(pair.first);
    item->setData(pair.second);
    m_model->appendRow(item);

    emit countChanged(this->rowCount());
}

void PairListModel::removePair(const QModelIndex &index) {
    this->removePair(index.row());
}

void PairListModel::removePair(int row) {
    m_model->removeRow(row);

    emit countChanged(this->rowCount());
}

void PairListModel::onItemAdded(const QString &name, int count) {
    this->addPair(QPair<QString, int>(name, count));
}

void PairListModel::onItemUpdated(const QString &name, int count) {
    QModelIndexList indexes = m_model->match(m_model->index(0, 0), NameRole, name, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        m_model->setData(indexes.first(), count, CountRole);
    }
}

void PairListModel::onItemDeleted(const QString &name) {
    QModelIndexList indexes = m_model->match(m_model->index(0, 0), NameRole, name, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removePair(indexes.first().row());
    }
}
