/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef DATABASE_H
#define DATABASE_H

#include <QObject>
#include <QStringList>
#include "station.h"

class Database : public QObject
{
    Q_OBJECT

public:
    explicit Database(QObject *parent = 0);
    ~Database();

    static Database* instance();
    
public slots:
    // Stations
    bool addStation(const QString &title,
                    const QString &description,
                    const QUrl &logo,
                    const QString &genre,
                    const QString &country,
                    const QString &language,
                    const QUrl &source,
                    bool favourite) const;
    bool updateStation(const QString &id, const QVariantMap &properties, bool notify = true) const;
    bool updateStation(const QString &id, const QString &property, const QVariant &value, bool notify = true) const;
    bool deleteStation(const QString &id) const;

    static bool asyncAddStation(const QString &title,
                    const QString &description,
                    const QUrl &logo,
                    const QString &genre,
                    const QString &country,
                    const QString &language,
                    const QUrl &source,
                    bool favourite);
    static bool asyncUpdateStation(const QString &id, const QVariantMap &properties, bool notify = true);
    static bool asyncUpdateStation(const QString &id, const QString &property, const QVariant &value, bool notify = true);
    static bool asyncDeleteStation(const QString &id);

    QList<Station*> getRecentlyPlayedStations() const;
    QList<Station*> getFavouriteStations() const;
    QList<Station*> getStationsById(const QStringList &ids) const;
    QList<Station*> getStationsByTitle(const QString &title) const;
    QList<Station*> getStationsByGenre(const QString &genre) const;
    QList<Station*> getStationsByCountry(const QString &country) const;
    QList<Station*> getStationsByLanguage(const QString &language) const;

    void asyncGetRecentlyPlayedStations() const;
    void asyncGetFavouriteStations() const;
    void asyncGetStationsById(const QStringList &ids) const;
    void asyncGetStationsByTitle(const QString &title) const;
    void asyncGetStationsByGenre(const QString &genre) const;
    void asyncGetStationsByCountry(const QString &country) const;
    void asyncGetStationsByLanguage(const QString &language) const;

    QList< QPair<QString, int> > getGenres() const;
    QList< QPair<QString, int> > getCountries() const;
    QList< QPair<QString, int> > getLanguages() const;

    void asyncGetGenres() const;
    void asyncGetCountries() const;
    void asyncGetLanguages() const;

    // Podcasts
    bool addPodcast(const QString &title,
                    const QString &description,
                    const QUrl &logo,
                    const QUrl &source,
                    int service = Services::NoService) const;
    bool updatePodcast(const QString &id, const QVariantMap &properties, bool notify = true) const;
    bool updatePodcast(const QString &id, const QString &property, const QVariant &value, bool notify = true) const;
    bool deletePodcast(const QString &id) const;

    static bool asyncAddPodcast(const QString &title,
                    const QString &description,
                    const QUrl &logo,
                    const QUrl &source,
                    int service = Services::NoService);
    static bool asyncUpdatePodcast(const QString &id, const QVariantMap &properties, bool notify = true);
    static bool asyncUpdatePodcast(const QString &id, const QString &property, const QVariant &value, bool notify = true);
    static bool asyncDeletePodcast(const QString &id);

    QList<Station*> getPodcasts() const;
    QList<Station*> getPodcastsById(const QStringList &ids) const;
    QList<Station*> getPodcastsByTitle(const QString &title) const;

    void asyncGetPodcasts() const;
    void asyncGetPodcastsById(const QStringList &ids) const;
    void asyncGetPodcastsByTitle(const QString &title) const;

private:
    // Stations
    bool addGenre(const QString &genre) const;
    bool updateGenreStationCount(const QString &genre) const;
    bool deleteGenre(const QString &genre) const;

    bool addCountry(const QString &country) const;
    bool updateCountryStationCount(const QString &country) const;
    bool deleteCountry(const QString &country) const;

    bool addLanguage(const QString &language) const;
    bool updateLanguageStationCount(const QString &language) const;
    bool deleteLanguage(const QString &language) const;

    int getStationCount(const QString &property, const QVariant &value) const;

signals:
    void alert(const QString &message) const;
    void error(const QString &errorString) const;

    // Stations
    void gotStations(const QList<Station*> &stations) const;
    void gotGenres(const QList< QPair<QString, int> > &genres) const;
    void gotCountries(const QList< QPair<QString, int> > &countries) const;
    void gotLanguages(const QList< QPair<QString, int> > &languages) const;

    void stationAdded() const;
    void stationUpdated(const QString &id, const QVariantMap &properties) const;
    void stationUpdated(const QString &id, const QString &property, const QVariant &value) const;
    void stationDeleted(const QString &id) const;

    void genreAdded(const QString &name, int stationCount) const;
    void genreUpdated(const QString &genre, int stationCount) const;
    void genreDeleted(const QString &genre) const;

    void countryAdded(const QString &name, int stationCount) const;
    void countryUpdated(const QString &country, int stationCount) const;
    void countryDeleted(const QString &country) const;

    void languageAdded(const QString &name, int stationCount) const;
    void languageUpdated(const QString &language, int stationCount) const;
    void languageDeleted(const QString &language) const;

    // Podcasts
    void gotPodcasts(const QList<Station*> &podcasts) const;

    void podcastAdded() const;
    void podcastUpdated(const QString &id, const QVariantMap &properties) const;
    void podcastUpdated(const QString &id, const QString &property, const QVariant &value) const;
    void podcastDeleted(const QString &id) const;

private:
    static Database *self;
};

#endif // DATABASE_H
