/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.meego 1.0
import com.cuteradio.items 1.0
import "file:///usr/lib/qt4/imports/com/nokia/meego/UIConstants.js" as UI

MyPage {
    id: root

    function loadStation(stationItem) {
        station.loadStation(stationItem);
    }

    title: !station.title ? qsTr("Station info") : station.title
    tools: ToolBarLayout {

        BackToolIcon {}

        ToolIcon {
            iconSource: station.favourite ? "images/favourite-" + Settings.activeColorString + ".png"
                                          : "images/favourite-white.png"
            onClicked: Database.asyncUpdateStation(station.id, "favourite", station.favourite ? 0 : 1, false)
        }

        ToolIcon {
            platformIconId: "toolbar-edit"
            onClicked: {
                loader.sourceComponent = editStationDialog;
                loader.item.loadStation(station);
                loader.item.open();
            }
        }

        ToolIcon {
            platformIconId: "toolbar-delete"
            onClicked: {
                loader.sourceComponent = confirmDeleteDialog;
                loader.item.open();
            }
        }
    }

    Station {
        id: station
    }

    Flickable {
        id: flicker

        anchors {
            fill: parent
            margins: UI.PADDING_DOUBLE
        }
        contentHeight: column.height + UI.PADDING_DOUBLE

        Column {
            id: column

            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }
            spacing: UI.PADDING_DOUBLE

            Logo {
                width: 120
                height: 120
                image: station.logo
                text: station.title
                enabled: (!MediaPlayer.currentStation) || (MediaPlayer.currentStation.id != station.id)
                onClicked: appWindow.playStation(station)
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                text: station.description
            }

            SeparatorLabel {
                width: parent.width
                text: qsTr("Properties")
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Genre") + ": " + station.genre
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Country") + ": " + station.country
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Language") + ": " + station.language
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Source") + ": " + station.source
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: UI.FONT_SMALL
                text: qsTr("Last played") + ": " + Utils.dateFromMSecs(station.lastPlayed)
                visible: station.lastPlayed > 0
            }
        }
    }

    ScrollDecorator {
        flickableItem: flicker
    }

    Loader {
        id: loader
    }

    Component {
        id: editStationDialog

        AddStationDialog {}
    }

    Component {
        id: confirmDeleteDialog

        QueryDialog {
            titleText: qsTr("Delete station?")
            icon: "images/info.png"
            message: qsTr("Do you wish to delete the station") + " '" + station.title + "' " + qsTr("from the database?")
            acceptButtonText: qsTr("Yes")
            rejectButtonText: qsTr("No")
            onAccepted: Database.asyncDeleteStation(station.id)
        }
    }

    Connections {
        target: Database
        onStationDeleted: if ((id === station.id) && (status === PageStatus.Active)) appWindow.pageStack.pop();
    }
}
