/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QRegExp>
#include <QDomDocument>
#include <QDomElement>
#include "utils.h"
#include "streamextractor.h"

#define MAX_RETRIES 2

StreamExtractor::StreamExtractor(QObject *parent) :
    QObject(parent),
    m_nam(new QNetworkAccessManager(this)),
    m_retries(MAX_RETRIES)
{
}

StreamExtractor::StreamExtractor(QNetworkAccessManager *manager, QObject *parent) :
    QObject(parent),
    m_nam(!manager ? new QNetworkAccessManager(this) : manager)
{
}

StreamExtractor::~StreamExtractor() {}

void StreamExtractor::cancelCurrentOperation() {
    emit currentOperationCancelled();
}

int StreamExtractor::retries() const {
    return m_retries;
}

void StreamExtractor::setRetries(int retries) {
    m_retries = retries;
}

void StreamExtractor::getStreamUrl(const QUrl &url) {
    this->setRetries(MAX_RETRIES);
    QNetworkRequest request(url);
    QNetworkReply *reply = m_nam->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseResponse()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void StreamExtractor::parseResponse() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    switch (reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        reply->deleteLater();
        return;
    default:
        emit error(reply->errorString());
        reply->deleteLater();
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        this->getStreamUrl(redirect);
    }
    else {
        QString response(reply->readAll());
        QString format = reply->url().toString().section('.', -1).toLower();
        QUrl url(this->getUrlFromPlaylistFile(response, format));

        if (url.isEmpty()) {
            emit error(tr("Cannot find stream url"));
        }
        else {
            if ((Utils::urlIsPlaylist(url)) && (this->retries())) {
                this->setRetries(this->retries() - 1);
                this->getStreamUrl(url);
            }
            else {
                emit gotStreamUrl(url);
            }
        }
    }

    reply->deleteLater();
}

QString StreamExtractor::getUrlFromPlaylistFile(const QString &response, const QString &format) {
    if (format == "asx") {
        return this->getUrlFromASXFile(response);
    }
    else if (format == "pls") {
        return this->getUrlFromPLSFile(response);
    }
    else if (format == "m3u") {
        return this->getUrlFromM3UFile(response);
    }
    else if (format == "smil") {
        return this->getUrlFromSMILFile(response);
    }
    else {
        return this->getUrlFromUnknownFile(response);
    }
}

QString StreamExtractor::getUrlFromASXFile(const QString &response) {
    QDomDocument doc;
    doc.setContent(response.toLower());
    QDomNode node = doc.documentElement().namedItem("entry");

    return node.firstChildElement("ref").attribute("href");
}

QString StreamExtractor::getUrlFromPLSFile(const QString &response) {
    return response.section(QRegExp("File\\d=", Qt::CaseInsensitive), 1, 1).section(QRegExp("\\s"), 0, 0);
}

QString StreamExtractor::getUrlFromM3UFile(const QString &response) {
    QRegExp re("http(s|)://\\S+", Qt::CaseInsensitive);

    return re.indexIn(response) >= 0 ? re.cap() : QString();
}

QString StreamExtractor::getUrlFromSMILFile(const QString &response) {
    QDomDocument doc;
    doc.setContent(response.toLower());
    QDomNode node = doc.documentElement().namedItem("body");

    return node.firstChildElement("audio").attribute("src");
}

QString StreamExtractor::getUrlFromUnknownFile(const QString &response) {
    QRegExp re("http(s|)://[^\\s\"'<>]+", Qt::CaseInsensitive);

    return re.indexIn(response) >= 0 ? re.cap() : QString();
}
