/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QMenuBar>
#include <QVBoxLayout>
#include <QLabel>
#include <QAction>
#include <QMessageBox>
#include "stationinfowindow.h"
#include "separatorlabel.h"
#include "listview.h"
#include "nowplayingwindow.h"
#include "nowplayingbutton.h"
#include "scrollarea.h"
#include "addstationdialog.h"
#include "../base/utils.h"
#include "../base/database.h"
#include "../base/station.h"
#include "../mediaplayer/phonon/mediaplayer.h"

StationInfoWindow::StationInfoWindow(Station *station, QWidget *parent) :
    StackedWindow(parent),
    m_station(station),
    m_playAction(this->menuBar()->addAction(tr("Play"), this, SLOT(playStation()))),
    m_favouriteAction(this->menuBar()->addAction(station->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this, SLOT(toggleFavourite()))),
    m_editAction(this->menuBar()->addAction(tr("Edit"), this, SLOT(showEditDialog()))),
    m_deleteAction(this->menuBar()->addAction(tr("Delete"), this, SLOT(deleteStation()))),
    m_descriptionLabel(new QLabel(this)),
    m_genreLabel(new QLabel(this)),
    m_countryLabel(new QLabel(this)),
    m_languageLabel(new QLabel(this)),
    m_sourceLabel(new QLabel(this)),
    m_lastPlayedLabel(new QLabel(this)),
    m_scrollArea(new ScrollArea(this))
{
    this->setWindowTitle(m_station->title());
    this->setCentralWidget(m_scrollArea);

    m_descriptionLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->addWidget(m_descriptionLabel);
    vbox->addWidget(new SeparatorLabel(tr("Properties"), this));
    vbox->addWidget(m_genreLabel);
    vbox->addWidget(m_countryLabel);
    vbox->addWidget(m_languageLabel);
    vbox->addWidget(m_sourceLabel);
    vbox->addWidget(m_lastPlayedLabel);
    vbox->addSpacerItem(new QSpacerItem(10, 10, QSizePolicy::Expanding, QSizePolicy::Expanding));

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setMonitorScrolling(true);

    this->connect(m_scrollArea, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_scrollArea, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    this->connect(m_station, SIGNAL(titleChanged(QString)), this, SLOT(onTitleChanged(QString)));
    this->connect(m_station, SIGNAL(descriptionChanged(QString)), this, SLOT(onDescriptionChanged(QString)));
    this->connect(m_station, SIGNAL(genreChanged(QString)), this, SLOT(onGenreChanged(QString)));
    this->connect(m_station, SIGNAL(countryChanged(QString)), this, SLOT(onCountryChanged(QString)));
    this->connect(m_station, SIGNAL(languageChanged(QString)), this, SLOT(onLanguageChanged(QString)));
    this->connect(m_station, SIGNAL(sourceChanged(QUrl)), this, SLOT(onSourceChanged(QUrl)));
    this->connect(m_station, SIGNAL(lastPlayedChanged(qint64)), this, SLOT(onLastPlayedChanged(qint64)));
    this->connect(m_station, SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));
    this->connect(MediaPlayer::instance(), SIGNAL(currentStationChanged(Station*)), this, SLOT(onCurrentStationChanged(Station*)));

    this->onDescriptionChanged(m_station->description());
    this->onGenreChanged(m_station->genre());
    this->onCountryChanged(m_station->country());
    this->onLanguageChanged(m_station->language());
    this->onSourceChanged(m_station->source());
    this->onLastPlayedChanged(m_station->lastPlayed());
    this->onCurrentStationChanged(MediaPlayer::instance()->currentStation());
    this->onOrientationChanged();
}

void StationInfoWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void StationInfoWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void StationInfoWindow::playStation() {
    MediaPlayer::instance()->play(m_station);
}

void StationInfoWindow::toggleFavourite() {
    Database::asyncUpdateStation(m_station->id(), "favourite", m_station->favourite() ? 0 : 1);
}

void StationInfoWindow::showEditDialog() {
    AddStationDialog *dialog = new AddStationDialog(m_station, this);
    dialog->open();
}

void StationInfoWindow::deleteStation() {
    if ((QMessageBox::question(this, tr("Delete station?"), tr("Do you wish to delete the station '%1' from the database?")
                               .arg(m_station->title()), QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes) && (Database::asyncDeleteStation(m_station->id()))) {
        this->close();
    }
}

void StationInfoWindow::onTitleChanged(const QString &title) {
    this->setWindowTitle(title);
}

void StationInfoWindow::onDescriptionChanged(const QString &description) {
    m_descriptionLabel->setText(description.isEmpty() ? tr("No description") : description);
}

void StationInfoWindow::onGenreChanged(const QString &genre) {
    m_genreLabel->setText(QString("%1: %2").arg(tr("Genre")).arg(genre.isEmpty() ? tr("Unknown") : genre));
}

void StationInfoWindow::onCountryChanged(const QString &country) {
    m_countryLabel->setText(QString("%1: %2").arg(tr("Country")).arg(country.isEmpty() ? tr("Unknown") : country));
}

void StationInfoWindow::onLanguageChanged(const QString &language) {
    m_languageLabel->setText(QString("%1: %2").arg(tr("Language")).arg(language.isEmpty() ? tr("Unknown") : language));
}

void StationInfoWindow::onSourceChanged(const QUrl &source) {
    m_sourceLabel->setText(QString("%1: %2").arg(tr("Source")).arg(source.toString()));
}

void StationInfoWindow::onLastPlayedChanged(qint64 lastPlayed) {
    m_lastPlayedLabel->setText(QString("%1: %2").arg(tr("Last played")).arg(lastPlayed > 0 ? Utils::dateFromMSecs(lastPlayed) : m_station->lastPlayedString()));
}

void StationInfoWindow::onFavouriteChanged(bool favourite) {
    m_favouriteAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void StationInfoWindow::onCurrentStationChanged(Station *station) {
    m_playAction->setEnabled((!station) || (station->id() != m_station->id()));
}
