/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QSettings>
#include "stationfilterlistmodel.h"

StationFilterListModel::StationFilterListModel(QObject *parent) :
    QSortFilterProxyModel(parent),
    m_model(new StationListModel(this))
  #if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
  ,m_highlight(QSettings().value("Appearance/activeColor", "#0881cb").toString())
  #endif
{


    m_roleNames = m_model->roleNames();
    m_roleNames[QueryMatchRole] = "queryMatch";
    this->setSourceModel(m_model);
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->setFilterRole(StationListModel::TitleRole);
    this->setDynamicSortFilter(true);

    this->connect(m_model, SIGNAL(countChanged(int)), this, SIGNAL(countChanged(int)));
    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SIGNAL(loadingChanged(bool)));
    this->connect(m_model, SIGNAL(searchQueryChanged(QString)), this, SIGNAL(searchQueryChanged(QString)));
    this->connect(m_model, SIGNAL(queryTypeChanged(Queries::QueryType)), this, SLOT(onQueryTypeChanged(Queries::QueryType)));
}

StationFilterListModel::~StationFilterListModel() {}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> StationFilterListModel::roleNames() const {
    return m_roleNames;
}
#endif

QVariant StationFilterListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case QueryMatchRole:
        return this->queryMatch(QSortFilterProxyModel::data(index, this->filterRole()).toString());
    default:
        return QSortFilterProxyModel::data(index, role);
    }
}

QVariant StationFilterListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

bool StationFilterListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return QSortFilterProxyModel::setData(this->index(row, 0), value, this->roleNames().key(role));
}

bool StationFilterListModel::loading() const {
    return m_model->loading();
}

QString StationFilterListModel::searchQuery() const {
    return m_model->searchQuery();
}

Queries::QueryType StationFilterListModel::queryType() const {
    return m_model->queryType();
}

QString StationFilterListModel::query() const {
    return m_query;
}

void StationFilterListModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit queryChanged(query);
    }
}

QString StationFilterListModel::queryMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#else
        str.replace(re, "<b>" + re.cap() + "</b>");
#endif
    }

    return str;
}

#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
QString StationFilterListModel::highlightColor() const {
    return m_highlight;
}

void StationFilterListModel::setHighlightColor(const QString &color) {
    if (color != this->highlightColor()) {
        m_highlight = color;
        emit highlightColorChanged(color);
    }
}
#endif

void StationFilterListModel::onQueryTypeChanged(Queries::QueryType queryType) {
    switch (queryType) {
    case Queries::RecentlyPlayedStations:
    case Queries::Recordings:
        this->setSortRole(StationListModel::LastPlayedRole);
        this->sort(0, Qt::DescendingOrder);
        break;
    default:
        this->setSortRole(StationListModel::TitleRole);
        this->sort(0, Qt::AscendingOrder);
        break;
    }

    emit queryTypeChanged(queryType);
}

Station* StationFilterListModel::get(const QModelIndex &index) const {
    return m_model->get(this->mapToSource(index));
}

Station* StationFilterListModel::get(int row) const {
    return m_model->get(this->mapToSource(this->index(row, 0)).row());
}

void StationFilterListModel::searchStations(const QString &query) const {
    m_model->searchStations(query);
}

void StationFilterListModel::showRecentlyPlayedStations() const {
    m_model->showRecentlyPlayedStations();
}

void StationFilterListModel::showFavouriteStations() const {
    m_model->showFavouriteStations();
}

void StationFilterListModel::showStationsByGenre(const QString &genre) const {
    m_model->showStationsByGenre(genre);
}

void StationFilterListModel::showStationsByCountry(const QString &country) const {
    m_model->showStationsByCountry(country);
}

void StationFilterListModel::showStationsByLanguage(const QString &language) const {
    m_model->showStationsByLanguage(language);
}

void StationFilterListModel::clear() {
    m_model->clear();
}

void StationFilterListModel::reload() {
    m_model->reload();
}
