/***********************************************************************
 **
 **   airfield.cpp
 **
 **   This file is part of Cumulus.
 **
 ************************************************************************
 **
 **   Copyright (c):  2000      by Heiner Lamprecht, Florian Ehinger
 **                   2008-2009 by Axel Pauli
 **
 **   This file is distributed under the terms of the General Public
 **   Licence. See the file COPYING for more information.
 **
 **   $Id: airfield.cpp 3376 2009-05-05 20:24:02Z axel $
 **
 ***********************************************************************/

#include "airfield.h"
#include "reachablelist.h"
#include "map.h"

Airfield::Airfield( const QString& name,
                    const QString& icao,
                    const QString& shortName,
                    const BaseMapElement::objectType typeId,
                    const WGSPoint& wgsPos,
                    const QPoint& pos,
                    const Runway& rw,
                    const unsigned int elevation,
                    const QString& frequency,
                    const QString comment,
                    bool winch,
                    bool towing,
                    bool landable )
    : SinglePoint(name, shortName, typeId, wgsPos, pos, elevation),
    icao(icao),
    frequency(frequency),
    comment(comment),
    rwData(rw),
    winch(winch),
    towing(towing),
    landable(landable)
{
  // calculate the default runway shift in 1/10 degrees.
  rwShift = 90/10; // default direction is 90 degrees

  // calculate the real runway shift in 1/10 degrees.
  if ( rwData.direction/256 <= 36 )
    {
      rwShift = (rwData.direction/256 >= 18 ? (rwData.direction/256)-18 : rwData.direction/256);
    }
}

Airfield::~Airfield()
{
}

QString Airfield::getInfoString() const
  {
    QString text, elev;
    QString path = "cumulus/";

    elev = Altitude::getText(elevation,true,0).replace(QRegExp("\\s"),"&nbsp;");

    text = "<HTML><TABLE BORDER=0><TR><TD>"
           "<IMG SRC=" + path + glConfig->getPixmapName(typeID) + "></TD>"
           "<TD>" + name;
    if (!icao.isEmpty())
      text += " (" + icao + ")";

    text += "<FONT SIZE=-1><BR><BR>" + elev;

    if (!frequency.isEmpty())
      text += "&nbsp;/&nbsp;" + frequency + "&nbsp;Mhz.";

    text += "&nbsp;&nbsp;</FONT></TD></TR></TABLE></HTML>";

    return text;
  }

bool Airfield::drawMapElement( QPainter* targetP )
{
  if ( ! isVisible() )
    {
      curPos = QPoint(-5000, -5000);
      return false;
    }

  extern MapConfig* _globalMapConfig;
  extern MapMatrix* _globalMapMatrix;

  int scale = _globalMapMatrix->getScaleRatio()/50;

  //qDebug("Airfield::drawMapElement(): scale: %d %d",scale, _globalMapMatrix->getScaleRatio()  );
  QColor col = ReachableList::getReachColor( wgsPosition );

  curPos = glMapMatrix->map(position);

  // draw also the small dot's in reachability color
  targetP->setPen(QPen(col, 2));

  int iconSize = 32;
  int xOffset  = 16;
  int yOffset  = 16;
  int cxOffset = 16;
  int cyOffset = 16;

  if( typeID == BaseMapElement::Outlanding )
   {
    // The lower end of the beacon shall directly point to the point at the map.
    xOffset  = 16;
    yOffset  = 32;
    cxOffset = 16;
    cyOffset = 16;
  }

  if ( _globalMapConfig->useSmallIcons() )
    {
      iconSize = 16;
      xOffset  = 8;
      yOffset  = 8;
      cxOffset = 8;
      cyOffset = 8;

      if( typeID == BaseMapElement::Outlanding )
        {
          // The lower end of the beacon shall directly point to the
          // point at the map.
          xOffset  = 8;
          yOffset  = 16;
          cxOffset = 8;
          cyOffset = 8;
        }
    }

  if ( !glMapMatrix->isSwitchScale2() )
    {
      targetP->drawEllipse(curPos.x(), curPos.y(), scale, scale );
    }
  else
    {
      if (col == Qt::green)
        {
          targetP->drawPixmap(curPos.x() - cxOffset, curPos.y() - cyOffset,
                              glConfig->getGreenCircle(iconSize));
        }
      else if (col == Qt::magenta)
        {
          targetP->drawPixmap(curPos.x() - cxOffset, curPos.y() - cyOffset,
                              glConfig->getMagentaCircle(iconSize));
        }

      if ( glConfig->isRotatable( typeID ) )
        {
          QPixmap image( glConfig->getPixmapRotatable(typeID, winch) );
          targetP->drawPixmap(curPos.x() - xOffset, curPos.y() - yOffset, image,
                              rwShift*iconSize, 0, iconSize, iconSize);
        }
      else
        {
          QPixmap image( glConfig->getPixmap(typeID) );
          targetP->drawPixmap(curPos.x() - xOffset, curPos.y() - yOffset, image  );
        }
    }

  return true;
}
