/***********************************************************************
**
**   elevationcolorimage.cpp
**
**   This file is part of Cumulus.
**
************************************************************************
**
**   Copyright (c):  2009-2012 Axel Pauli, axel@kflog.org
**
**   This file is distributed under the terms of the General Public
**   License. See the file COPYING for more information.
**
**   $Id: elevationcolorimage.cpp 5002 2012-03-14 14:14:27Z axel $
**
***********************************************************************/

#include <QtGui>

#include "elevationcolorimage.h"
#include "mapdefaults.h"
#include "altitude.h"

// Definition of different font sizes
#define PixelSize 10
#define PointSize  8

/** A reference to the terrain color array has to be passed. The colors
 *  from the array are taken for the elevation color bars. Update first
 *  colors in the array before a new paintEvent is fired.
 */
ElevationColorImage::ElevationColorImage( QColor colors[], QWidget *parent ) :
  QWidget(parent)
{
  terrainColors = colors;
}

ElevationColorImage::~ElevationColorImage()
{
}

QSize ElevationColorImage::sizeHint() const
{
  return QSize(160, 480);
}

QSize ElevationColorImage::minimumSizeHint() const
{
  // Minimum height of one color bar should be 8 pixels plus
  // 10 pixels as reserve.

  QFont myFont = font();

  if( myFont.pointSize() != -1 )
    {
      myFont.setPointSize( PointSize );
    }
  else
    {
      myFont.setPixelSize( PixelSize );
    }

  QFontMetrics fm( myFont );

  int tw = fm.width( "10000ft" );

  return QSize( 70 + tw + 10, 51*8 + 10 );
}

void ElevationColorImage::paintEvent( QPaintEvent * /* event */ )
{
  QPainter painter(this);
  int x, y;
  int altitude;
  QString text;
  QString unit = ( Altitude::getUnit() == Altitude::meters ) ? "m" : "ft";

  // calculate height per color bar in pixels
  int barHeight = height() / 51;

  if( barHeight > 10 )
    {
      // limit height to 10 pixels, if higher
      barHeight = 10;
    }

  // set horizontal start point
  x = 10;

  // center vertical start point
  y = ( height() - barHeight*51 ) / 2;

  // move null point of painter to new position
  painter.translate( QPoint(x, y) );

  // set font size used for text painting
  QFont newFont = font();

  if( newFont.pointSize() != -1 )
    {
      newFont.setPointSize( PointSize );
    }
  else
    {
      newFont.setPixelSize( PixelSize );
    }

  painter.setFont( newFont );
  QPen pen;
  pen.setWidth(3);
  painter.setPen(pen);

  // Draw vertical color bars
  for( int i = 0; i < SIZEOF_TERRAIN_COLORS; i++ )
    {
      painter.fillRect( 0, i*barHeight, 50, barHeight, terrainColors[SIZEOF_TERRAIN_COLORS-1-i] );
    }

  // Draw vertical scale line
  painter.drawLine( QLine( 60, 0, 60, barHeight*51) );

  // Draw bottom line
  y = barHeight*51;
  painter.drawLine( QLine( 55, y, 65, y ) );

  // Draw Arrows at top
  painter.drawLine( QLine( 60, 0, 55, 15 ) );
  painter.drawLine( QLine( 60, 0, 65, 15 ) );

  // Draw right scale bars for zero
  y = barHeight*50 - barHeight/2;
  painter.drawLine( QLine( 60, y, 65, y ) );

  // Draw scale text for zero bar
  text = "0";
  y = barHeight*50;
  painter.drawText( 70, y, text );

  int idx[11]    = {45, 42, 37, 32, 28, 24, 20, 16, 12, 8, 4};
  int height[11] = {100, 250, 500, 1000, 2000, 3000, 4000, 5000, 6000, 7000, 8000};

  for( int i = 0; i < 11; i++ )
    {
      // Draw right scale bars for ...m
      y = barHeight * idx[i]- barHeight/2;
      painter.drawLine( QLine( 60, y, 65, y ) );

      // Draw scale text for ...m bar
      altitude = ( Altitude::getUnit() == Altitude::meters ) ? height[i] : (int) (height[i] * 3.2808);
      text = QString::number(altitude) + unit;
      y = barHeight * idx[i];
      painter.drawText( 70, y, text );
    }
}
