# -*- coding: utf-8 -*-
# Copyright (C) 2009 by Daniel Martin Yerga <dyerga@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# CasualServices: Search and share your favourite places

"""Module to support the API from 11870.com.

Exported Classes:

API -- Class to access to 11870.com functions

"""
from hashlib import md5
import os
import httplib2
import feedparser

APPKEY='15603af70eb87424050b23c3a91c5d9a'
SECRET='a08e802b035966d4e4c1e7f5e0a5a58f'

#URLs for authentication/validation etc.
TEMPTOKENURL = 'http://11870.com/manage-api/temp-token?appToken=%s&authSign=%s'
AUTHURL = 'http://11870.com/manage-api/create-token?tempToken=%s&privilege=write'
AUTHTOKENURL = 'http://11870.com/manage-api/auth-token?tempToken=%s'

#URLs used for searches:
#URLs without authentication
SEARCHURL="http://11870.com/api/v1/search?q=%s&appToken=%s&authSign=%s&page=%s"
SEARCHURLGEO="http://11870.com/api/v1/search?q=%s&appToken=%s&authSign=%s&page=%s&lat=%s&lon=%s&radius=%s"
SEARCHCITY="http://11870.com/api/v1/search?sl=%s&appToken=%s&authSign=%s&page=%s"
SEARCHLOC="http://11870.com/api/v1/search?q=%s&l=%s&appToken=%s&authSign=%s&page=%s"
#URLs with authentication
SEARCHURLAUTH="http://11870.com/api/v1/search?q=%s&page=%s"
SEARCHURLGEOAUTH="http://11870.com/api/v1/search?q=%s&page=%s&lat=%s&lon=%s&radius=%s"
SEARCHCITYAUTH="http://11870.com/api/v1/search?sl=%s&page=%s"
SEARCHLOCAUTH="http://11870.com/api/v1/search?q=%s&l=%s&page=%s"

#User's sites URL
USERSITES="http://11870.com/api/v1/sites/%s"

class API:
    """Class to access to 11870.com functions.

    Public functions:
    search -- Return a dictionary with the results of the search on 11870.com.

    searchgeo -- Return a dictionary with the results of the search on 11870.com.
    The obtained results are near to a GPS position.

    get_url_data -- Return the output of a web page.

    get_user_sites -- Return the user's saved services in 11870.com

    get_temptoken -- Return the temporal token from 11870.com.

    get_authtoken_on_web -- Return the authentication token from 11870.com.

    validate_on_web -- Launch the web browser to validate the application on 11870.com.

    has_authtoken -- Return True if exists an authentication token.

    get_authtoken -- Return the authentication token from the configuration file.

    get_authtoken_filename -- Return the configuration filename.

    get_authtoken_path -- Return the path for the configuration file.

    save_authtoken -- Save the authentication token to the configuration file.

    """
    def __init__(self, username, authtoken):
        """Define some variables to use in some functions.
        Check if exists authentication token.

        """
        self.authsign = md5(APPKEY + SECRET).hexdigest()

        #if True, the functions on 11870.com will be used with the username
        if authtoken:
            self.auth = True
            self.authtoken = authtoken
            self.username = username
        else:
            self.auth = False


    def search(self, text, page):
        """Return a dictionary with the results of the search on 11870.com.

        @type   text: string
        @param  text: The text to be searched on 11870.com.
        @type   page: string
        @param  page: The number of the page of results.

        @rtype: dictionary
        @returns: A dictionary with the results of the search on 11870.com

        """
        h = httplib2.Http()

        #It's possible search w/o authentication to 11870.com
        #But it needs a different URL
        if self.auth:
            print self.authtoken, self.username
            h.add_credentials(self.username, self.authtoken)
            request, content = h.request(SEARCHURLAUTH % (text, page))
        else:
            request, content = h.request(SEARCHURL % (text, APPKEY, self.authsign, page))

        print content, '\n\n'
        feed = feedparser.parse(content)
        #print feed


        return feed

    def searchgeo(self, text, page, lat, lon, radius):
        """Return a dictionary with the results of the search on 11870.com.
        The obtained results are near to a GPS position.

        @type   text: string
        @param  text: The text to be searched on 11870.com.
        @type   page: string
        @param  page: The number of the page of results.
        @type   lat: string
        @param  lat: Geographic latitude.
        @type   lon: string
        @param  lon: Geographic longitude.
        @type   radius: string
        @param  radius: Radius where the services should be localized (in Km).

        @rtype: dictionary
        @returns: A dictionary with the results of the search on 11870.com.
        The obtained results are near to a GPS position.

        """
        h = httplib2.Http()

        #It's possible search w/o authentication to 11870.com
        #But it needs a different URL
        if self.auth:
            h.add_credentials(self.username, self.authtoken)
            request, content = h.request(SEARCHURLGEOAUTH % (text, page, lat, lon, radius))
        else:
            request, content = h.request(SEARCHURLGEO % (text, APPKEY, self.authsign, page, lat, lon, radius))

        feed = feedparser.parse(content)

        return feed

    def searchlocated(self, text, location, page):
        """Return a dictionary with the results of the search on 11870.com.

        @type   text: string
        @param  text: The text to be searched on 11870.com.
        @type   location: string
        @param  location: Location of the service (ex. Madrid)
        @type   page: string
        @param  page: The number of the page of results.

        @rtype: dictionary
        @returns: A dictionary with the results of the search on 11870.com.

        """
        h = httplib2.Http()

        #It's possible search w/o authentication to 11870.com
        #But it needs a different URL
        if self.auth:
            h.add_credentials(self.username, self.authtoken)
            request, content = h.request(SEARCHLOCAUTH % (text, location, page))
        else:
            request, content = h.request(SEARCHLOC % (text, location, APPKEY, self.authsign, page))

        feed = feedparser.parse(content)

        return feed

    def searchcity(self, city, page):
        """Return a dictionary with the results of the search on 11870.com.

        @type   city: string
        @param  city: City to be searched in 11870.com
        @type   page: string
        @param  page: The number of the page of results.

        @rtype: dictionary
        @returns: A dictionary with the results of the search on 11870.com.

        """
        h = httplib2.Http()

        #It's possible search w/o authentication to 11870.com
        #But it needs a different URL
        if self.auth:
            h.add_credentials(self.username, self.authtoken)
            request, content = h.request(SEARCHCITYAUTH % (city, page))
        else:
            request, content = h.request(SEARCHCITY % (text, APPKEY, self.authsign, page))

        feed = feedparser.parse(content)

        return feed

    def get_user_sites(self, userslug):
        """Return a dictionary with the user's saved services in 11870.com

        @type   userslug: string
        @param  userslug: Profile name in 11870.com for the user

        @rtype: dictionary
        @returns: A dictionary with the user's saved services in 11870.com

        """
        h = httplib2.Http()

        if self.auth:
            h.add_credentials(self.username, self.authtoken)
            request, content = h.request(USERSITES % (userslug))
        else:
            return None

        feed = feedparser.parse(content)

        return feed


    def create_new_service(self):
        import urllib
        h = httplib2.Http()

        #New service -> Error 400
#        data = """<?xml version="1.0" encoding="utf-8"?>
#        <entry xmlns:app='http://www.w3.org/2007/app' xmlns:georss='http://www.georss.org/georss/10' xmlns="http://www.w3.org/2005/Atom" xmlns:oos="http://11870.com/api/oos" xml:lang="es">
#        <id>http://11870.com/pro/test-new-service</id>
#        <title>Test new service</title>
#        <content>some commentary</content>
#        <oos:useraddress>Calle Falsa</oos:useraddress>
#        <oos:locality slug="/es/madrid">Madrid</oos:locality>
#        <oos:country slug="/es">España</oos:country>
#        <author><name>yerga</name></author>
#        <updated>2010-03-12T19:21:08.000Z</updated>
#        </entry>"""


        ##New comment in a saved service -> Error 401
        data = """<?xml version="1.0" encoding="utf-8"?>
        <entry xmlns:app='http://www.w3.org/2007/app' xmlns:georss='http://www.georss.org/georss/10' xmlns="http://www.w3.org/2005/Atom" xmlns:oos="http://11870.com/api/oos" xml:lang="es">
        <id>http://11870.com/pro/buenafuente</id>
        <oos:id>57244</oos:id>
        <summary type="text" xml:lang="es">Sitio agradable</summary>
        <content type="text" xml:lang="es">prueba</content>
        <author><name>yerga</name></author>
        <updated>2010-03-12T19:13:08.000Z</updated>
        <title type="text">Buenafuente</title>
         </entry>"""


        ##New comment in a not saved service -> Error 400
#        data = """<?xml version="1.0" encoding="utf-8"?>
#        <entry xmlns:app='http://www.w3.org/2007/app' xmlns:georss='http://www.georss.org/georss/10' xmlns="http://www.w3.org/2005/Atom" xmlns:oos="http://11870.com/api/oos" xml:lang="es">
#        <id>http://11870.com/pro/sidreria-yumay</id>
#        <content type="text" xml:lang="es">Buen restaurante para comidas, pescados y mariscos.</content>
#        <summary type="text" xml:lang="es">Sitio agradable</summary>
#        <updated>2010-03-12T19:15:08.000Z</updated>
#        <author><name>yerga</name></author>
#        <oos:id>105215</oos:id>
#        <title type="text">sidreria yumay</title>
#        </entry>"""


        headers = {'Content-Type': 'application/atom+xml', 'Authorization': 'WSSE realm="11870.com" profile="UsernameToken"'}

        print data
        if self.auth:
            h.add_credentials(self.username, self.authtoken)
            resp, content = h.request("http://11870.com/api/v1/sites/yerga", "POST", body=data, headers=headers)

        print resp, '\n\n'
        print content

        #feed = feedparser.parse(content)
        #return feed


    def get_url_data(self, url):
        """Return the output of a web page.

        @type   url: string
        @param  url: Web address.

        @rtype: string
        @returns: String, output of a web page.

        """
        import urllib
        result = urllib.urlopen(url)
        result = result.read()

        #print '\nresult: %s\n' % result

        return result

    def get_temptoken(self):
        """Return the temporal token from 11870.com.

        @rtype: string
        @returns: String, temporal token from 11870.com.

        """
        tempurl = TEMPTOKENURL % (APPKEY, self.authsign)
        #print 'tempurl: %s\n' % tempurl
        temptoken = self.get_url_data(tempurl)

        #the temptoken obtained has this format: <tempToken>xxxx</tempToken>
        #we need the string, so remove the tags
        temptoken = temptoken.replace('<tempToken>', '').replace('</tempToken>', '')
        return temptoken

    def get_authtoken_on_web(self, temptoken):
        """Return the authentication token from 11870.com.

        @type   temptoken: string
        @param  temptoken: Temporal token from 11870.com.

        @rtype: string
        @returns: String, the authentication token from 11870.com.

        """
        import urllib
        authtokenurl = AUTHTOKENURL % temptoken
        #print 'authtokenurl: %s\n' % authtokenurl
        authtoken = self.get_url_data(authtokenurl)

        #the authtoken obtained has this format <authToken>xxxx</authToken>
        #we need the string, so remove the tags
        authtoken = authtoken.replace('<authToken>', '').replace('</authToken>', '')
        return authtoken

    def validate_on_web(self, temptoken):
        """Launch the web browser to validate the application on 11870.com.

        @type   temptoken: string
        @param  temptoken: Temporal token from 11870.com.

        """
        import dbus
        authurl = AUTHURL % temptoken
        #print 'authurl: %s\n' % authurl

        #We use dbus to launch the N900 browser
        #There are different ways to launch the browser, but dbus is always cool!
        bus = dbus.SystemBus()
        proxy = bus.get_object("com.nokia.osso_browser", "/com/nokia/osso_browser/request")
        iface = dbus.Interface(proxy, 'com.nokia.osso_browser')
        iface.open_new_window(authurl)
