#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (C) 2009 by Daniel Martin Yerga
# <dyerga@gmail.com>
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# CasualServices: Search and share your favourite places
# Version 0.4
#

VERSION = "0.4"

import gtk
import hildon
import gobject
import logging
import os
import thread
import osmgpsmap
import pango
import gettext
import osso
import settings
import cgi

try:
    import location
except:
    print 'location is not available'

from services import once
from services import glocal

from portrait import FremantleRotation

ossoc = osso.Context("org.maemo.cservices", VERSION, False)

#Some useful long functions used in the code, we use smaller alias
fhsize = gtk.HILDON_SIZE_FINGER_HEIGHT
horbtn = hildon.BUTTON_ARRANGEMENT_HORIZONTAL
verbtn = hildon.BUTTON_ARRANGEMENT_VERTICAL
ui_normal = gtk.HILDON_UI_MODE_NORMAL
#Show a busy indication in the window menu
winprogind = hildon.hildon_gtk_window_set_progress_indicator

HOME = os.path.expanduser("~")
configdir = HOME + '/.cservices/'
logfile = configdir + 'log.txt'

if not os.path.exists(configdir):
    os.mkdir(configdir)

#detect if is running locally or not
#if it's running in /opt, it will be in the N900
import sys
runningpath = sys.path[0]

if '/opt' in runningpath:
    locally = False
else:
    locally = True

#Get the right paths of useful directories
if locally:
    imgdir = 'pixmaps/'
else:
    appdir = '/opt/cservices/'
    imgdir = appdir + 'pixmaps/'

#Define the logger used to manager the error/debug log
logger = logging.getLogger('cservices')
logging.basicConfig(filename=logfile,level=logging.ERROR, filemode='w')

DEBUG = True

if DEBUG:
    #set the main logger to DEBUG
    logger.setLevel(logging.DEBUG)

    #Create a handler for console debug
    console = logging.StreamHandler()
    console.setLevel(logging.DEBUG)
    # set a format which is simpler for console use
    formatter = logging.Formatter('%(name)-12s: %(levelname)-8s %(message)s')
    # tell the handler to use this format
    console.setFormatter(formatter)
    logging.getLogger('').addHandler(console)
    logger.debug("Starting log")

#Define some global list for services details
titles = []
phones = []
urls = []
localities = []
gmlpos = []
summaries = []
addresses = []
countries = []
areas = []
medialinks = []
listimg = []

latitudes = []
longitudes = []

#Some of this map sources doesn't work
#mapsources = {osmgpsmap.SOURCE_OPENSTREETMAP: 'Openstreemap',
#osmgpsmap.SOURCE_GOOGLE_STREET: 'Google Street',
#osmgpsmap.SOURCE_GOOGLE_HYBRID: 'Google Hybrid',
#osmgpsmap.SOURCE_GOOGLE_SATELLITE: 'Google Satellite',
#osmgpsmap.SOURCE_MAPS_FOR_FREE: 'Maps for free',
#osmgpsmap.SOURCE_OPENAERIALMAP: 'Open Aerial map',
#osmgpsmap.SOURCE_VIRTUAL_EARTH_HYBRID: 'Virtual Earth Hybrid',
#osmgpsmap.SOURCE_VIRTUAL_EARTH_SATELLITE: 'Virtual Earth Satellite',
#osmgpsmap.SOURCE_VIRTUAL_EARTH_STREET: 'Virtual Earth Street',
#osmgpsmap.SOURCE_YAHOO_HYBRID: 'Yahoo Hybrid',
#osmgpsmap.SOURCE_YAHOO_SATELLITE: 'Yahoo Satellite',
#osmgpsmap.SOURCE_YAHOO_STREET: 'Yahoo Street'}

mapsources = {osmgpsmap.SOURCE_OPENSTREETMAP: 'Openstreemap',
osmgpsmap.SOURCE_GOOGLE_STREET: 'Google Street',
osmgpsmap.SOURCE_GOOGLE_SATELLITE: 'Google Satellite'}

defaultimg = imgdir + 'defaultimg.png'
defaultpbuf = gtk.gdk.pixbuf_new_from_file(defaultimg)

def file_browser(window, action, title, folder, filename, file_ext):
    """Return a filename chosen from a Hildon.FileChooserDialog

    @type   window: gtk.Widget
    @param  window: Widget to be parent of the dialog
    @type   action: gtk.FILE_CHOOSER_ACTION
    @param  action: The open or save mode for the dialog
    @type   title: string
    @param  title: Title of the dialog
    @type   folder: string
    @param  folder: Current folder for the FileChooserDialog
    @type   filename: string
    @param  filename: Filename of the file to be saved
    @type   file_ext: string
    @param  file_ext: Filename extension


    @rtype: string
    @returns: String, a complete path+filename chosen for the user

    """

    #This is a generic FileChooserDialog
    #Can be used for Open or Save actions

    m = hildon.FileSystemModel()
    file_dialog = hildon.FileChooserDialog(window, action, m)
    file_dialog.set_title(title)

    file_dialog.set_default_response(gtk.RESPONSE_CANCEL)
    file_dialog.set_current_folder(folder)
    if (action==gtk.FILE_CHOOSER_ACTION_SAVE):
        file_dialog.set_current_name(filename)

    file_dialog.show_all()
    result = file_dialog.run()

    if result == gtk.RESPONSE_OK:
        namefile = file_dialog.get_filename()

        if (action==gtk.FILE_CHOOSER_ACTION_SAVE):
            namefile, extension = os.path.splitext(namefile)
            namefile = namefile + "." + file_ext
    else:
        namefile = False
    file_dialog.destroy()

    return namefile


gtk.gdk.threads_init()

class MainWindow:
    """Class to create and use the UI of the application

    """
    def __init__(self):
        """Start the application UI.

        """
        #Define where gettext should search for translation files
        gettext.install('cservices','/opt/cservices/share/locale')
        self.program = hildon.Program()
        self.program.__init__()
        gtk.set_application_name("CasualServices")

        self.window = hildon.StackableWindow()
        self.window.set_default_size(800, 480)
        self.window.connect("destroy", gtk.main_quit)
        self.program.add_window(self.window)

        self.rotation = FremantleRotation('cservices', None, VERSION, 0)
        #Variable for pagination in searches, the first page is 1 obviously
        self.page = '1'
        self.radius = '10'
        self.profile = ''
        self.location = ''
        self.settings = settings.Settings()
        self.mapsource, self.username, self.authtoken, self.profile = self.settings.load_config()

        #Create the widgets of the main window
        vbox = self.initial_screen()
        self.window.add(vbox)

        #Create the menu, and set it to the main window
        menu = self.make_menu()
        self.window.set_app_menu(menu)

        self.window.show_all()

        #Hide the buttons to change the page in the searches
        self.backbtn.hide()
        self.forwbtn.hide()
        self.allmapbtn.hide()

    def initial_screen(self):
        """Return a gtk.VBox with the UI widgets for the initial screen.

        @rtype: gtk.VBox
        @returns: gtk.VBox with the UI widgets for the initial screen.

        """

        vbox = gtk.VBox()

        parea = hildon.PannableArea()
        #This signal is for the gestures to change page in searches
        #It is not really gestures but horizontal movement
        parea.connect("horizontal-movement", self.horizontal_mov)

        self.searchtv = hildon.GtkTreeView(ui_normal)

        #Create an empty search model, to start w/o results
        self.search_model = self.create_search_model('')
        self.searchtv.set_model(self.search_model)

        self.searchtv.connect("row-activated", self.selected_service)

        self.add_columns_to_searchlist(self.searchtv)

        parea.add(self.searchtv)

        #Toolbar of the main window
        toolbar = self.maintoolbar()

        vbox.pack_start(parea, True, True, 0)
        vbox.pack_start(toolbar, False, False, 0)

        return vbox

    def horizontal_mov(self, parea, direction, initial_x, initial_y):
        """Search in next or previous page with gestures

        @type   parea: hildon.PannableArea
        @param  parea: hildon.PannableArea to receive the signal
        @type   direction: number
        @param  direction: direction of the movement
        @type   initial_x: number
        @param  initial_x: the x coordinate of the point where the user clicked
        @type   initial_y: number
        @param  initial_y: the y coordinate of the point where the user clicked

        """
        #Direction 2 -> left movement in the pannable area
        #Direction 3 -> right movement in the pannable area
        if direction == 2:
            self.search_prev_page(parea)
        elif direction == 3:
            self.search_next_page(parea)

    def maintoolbar(self):
        """Return a gtk.HBox with the UI widgets for the main window toolbar.

        @rtype: gtk.HBox
        @returns: gtk.HBox with the UI widgets for the main window toolbar.

        """

        hbox = gtk.HBox()

        #button = hildon.CheckButton(fhsize)
        #img = gtk.Image()
        #img.set_from_file('/usr/share/icons/hicolor/48x48/hildon/camera_gps_location_pressed.png')
        #button.set_image(img)
        #button.connect("clicked", self.remove_recipes)
        #hbox.pack_start(button, False, False, 5)

        #Text entry for searches
        self.searchentry = hildon.Entry(fhsize)
        self.searchentry.connect("activate", self.search_services)
        self.searchentry.set_placeholder("Search...")
        hbox.pack_start(self.searchentry, True, True, 0)

        #Search button
        button = hildon.Button(fhsize, horbtn)
        img = gtk.image_new_from_icon_name("general_search",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        button.connect("clicked", self.search_services)
        hbox.pack_start(button, False, False, 5)

        #Previous page button (for searches)
        self.backbtn = hildon.Button(fhsize, horbtn)
        img = gtk.image_new_from_icon_name("general_back",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        self.backbtn.set_image(img)
        self.backbtn.connect("clicked", self.search_prev_page)
        hbox.pack_start(self.backbtn, False, False, 5)

        #Next page button (for searches)
        self.forwbtn = hildon.Button(fhsize, horbtn)
        img = gtk.image_new_from_icon_name("general_forward",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        self.forwbtn.set_image(img)
        self.forwbtn.connect("clicked", self.search_next_page)
        hbox.pack_start(self.forwbtn, False, False, 5)

        self.allmapbtn = hildon.Button(fhsize, horbtn)
        img = gtk.image_new_from_icon_name("general_map",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        self.allmapbtn.set_image(img)
        self.allmapbtn.connect("clicked", self.show_all_in_map)
        hbox.pack_start(self.allmapbtn, False, False, 5)

        return hbox

    def make_menu(self):
        """Return a hildon.AppMenu with the menu items for the main window.

        @rtype: hildon.AppMenu
        @returns: hildon.AppMenu with the menu items for the main window.

        """
        menu = hildon.AppMenu()

        #button = hildon.CheckButton(fhsize)
        #button.set_label(_("Activate GPS"))
        #button.set_active(False)
        #menu.append(button)
        #button.connect("toggled", self.activate_gps)

        button = gtk.Button(_("Location to search"))
        button.connect("clicked", self.show_location_dlg)
        menu.append(button)

        self.nearbtn = hildon.CheckButton(fhsize)
        self.nearbtn.connect("toggled", self.radius_menu_toggled)
        self.nearbtn.set_label(_("Search near you"))
        self.nearbtn.set_active(False)
        menu.append(self.nearbtn)

        button = gtk.Button(_("Your services at 11870.com"))
        button.connect("clicked", self.get_user_sites)
        menu.append(button)

        button = gtk.Button(_("Settings"))
        button.connect("clicked", self.show_settings_dlg)
        menu.append(button)

        button = gtk.Button(_("About"))
        button.connect("clicked", About)
        menu.append(button)


        #button = gtk.Button(_("Create new service"))
        #button.connect("clicked", self.create_service)
        #menu.append(button)

        menu.show_all()
        return menu

    def radius_menu_toggled(self, widget):
        if widget.get_active():
            self.show_radius_dlg(widget)

    def show_radius_dlg(self, widget):
        """Show the dialog to define a radius where search

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        dlg = gtk.Dialog(title=_('Radius where search'), parent=None, flags=0)
        dlg.set_has_separator(False)

        dlg.add_button(_("Save"), gtk.RESPONSE_OK)

        hbox = gtk.HBox()

        label = gtk.Label(_("Radius (km)"))
        entry = hildon.Entry(fhsize)
        entry.set_text(self.radius)
        hbox.pack_start(label, False, False, 0)
        hbox.pack_start(entry, True, True, 0)

        dlg.vbox.pack_start(hbox, True, True, 0)

        dlg.show_all()
        result = dlg.run()
        if result == gtk.RESPONSE_OK:
            self.radius = entry.get_text()

        dlg.destroy()

    def show_location_dlg(self, widget):
        """Show the dialog to define a location where search

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        dlg = gtk.Dialog(title=_('Location to search'), parent=None, flags=0)
        dlg.set_has_separator(False)

        dlg.add_button(_("Save"), gtk.RESPONSE_OK)

        hbox = gtk.HBox()

        label = gtk.Label(_("Location"))
        entry = hildon.Entry(fhsize)
        entry.set_text(self.location)
        hbox.pack_start(label, False, False, 0)
        hbox.pack_start(entry, True, True, 0)

        dlg.vbox.pack_start(hbox, True, True, 0)

        dlg.show_all()
        result = dlg.run()
        if result == gtk.RESPONSE_OK:
            self.location = entry.get_text()

        dlg.destroy()

    def validate_once(self, widget, dlg):
        """Check if exists authentication token in the configuration file.
        If it does not exist, try get one from 11870.com

        @type   widget: gtk.Widget
        @param  widget: gtk widget that emitted the signal

        """

        if self.authtoken:
            #print 'has authtoken: ', self.authtoken
            #Show banner to the user to note that does not need a authtoken
            self.show_info_banner(widget,
                            _("Authorization to 11870.com already completed"))

        else:
            dlg.destroy()
            onceAPI = once.API(self.username, False)
            #Authentication token is not available, then try to get one
            temptoken = onceAPI.get_temptoken()

            #Launch the web browser in a different thread
            thread.start_new_thread(onceAPI.validate_on_web, (temptoken,))

            #Show the dialog to validate the authentication
            dialog = self.validate_dialog()
            if dialog:
                self.authtoken = onceAPI.get_authtoken_on_web(temptoken)
                self.settings.save_config(str(self.mapsource), self.username,
                            self.authtoken, self.profile)
                self.show_info_banner(widget,
                            _("Authorization to 11870.com successful"))
            else:
                #If it's not possible get an authtoken or if the user cancel it
                self.authenticate = False
                msg = _(("Authorization to 11870.com failed\n"
                "You will not able to use some 11870.com features"))
                self.show_info_banner(widget, msg)


    def validate_dialog(self):
        """Launch a dialog to accept the validation from 11870.com webpage

        """
        dialog = gtk.Dialog(title=_('Authorize 11870.com'), parent=self.window,
                            buttons=("OK", gtk.RESPONSE_OK))
        dialog.set_transient_for(self.window)
        label1 = gtk.Label(_('Press OK after you have given authorization.'))
        dialog.vbox.pack_start(label1, True, True, 0)

        dialog.show_all()
        result = dialog.run()
        if result == gtk.RESPONSE_OK:
            dialog.destroy()
            return True
        #else:
        #    dialog.destroy()
        #    return False

        dialog.destroy()

    def search_prev_page(self, widget):
        """Search the previous page from a previous search on 11870.com

        """
        self.page = str(int(self.page)-1)
        #Do a new search with the new page
        self.search_services(widget)

    def search_next_page(self, widget):
        """Search the next page from a previous search on 11870.com

        """
        self.page = str(int(self.page)+1)
        #Do a new search with the new page
        self.search_services(widget)

    def create_search_model(self, searchlist):
        """Return a gtk.ListStore with the results of a search on 11870.com

        @type   searchlist: list
        @param  searchlist: list of results from a search on 11870.com

        @rtype: gtk.ListStore
        @returns: Formatted gtk.ListStore with the results of a search on 11870.com

        """
        import style
        lstore = gtk.ListStore(gtk.gdk.Pixbuf, gobject.TYPE_STRING)

        #This is to use small text in the search list
        #This text is the location of the service
        #So, it show in big text the name of the service,
        #and in small text its location
        head_font = style.get_font_desc('SystemFont')
        head_color = style.get_color('ButtonTextColor')
        head = (head_font.to_string(), head_color.to_string())
        head = '<span font_desc="%s" foreground="%s">%%s</span>' % head
        sub_font = style.get_font_desc('SmallSystemFont')
        sub_color = style.get_color('SecondaryTextColor')
        sub = (sub_font.to_string(), sub_color.to_string())
        sub = '<span font_desc="%s" foreground="%s">%%s - %%s - %%s</span>' % sub
        lstore._markup_template = '\n'.join((head, sub))

        #searchlist = [titles, localities, areas, countries, listimg]

        if searchlist:
            print searchlist[4]
            #If there are search results, add it to the gtk.liststore
            for i in range(len(searchlist[0])):
                liter = lstore.append()
                txtformatted = lstore._markup_template % (cgi.escape(searchlist[0][i]),
                                cgi.escape(searchlist[1][i]),
                                cgi.escape(searchlist[2][i]),
                                cgi.escape(searchlist[3][i]))

                lstore.set(liter, 0, self.set_pix(searchlist[4][i], False), 1, txtformatted)

        return lstore

    def set_pix(self, urlimg, infoscreen):
        """Return a gtk.gdk.Pixbuf with the image from the URL

        @type   urlimg: string
        @param  urlimg: URL for the image to be downloaded

        @rtype: gtk.gdk.Pixbuf
        @returns: gtk.gdk.Pixbuf with the image from the URL

        """
        import urllib2
        try:
            myimg = urllib2.urlopen(urlimg)
            imgdata = myimg.read()

            pbl = gtk.gdk.PixbufLoader()
            pbl.write(imgdata)

            pbuf = pbl.get_pixbuf()
            pbl.close()

            return pbuf
        except:
            if infoscreen:
                return None
            else:
                return defaultpbuf

    def add_columns_to_searchlist(self, treeview):
        """Add gtk.TreeviewColumns to the gtk.TreeView for searchs

        @type   treeview: gtk.Treeview
        @param  treeview: gtk.Treeview where the columns will be added
        """
        model = treeview.get_model()

        renderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn('img', renderer, pixbuf=0)
        treeview.append_column(column)

        renderer = gtk.CellRendererText()
        #Wrap the name of the service if it's bigger than 790
        #N900 screen resolution is 800 of width
        renderer.set_property('wrap-mode', pango.WRAP_WORD)
        renderer.set_property('wrap-width', 790)
        column = gtk.TreeViewColumn('Title', renderer, markup=1)
        column.set_property("expand", True)
        treeview.append_column(column)

    def selected_service(self, widget, path, column):
        """Show a new window with the information from a selected service

        @type   widget: gtk.Widget
        @param  widget: gtk.Treeview from the search that emits the signal
        @type   path: number
        @param  path: path from the selected row of the treeview
        @type   column: gtk.TreeViewColumn
        @param  column: selected column from the treeview

        """
        self.serwin = hildon.StackableWindow()
        self.serwin.get_screen().connect("size-changed", self.orientation_changed)
        self.serwin.set_title(_("Information"))

        menu = hildon.AppMenu()
        button = gtk.Button(_("Add to Contacts"))
        button.connect("clicked", self.add_to_contacts, path)
        menu.append(button)
        menu.show_all()
        self.serwin.set_app_menu(menu)

        vbox = gtk.VBox()

        #TODO: show an image with the site of the result??
        #btn = gtk.Button()
        #img = gtk.image_new_from_file(imgdir + '11870.png')
        #btn.set_image(img)
        #btn.set_relief(gtk.RELIEF_NONE)
        #btn.set_alignment(0.00, 0.5)
        #vbox.pack_start(btn, True, True, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('<b>%s:</b> %s' % (_("Title"), cgi.escape(titles[path[0]])))
        label.set_line_wrap(gtk.WRAP_WORD)
        vbox.pack_start(label, True, True, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_line_wrap(gtk.WRAP_WORD)
        label.set_line_wrap(True)
        label.set_width_chars(38)

        if summaries[path[0]] != _('Not available') and summaries[path[0]]:
            if len(summaries[path[0]]) > 120:
                summary = summaries[path[0]][:120] + '...'
            else:
                summary = summaries[path[0]]

            label.set_markup("<b>%s:</b> %s" % (_("Summary"), cgi.escape(summary)))
            vbox.pack_start(label, True, True, 0)

        if phones[path[0]] != _('Not available'):
            button = hildon.Button(fhsize, verbtn)
            button.set_title(_('Phone'))
            button.set_value(phones[path[0]])
            button.connect("clicked", self.make_phonecall, phones[path[0]])
            vbox.pack_start(button, False, False, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('<b>%s:</b> %s' % (_("Address"), cgi.escape(addresses[path[0]])))
        label.set_line_wrap(gtk.WRAP_WORD)
        vbox.pack_start(label, True, True, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('<b>%s:</b> %s' % (_("Locality"), cgi.escape(localities[path[0]])))
        label.set_line_wrap(gtk.WRAP_WORD)
        vbox.pack_start(label, True, True, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('<b>%s:</b> %s' % (_("Area"), cgi.escape(areas[path[0]])))
        label.set_line_wrap(gtk.WRAP_WORD)
        vbox.pack_start(label, True, True, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('<b>%s:</b> %s' % (_("Country"), cgi.escape(countries[path[0]])))
        label.set_line_wrap(gtk.WRAP_WORD)
        vbox.pack_start(label, True, True, 0)

        if urls[path[0]] != _('Not available'):
            button = hildon.Button(fhsize, verbtn)
            button.set_title(_('Web page'))
            button.set_value(urls[path[0]])
            button.connect("clicked", self.show_webpage, urls[path[0]])
            vbox.pack_start(button, False, False, 0)

        if medialinks[path[0]]:
            self.mediabox = gtk.HBox()
            vbox.pack_start(self.mediabox, True, True, 0)

        self.serparea = hildon.PannableArea()
        self.serparea.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)
        self.serparea.add_with_viewport(vbox)

        self.mapbox = gtk.VBox()
        self.map = osmgpsmap.GpsMap(repo_uri = \
                    osmgpsmap.source_get_repo_uri(int(self.mapsource)), \
                    tile_cache = HOME + '/MyDocs/.maps')

        lat, lon = gmlpos[path[0]].split(' ')
        self.mapbox.pack_start(self.map, True, True, 0)

        hbox1 = gtk.HBox()
        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.zoomin_map, self.map)
        img = gtk.image_new_from_icon_name("pdf_zoomin",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox1.pack_start(button, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.center_map_on_service, lat, lon, self.map)
        img = gtk.image_new_from_icon_name("sketch_circle",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox1.pack_start(button, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.zoomout_map, self.map)
        img = gtk.image_new_from_icon_name("pdf_zoomout",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox1.pack_start(button, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.show_your_position_on_map, self.map)
        img = gtk.image_new_from_icon_name("gps_location",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox1.pack_start(button, True, True, 0)

        self.mapbox.pack_start(hbox1, False, False, 0)

        self.horbox = gtk.HBox()
        self.verbox = gtk.VBox()

        if self.is_portrait():
            self.serparea.set_size_request(-1, 480)
            self.verbox.pack_start(self.serparea, False, False, 0)
            self.verbox.pack_start(self.mapbox, True, True, 0)
            self.serwin.add(self.verbox)
        else:
            self.serparea.set_size_request(480, -1)
            self.horbox.pack_start(self.serparea, False, False, 0)
            self.horbox.pack_start(self.mapbox, True, True, 0)
            self.serwin.add(self.horbox)

        self.serwin.show_all()
        try:
            #Centering the map in the service position
            self.map.set_mapcenter (float(lat), float(lon), 13)
        except ValueError:
            pass


        self.add_pixmap_to_map(lat, lon)

        if medialinks[path[0]]:
            self.show_images(self.serwin, path)

    def show_images(self, win, path):
        """Start a thread to call the function to show images from a URL

        @type   win: gtk.Window
        @param  win: gtk.Window where show the animation icon
        @type   path: number
        @param  path: path from the selected row of the treeview

        """
        winprogind(win, 1)
        thread.start_new(self._do_show_images, (win, path))

    def _do_show_images(self, win, path):
        """Show images from URL in the information screen

        @type   win: gtk.Window
        @param  win: gtk.Window where show the animation icon
        @type   path: number
        @param  path: path from the selected row of the treeview

        """
        for i in range(len(medialinks[path[0]])):
            img = gtk.Image()
            pix = self.set_pix(medialinks[path[0]][i], True)
            img.set_from_pixbuf(pix)
            self.mediabox.pack_start(img, True, True, 0)

        self.mediabox.show_all()
        winprogind(win, 0)

    def add_to_contacts(self, widget, path):
        """Show images from URL in the information screen

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   path: number
        @param  path: path from the selected row of the treeview

        """
        import agenda
        ag = agenda.Agenda()
        #data = [name, address, phone, locality, region, country, url, img]
        #print titles[path[0]]       #sidreria yumay
        #print addresses[path[0]]    #Rafael Suárez, 7
        #print phones[path[0]]       #+34 985 570 826
        #print localities[path[0]]   #Avilés
        #print areas[path[0]]        #Asturias
        #print countries[path[0]]    #España
        #print urls[path[0]]         #http://www.sidreriayumay.com/
        #print medialinks[path[0]]   #[]

        data = [titles[path[0]], addresses[path[0]],
                phones[path[0]].replace(' ', ''), localities[path[0]],
                areas[path[0]], countries[path[0]],
                urls[path[0]], medialinks[path[0]]]
        result = ag.add_contact(data)

        if result:
            self.show_info_banner(widget, _("Service added to Contacts"))

    def is_portrait(self):
        """Check if the window is in portrait

        @rtype: boolean
        @returns: True if in portrait, False if in landscape.

        """
        width = gtk.gdk.screen_width()
        height = gtk.gdk.screen_height()
        if width > height:
            return False
        else:
            return True

    def orientation_changed(self, screen):
        """Reorganize widgets in info screen when changing from portrait
        to landscape or viceversa.

        @type   screen: gtk.gdk.Screen
        @param  screen: gtk.gdk.Screen that emits the signal

        """

        if not self.map.get_parent_window():
            print 'not in the right screen, doing nothing'
            return

        try:
            if self.is_portrait():
                self.serparea.set_size_request(-1, 480)

                self.horbox.remove(self.serparea)
                self.horbox.remove(self.mapbox)
                self.serwin.remove(self.horbox)

                self.verbox.pack_start(self.serparea, False, False, 0)
                self.verbox.pack_start(self.mapbox, True, True, 0)

                self.serwin.add(self.verbox)
                self.serwin.show_all()
            else:
                self.serparea.set_size_request(480, -1)

                self.verbox.remove(self.serparea)
                self.verbox.remove(self.mapbox)
                self.serwin.remove(self.verbox)

                self.horbox.pack_start(self.serparea, False, False, 0)
                self.horbox.pack_start(self.mapbox, True, True, 0)

                self.serwin.add(self.horbox)
                self.serwin.show_all()
        except:
            pass


    def add_pixmap_to_map(self, lat, lon):
        """Add pixbuf to map in a defined position

        @type   lat: text
        @param  lat: Geographic latitude.
        @type   lon: text
        @param  lon: Geographic longitude.

        """

        try:
            pixbuf = gtk.gdk.pixbuf_new_from_file(imgdir + 'place.png')
            self.map.add_image(float(lat), float(lon), pixbuf)
        except ValueError:
            self.show_info_banner(widget, _("Map location not available"))

    def show_webpage(self, widget, url):
        """Open the webbrowser and show the specified URL

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   url: text
        @param  url: URL to be shown

        """
        import dbus
        #We use dbus to launch the N900 browser
        #There are different ways to launch the browser, but dbus is always cool!
        bus = dbus.SystemBus()
        proxy = bus.get_object("com.nokia.osso_browser", "/com/nokia/osso_browser/request")
        iface = dbus.Interface(proxy, 'com.nokia.osso_browser')
        iface.open_new_window(authurl)

    def show_your_position_on_map(self, widget, themap):
        """Show the position of the GPS in the map

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        self.activate_gps(widget)

        #FIXME: wait for a real GPS fix to show the position in the map

        lat, lon = self.gps.get_data()
        print lat, lon

        #Draw our position in the map
        themap.draw_gps(float(lat), float(lon), 1.0)


    def zoomin_map(self, widget, themap):
        """Zoom in the map

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        themap.set_zoom(themap.props.zoom + 1)

    def zoomout_map(self, widget, themap):
        """Zoom out the map

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        themap.set_zoom(themap.props.zoom - 1)

    def center_map_on_service(self, widget, lat, lon, themap):
        """Center the map where the service is located

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   lat: number
        @param  lat: Geographic latitude of the service
        @type   lon: number
        @param  lon: Geographic longitude of the service

        """
        themap.set_mapcenter (float(lat), float(lon), 13)

    def search_services(self, widget):
        """Start a new thread to search for services

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        #If there are spaces, replace it for + to search in 11870.com
        text = self.searchentry.get_text().replace(' ', '+')
        winprogind(self.window, 1)
        #New thread to do the search, we don't wanna hang the UI
        thread.start_new(self._do_search_services, (widget, text))

    def _do_search_services(self, widget, text):
        """Search for services in 11870.com, and show the results in the UI

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   text: string
        @param  text: text to be searched

        """

        try:
            #Clearing the previous search
            self.search_model.clear()

            #Call the 11870.com API
            self.onceAPI = once.API(self.username, self.authtoken)
            self.GLocal = glocal.GLocal()

            #if option to search "near your position" is active
            if self.nearbtn.get_active():
                #glocalresults = {}
                #Activate the GPS to check your position
                self.activate_gps(widget)
                lat, lon = self.gps.get_data()
                #FIXME: GPS stuff
                #lat = '43.535000'
                #lon = '-5.942500'
                #feed = self.onceAPI.searchgeo(text, self.page, lat, lon, self.radius)
                #feed = {'entries': {}}
                glocalresults = self.GLocal.searchgeo(text, self.page, lat, lon, self.radius)
            #If search for location has a text defined
            elif self.location:
                #glocalresults = {}
                feed = self.onceAPI.searchlocated(text, self.location, self.page)
                glocalresults = self.GLocal.searchlocated(text, self.location, self.page)
            #Normal search
            else:
                feed = self.onceAPI.search(text, self.page)
                #feed = self.onceAPI.searchcity(text, self.page)
                glocalresults = self.GLocal.search(text, self.page)
                #glocalresults = {}

            #Parse the dictionary feed with the data from 11870
            searchlist = self.parse_results(feed, glocalresults)

            #Show the search results in the UI
            self.search_model = self.create_search_model(searchlist)
            self.searchtv.set_model(self.search_model)
            winprogind(self.window, 0)

            #If there are not any results, show a banner
            if searchlist == [[], [], [], []]:
                self.show_banner_thread(self.window,
                                        _('No results for this search'))
            else:
                #Show the buttons to change the page in the search
                self.backbtn.show()
                self.forwbtn.show()
                self.allmapbtn.show()
        except:
            logger.exception('Searching services')
            winprogind(self.window, 0)
            self.show_banner_thread(self.window, _('Error searching services'))

    def get_user_sites(self, widget):
        """Start a new thread to get the user's services

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        winprogind(self.window, 1)
        #New thread to get the users site, we don't wanna hang the UI
        thread.start_new(self._really_get_user_sites, ())

    def _really_get_user_sites(self):
        """Get user's saved services in 11870, and show the results in the UI

        """
        try:
            self.onceAPI = once.API(self.username, self.authtoken)
            feed = self.onceAPI.get_user_sites(self.profile)

            if not feed:
                winprogind(self.window, 0)
                self.show_banner_thread(self.window,
                                    _('Error. Check settings or connection.'))
                return

            #Clearing the previous search
            self.search_model.clear()

            #Parse the dictionary feed with the data from 11870
            #No Google Local results
            glocalresults = {}
            searchlist = self.parse_results(feed, glocalresults)

            #Show the results in the UI
            self.search_model = self.create_search_model(searchlist)
            self.searchtv.set_model(self.search_model)
            winprogind(self.window, 0)
        except:
            logger.exception('getting user services')
            winprogind(self.window, 0)
            self.show_banner_thread(self.window, _('Error getting services'))

    def create_service(self, widget):
        self.onceAPI = once.API(self.username, self.authtoken)
        self.onceAPI.create_new_service()

    def parse_results(self, feed, googlecontent):
        """Return a list of lists with data from the feed

        @type   feed: dictionary
        @param  feed: dictionary from a feedparser feed

        @rtype: list
        @returns: Return a list of lists with data from the feed

        """
        del titles[:]
        del phones[:]
        del urls[:]
        del localities[:]
        del gmlpos[:]
        del summaries[:]
        del addresses[:]
        del countries[:]
        del areas[:]
        del medialinks[:]
        del listimg[:]

        #if some data is not available, show this
        nodata = _('Not available')

        #Parse the dictionary and add the data to the different lists
        for i in feed['entries']:
            if i.has_key('title'):
                titles.append(i['title'])
            else:
                titles.append(nodata)

            if i.has_key('telephone'):
                phones.append(i['telephone'])
            else:
                phones.append(nodata)

            if i.has_key('url'):
                urls.append(i['url'])
            else:
                urls.append(nodata)

            if i.has_key('locality'):
                localities.append(i['locality'])
            else:
                localities.append(nodata)

            if i.has_key('pos'):
                gmlpos.append(i['pos'])
            else:
                gmlpos.append(nodata)

            if i.has_key('summary'):
                summaries.append(i['summary'].replace('&', '&amp;'))
            else:
                summaries.append(nodata)

            if i.has_key('useraddress'):
                addresses.append(i['useraddress'])
            else:
                addresses.append(nodata)

            if i.has_key('country'):
                countries.append(i['country'])
            else:
                countries.append(nodata)

            if i.has_key('subadministrativearea'):
                areas.append(i['subadministrativearea'])
            else:
                areas.append(nodata)

            mlinks = []
            if i.has_key('links'):
                for j in i['links']:
                    if j['rel'] == 'media':
                        if 'ps_' in j['href']:
                            if "http://11870.com" in j['href']:
                                mlinks.append(j['href'])
                            else:
                                mlinks.append('http://11870.com/%s' % j['href'])
                    #else:
                    #    print 'normal url'
                medialinks.append(mlinks)

            else:
                medialinks.append(nodata)

        for i in medialinks:
            try:
                listimg.append(i[0])
            except:
                listimg.append('')


        self.parse_google_search(googlecontent)

        searchlist = [titles, localities, areas, countries, listimg]

        return searchlist

    def parse_google_search(self, content):
        """Add Google Local results to the service details lists

        @type   content: dictionary
        @param  content: dictionary from a json object

        """

        nodata = _('Not available')

        for i in content:
            if i.has_key('titleNoFormatting'):
                titles.append(i['titleNoFormatting'])
            else:
                titles.append(nodata)

            if i.has_key('phoneNumbers'):
                print i['phoneNumbers'][0]['number']
                phones.append(i['phoneNumbers'][0]['number'])
            else:
                phones.append(nodata)

            #TODO: add some URL
            #if i.has_key(''):
            #    urls.append(i[''])
            #else:
            urls.append(nodata)

            summaries.append(nodata)

            if i.has_key('lat'):
                if i.has_key('lng'):
                    gmlpos.append("%s %s" % (i['lat'], i['lng']))
                else:
                    gmlpos.append(nodata)
            else:
                gmlpos.append(nodata)


            if i.has_key('city'):
                localities.append(i['city'])
            else:
                localities.append(nodata)

            if i.has_key('country'):
                countries.append(i['country'])
            else:
                countries.append(nodata)

            if i.has_key('streetAddress'):
                addresses.append(i['streetAddress'])
            else:
                addresses.append(nodata)

            if i.has_key('region'):
                areas.append(i['region'])
            else:
                areas.append(nodata)

            medialinks.append(nodata)

        for j in medialinks:
            listimg.append('')

    def show_all_in_map(self, widget):
        """Show a map with all services from a search

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        win = hildon.StackableWindow()
        win.set_title(_("All services in map"))

        vbox = gtk.VBox()
        themap = osmgpsmap.GpsMap(repo_uri = \
                osmgpsmap.source_get_repo_uri(int(self.mapsource)), \
                tile_cache = HOME + '/MyDocs/.maps')

        themap.add_layer(osmgpsmap.GpsMapOsd())
        themap.connect('button_press_event', self.map_clicked)

        vbox.pack_start(themap, True, True, 0)

        hbox = gtk.HBox()
        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.zoomin_map, themap)
        img = gtk.image_new_from_icon_name("pdf_zoomin",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox.pack_start(button, True, True, 0)

        #button = hildon.Button(fhsize, horbtn)
        #button.connect("clicked", self.center_map_on_service, lat, lon)
        #img = gtk.image_new_from_icon_name("sketch_circle",
        #                                    gtk.ICON_SIZE_LARGE_TOOLBAR)
        #button.set_image(img)
        #hbox.pack_start(button, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.zoomout_map, themap)
        img = gtk.image_new_from_icon_name("pdf_zoomout",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox.pack_start(button, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.connect("clicked", self.show_your_position_on_map, themap)
        img = gtk.image_new_from_icon_name("gps_location",
                                            gtk.ICON_SIZE_LARGE_TOOLBAR)
        button.set_image(img)
        hbox.pack_start(button, True, True, 0)

        vbox.pack_start(hbox, False, False, 0)

        win.add(vbox)
        win.show_all()

        pixbuf = gtk.gdk.pixbuf_new_from_file(imgdir + 'place.png')

        del latitudes[:]
        del longitudes[:]

        for coord in gmlpos:
            lat, lon = coord.split(' ')
            if coord != 'Not available':
                latitudes.append(lat)
                longitudes.append(lon)
            else:
                latitudes.append(False)
                longitudes.append(False)
            try:
                themap.add_image(float(lat), float(lon), pixbuf)
            except ValueError:
                pass

        #print latitudes, longitudes
        #Calculate average point to center the map
        #TODO: if services are very close, more zoom
        try:
            tlat = 0.0
            for nlat in latitudes:
                if nlat:
                    tlat = tlat + float(nlat)
            tlon = 0.0
            for nlon in longitudes:
                if tlon:
                    tlon = tlon + float(nlon)

            tfalses = latitudes.count(False)
            tnum = len(latitudes)-tfalses
            mlat = tlat/tnum
            mlon = tlon/tnum

            themap.set_mapcenter (float(mlat), float(mlon), 3)
        except Exception, e:
            print 'Failed to center map: ', e

    def map_clicked(self, themap, event):
        """Do actions when clicking in a map.
        Single click: check if the click is in a service
        Double click: zoom in the map

        @type   themap: osmgpsmap.GpsMap
        @param  themap: osmgpsmap.GpsMap that emits the signal
        @type   event: gtk.gdk.Event
        @param  event: gtk.gdk.Event that emits the signal

        """
        import math
        if event.button == 1:
            if event.type == gtk.gdk._2BUTTON_PRESS:
                themap.set_zoom(themap.props.zoom + 3)
            elif event.type == gtk.gdk.BUTTON_PRESS:
                rlat, rlon = themap.get_co_ordinates(event.x, event.y)
                #print math.degrees(rlat)
                #print math.degrees(rlon)
                self.check_if_tap_service(math.degrees(rlat), math.degrees(rlon), themap)

    def check_if_tap_service(self, taplat, taplon, themap):
        """Check if a click in the map is in a service

        @type   taplat: string
        @param  taplat: latitude in the map clicked
        @type   taplon: string
        @param  taplon: longitude in the map clicked
        @type   themap: osmgpsmap.GpsMap
        @param  themap: osmgpsmap.GpsMap that emits the signal

        """

        #print longitudes
        #print latitudes

        zoom = themap.props.zoom
        #print 'zoom; ', zoom

        #Different precision for different map zoom
        if zoom in range(1,6):
            decim = "%.f"
        elif zoom in range(6, 10):
            decim = "%.1f"
        elif zoom in range(10, 13):
            decim = "%.2f"
        elif zoom in range(13, 17):
            decim = "%.3f"
        elif zoom in range(17, 19):
            decim = "%.4f"

        taplat = decim % float(taplat)
        taplon = decim % float(taplon)

        #print taplat
        #print taplon

        for i in range(len(latitudes)):
            ni = decim % float(latitudes[i])
            #print 'ni: ', ni
            if ni == taplat:
                nj = decim % float(longitudes[i])
                #print 'nj: ', nj
                if nj == taplon:
                    #print 'a service selected'
                    self.show_service_info_dialog(i)
                    return
                break

    def show_service_info_dialog(self, index):
        """Show a dialog with information about a service

        @type   index: number
        @param  index: Index of the service in the results list

        """
        dlg = gtk.Dialog(title=_('Show service information'), parent=None, flags=0)
        dlg.set_has_separator(False)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('%s' % (cgi.escape(titles[index])))
        label.set_line_wrap(gtk.WRAP_WORD)
        dlg.vbox.pack_start(label, True, True, 0)


        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('%s' % (cgi.escape(addresses[index])))
        label.set_line_wrap(gtk.WRAP_WORD)
        dlg.vbox.pack_start(label, True, True, 0)

        label = gtk.Label()
        label.set_alignment(0.00, 0.5)
        label.set_markup('%s - %s - %s' % (cgi.escape(localities[index]),
                                cgi.escape(areas[index]),
                                cgi.escape(countries[index])))
        label.set_line_wrap(gtk.WRAP_WORD)
        dlg.vbox.pack_start(label, True, True, 0)


        if phones[index] != _('Not available'):
            button = hildon.Button(fhsize, verbtn)
            button.set_title(_('Phone'))
            button.set_value(phones[index])
            button.connect("clicked", self.make_phonecall, phones[index])
            dlg.vbox.pack_start(button, False, False, 0)

        dlg.show_all()
        result = dlg.run()
        dlg.destroy()

    def show_info_banner(self, widget, msg):
        """Show a information banner

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   msg: string
        @param  msg: text to be shown in the banner

        """
        hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint', msg)

    def show_banner_thread(self, widget, msg):
        """Show a information banner inside a thread

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   msg: string
        @param  msg: text to be shown in the banner

        """
        gtk.gdk.threads_enter()
        self.show_info_banner(widget, msg)
        gtk.gdk.threads_leave()

    def activate_gps(self, widget):
        """Activate N900 GPS

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        self.gps = GPS()

    def show_settings_dlg(self, widget):
        """Show the settings dialog

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal

        """
        dlg = gtk.Dialog(title=_('Settings'), parent=None, flags=0)
        dlg.set_has_separator(False)

        dlg.add_button(_("Save"), gtk.RESPONSE_OK)

        hbox = gtk.HBox()

        label = gtk.Label(_("Username"))
        entry = hildon.Entry(fhsize)
        entry.set_text(self.username)
        hbox.pack_start(label, False, False, 0)
        hbox.pack_start(entry, True, True, 0)
        dlg.vbox.pack_start(hbox, True, True, 0)

        hbox = gtk.HBox()
        label = gtk.Label(_("Profile name"))
        entry2 = hildon.Entry(fhsize)
        entry2.set_text(self.profile)
        hbox.pack_start(label, False, False, 0)
        hbox.pack_start(entry2, True, True, 0)
        dlg.vbox.pack_start(hbox, True, True, 0)

        mapbtn = hildon.PickerButton(fhsize, horbtn)
        data = mapsources.values()

        selector = self.create_selector(data)
        mapbtn.set_selector(selector)
        mapbtn.set_title(_("Map source"))
        mapbtn.set_value(mapsources[int(self.mapsource)])

        dlg.vbox.pack_start(mapbtn, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.set_label(_("Validate 11870.com"))
        button.connect("clicked", self.validate_once, dlg)
        dlg.vbox.pack_start(button, True, True, 0)

        dlg.show_all()
        result = dlg.run()
        if result == gtk.RESPONSE_OK:
            self.username = entry.get_text()
            self.profile = entry2.get_text()
            mapname = mapbtn.get_value()
            self.mapsource = mapsources.keys()[mapsources.values().index(mapname)]
            self.settings.save_config(str(self.mapsource), self.username, self.authtoken, self.profile)

        dlg.destroy()

    def create_selector(self, data):
        """Create a hildon.TouchSelector with some data in.

        @type   data: list
        @param  data: data to be added to the touchselector

        @rtype: hildon.TouchSelector
        @returns: a hildon.TouchSelector to be added to a parent widget

        """

        selector = hildon.TouchSelector(text=True)

        for i in range(len(data)):
            selector.append_text(data[i])

        return selector

    def make_phonecall(self, widget, phonenumber):
        """Make a phone call with the OS phone application

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   phonenumber: string
        @param  phonenumber: Phone number to be called

        """

        import subprocess, shlex
        phonenumber = phonenumber.replace(' ', '')
        call = 'dbus-send --system --dest=com.nokia.csd.Call --type=method_call --print-reply /com/nokia/csd/call com.nokia.csd.Call.CreateWith string:%s uint32:0' % phonenumber
        #print call

        args = call.split(' ')

        #print args
        p = subprocess.Popen(args)

#TODO: make this in the same class, easier to get data??

class GPS:
    """Class to manage the functions of the N900 GPS

    """
    def __init__(self):
        """Start the N900 GPS

        """
        self.control = location.GPSDControl.get_default()
        self.device = location.GPSDevice()
        self.control.set_properties(preferred_method=location.METHOD_USER_SELECTED,
                               preferred_interval=location.INTERVAL_DEFAULT)

        self.control.connect("error-verbose", self.on_error)
        self.device.connect("changed", self.on_changed, self.control)

        #Start the GPS system
        gobject.idle_add(self.start_location, self.control)

    def on_error(self, control, error):
        """Show GPS error if something fails

        @type   control: location.GPSDControl
        @param  control: GPSDControl object that emits the signal
        @type   error: number
        @param  error: error number of the GPS system

        """
        print "location error: %d... quitting" % error

    def on_changed(self, device, control):
        """Print latitude and longitude when the GPS has a fix

        @type   device: location.GPSDevice()
        @param  device: GPSDevice object that emits the signal
        @type   control: location.GPSDControl
        @param  control: GPSDControl object that emits the signal

        """
        if not device:
            return
        if device.fix:
            if device.fix[1] & location.GPS_DEVICE_LATLONG_SET:
                print "lat = %f, long = %f" % device.fix[4:6]
                control.stop()

    def start_location(self, control):
        """Start the GPSDControl object

        @type   control: location.GPSDControl
        @param  control: GPSD

        """
        control.start()
        return False

    def get_data(self):
        """Return the current latitude and longitude of the GPS

        @rtype: numbers
        @returns: latitude, longitude

        """
        #print "online: ", self.device.online
        #print "status: ", self.device.status
        #print "fix: ", self.device.fix
        #print "sats in view: ", self.device.satellites_in_view
        #print "sats in use: ", self.device.satellites_in_use
        #print "sats: ", self.device.satellites
        #print "cellinfo: ", self.device.cell_info

        #FIXME: solo cuando hay un fix

        lat, lon = self.device.fix[4:6]
        return lat, lon


class About:
    """Class for the About dialog

    """
    def __init__(self, widget):
        """Show the About dialog UI

        """
        self.abdialog = gtk.Dialog()
        self.abdialog.set_title(_("About CasualServices"))

        notebook = gtk.Notebook()
        notebook.set_show_tabs(False)
        notebook.set_scrollable(False)
        notebook.set_show_border(False)

        # Description page #
        vbox = gtk.VBox()

        label = gtk.Label()
        label.set_markup("<b><big>CasualServices %s</big></b>" % VERSION)
        vbox.pack_start(label, True, True, 0)

        label = gtk.Label("Search and share your favourite places")
        vbox.pack_start(label, True, True, 0)

        label = gtk.Label("GNU General Public License")
        vbox.pack_start(label, True, True, 0)

        url = "http://cusl4-cservices.forja.rediris.es/"
        webbtn = gtk.LinkButton(url, "Web")
        vbox.pack_start(webbtn, True, True, 0)
        gtk.link_button_set_uri_hook(self.launch_browser)

        label = gtk.Label(_("CasualServices uses the 11870.com API"))
        vbox.pack_start(label, True, True, 0)

        url = "http://11870.com"
        webbtn = hildon.Button(fhsize, horbtn)
        webbtn.set_label("Visit")
        webbtn.connect("clicked", self.launch_browser, url)
        image = gtk.Image()
        image.set_from_file(imgdir + '11870.png')
        webbtn.set_image(image)

        hbox = gtk.HBox()
        hbox.set_homogeneous(True)
        hbox.pack_start(webbtn, False, False, 0)
        vbox.pack_start(hbox, False, False, 0)


        notebook.append_page(vbox, gtk.Label())

        # Credits page #
        vbox = gtk.VBox()
        textview = hildon.TextView()
        textview.set_cursor_visible(False)
        textview.set_wrap_mode(gtk.WRAP_WORD)
        text = ("%s Daniel Martin Yerga (dyerga@gmail.com)\n"
        "\n"
        "%s:") % (_("Written by"), _("Translations"))
        textview.get_buffer().set_text(text)

        parea = hildon.PannableArea()
        parea.add(textview)

        vbox.pack_start(parea, True, True, 0)
        notebook.append_page(vbox, gtk.Label())

        # Report page #
        vbox = gtk.VBox()

        textview = hildon.TextView()
        textview.set_cursor_visible(False)
        textview.set_wrap_mode(gtk.WRAP_WORD)
        text = _("""Please, if you find some error in the application, report it:
1) Send the log from the button above (if there's an error in the log).
2) Press the button and read how to report a bug.""")
        textview.get_buffer().set_text(text)

        parea = hildon.PannableArea()
        parea.add(textview)

        hbox = gtk.HBox()
        hbox.set_homogeneous(True)

        button = hildon.Button(fhsize, horbtn)
        button.set_title(_("Report error"))
        url = "http://cusl4-cservices.forja.rediris.es/reporting.html"
        button.connect("clicked", self.launch_browser, url)
        hbox.pack_start(button, True, True, 0)

        button = hildon.Button(fhsize, horbtn)
        button.set_title(_("Log"))
        button.connect("clicked", Log, logfile)
        hbox.pack_start(button, True, True, 0)

        vbox.pack_start(hbox, False, False, 0)
        vbox.pack_start(parea, True, True, 0)

        notebook.append_page(vbox, gtk.Label())

        # Buttons #
        self.abdialog.vbox.pack_start(notebook, True, True, 0)

        hbox = gtk.HBox()

        descbutton = hildon.GtkRadioButton(fhsize)
        descbutton.set_mode(False)
        descbutton.set_active(True)
        descbutton.set_label(_('Description'))
        descbutton.connect("toggled", self.change_tab, notebook, 0)
        hbox.pack_start(descbutton, True, True, 0)

        button = hildon.GtkRadioButton(fhsize)
        button.set_mode(False)
        button.set_active(True)
        button.set_label(_('Credits'))
        button.set_group(descbutton)
        button.connect("toggled", self.change_tab, notebook, 1)
        hbox.pack_start(button, True, True, 0)

        button = hildon.GtkRadioButton(fhsize)
        button.set_mode(False)
        button.set_label(_('Report'))
        button.set_group(descbutton)
        button.connect("clicked", self.change_tab, notebook, 2)
        hbox.pack_start(button, True, True, 0)

        self.abdialog.vbox.pack_start(hbox, False, False, 0)

        self.abdialog.show_all()
        self.abdialog.run()
        self.abdialog.destroy()

    def change_tab(self, widget, notebook, number):
        notebook.set_current_page(number)

    def launch_browser(self, widget, url):
        """Launch the webbrowser

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   url: string
        @param  url: URL address to be opened by the browser

        """
        import dbus
        bus = dbus.SystemBus()
        proxy = bus.get_object("com.nokia.osso_browser",
                                "/com/nokia/osso_browser/request")
        iface = dbus.Interface(proxy, 'com.nokia.osso_browser')

        self.abdialog.destroy()

        iface.open_new_window(url)

class Log:
    """Class for show and manage the error log system

    """
    def __init__(self, widget, logfile):
        """Show the main dialog to manage the Log

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   logfile: string
        @param  logfile: filename for the log file.

        """

        #Log dialog UI
        dialog = gtk.Dialog(title='Log', parent=None)

        dialog.set_size_request(600, 350)

        parea = hildon.PannableArea()
        parea.set_property("mov-mode", hildon.MOVEMENT_MODE_BOTH)

        textview = hildon.TextView()
        textview.set_property("editable", False)
        textview.set_property("wrap-mode", gtk.WRAP_WORD)

        log = open(logfile, 'r')
        logtext = log.read()
        log.close()

        textview.get_buffer().set_text(logtext)
        parea.add(textview)

        dialog.vbox.pack_start(parea, True, True, 0)

        hbox = gtk.HBox()

        save_btn = hildon.Button(fhsize, horbtn)
        save_btn.set_title("Save")
        save_btn.connect('clicked', self.save, logfile, dialog)

        clear_btn = hildon.Button(fhsize, horbtn)
        clear_btn.set_title("Clear")
        clear_btn.connect('clicked', self.clear, textview, logfile)

        send_btn = hildon.Button(fhsize, horbtn)
        send_btn.set_title('Send')
        send_btn.connect('clicked', self.send, dialog, logfile)

        hbox.pack_start(save_btn, True, True, 0)
        hbox.pack_start(clear_btn, True, True, 0)
        hbox.pack_start(send_btn, True, True, 0)

        dialog.vbox.pack_start(hbox, False, False, 0)

        dialog.show_all()
        dialog.run()
        dialog.destroy()

    def clear(self, widget, textview, logfile):
        """Clear the log

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   textview: gtk.TextView
        @param  textview: gtk.TextView that has the log written
        @type   logfile: string
        @param  logfile: filename for the log file.

        """
        textview.get_buffer().set_text('')
        f = open(logfile, 'w')
        f.close()

    def save(self, widget, logfile, dlg):
        """Save the log as a text file

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   logfile: string
        @param  logfile: filename for the log file.
        @type   dlg: gtk.Dialog
        @param  dlg: main dialog for the Log system

        """
        import shutil
        #The user choose the filename for the log file
        filename = file_browser(dlg, gtk.FILE_CHOOSER_ACTION_SAVE,
                                _("Save log file"), HOME + '/MyDocs/.documents',
                                "cservices-log", "txt")
        if not filename:
            return

        try:
            #Copy the current log file to the path chosen for the user
            shutil.copyfile(logfile, filename)
            MainWindow.show_info_banner(widget, _('Log file saved'))
        except:
            logger.exception("Saving log file")
            MainWindow.show_info_banner(widget, _('Error saving the log file'))

    def send(self, widget, dlg, logfile):
        """Show the confirmation dialog to send the log to the server

        @type   widget: gtk.Widget
        @param  widget: gtk.Widget that emits the signal
        @type   logfile: string
        @param  logfile: filename for the log file.
        @type   dlg: gtk.Dialog
        @param  dlg: main dialog for the Log system

        """
        sendtxt = _("""You are going to send the log to the developers.
This helps the developers to track problems with the application.
It doesn't send any personal information (like passwords or similar).""")

        dialog = hildon.hildon_note_new_confirmation(dlg, sendtxt)
        dialog.set_button_texts("Send", "Cancel")
        dialog.show_all()
        response = dialog.run()
        if response == gtk.RESPONSE_OK:
            self.do_pre_send(dlg, logfile)

        dialog.destroy()

    def do_pre_send(self, dlg, logfile):
        hildon.hildon_gtk_window_set_progress_indicator(dlg, 1)
        thread.start_new_thread(self._do_send, (dlg, logfile))

    def _do_send(self, dlg, logfile):
        """Send the log to the developer server

        @type   logfile: string
        @param  logfile: filename for the log file.
        @type   dlg: gtk.Dialog
        @param  dlg: main dialog for the Log system

        """
        import pycurl, shutil, random, commands
        try:
            #Create a random name for the file to be sent
            rname = ''
            for i in random.sample('abcdefghijkl123456789', 18):
                rname += i

            #Copy the logfile to the new temporal file
            rnamepath = HOME + "/.cservices/" + rname
            shutil.copyfile(logfile, rnamepath)

            #Get different debug info, python, gtk, maemo versions, etc.
            gtkversion = "%s.%s.%s" % gtk.ver
            if os.path.exists("/etc/maemo_version"):
                mfile = open("/etc/maemo_version", 'r')
                maemoversion = mfile.read()
                mfile.close()
            else:
                maemoversion = ''

            #Check dirty memory, useful for python apps
            opsystem = ' '.join(os.uname())
            pyversion = os.sys.version
            pid = os.getpid()
            comm = ("awk '/Private_Dirty/{sum+=$2}END{print sum \"kB\"}'"
            " /proc/%s/smaps") % pid
            status, dirtymem = commands.getstatusoutput(comm)

            lfile = open(rnamepath, 'r')
            log = lfile.read()
            lfile.close()

            _version = "CasualServices %s" % VERSION

            #Formatting the text to be sent
            log = ("%s\nPython version: %s\nGtk version: %s\n"
            "Maemo version: %sOperating system: %s\n"
            "Dirty Memory: %s\nLog:\n%s") % (_version, pyversion, gtkversion,
            maemoversion, opsystem, dirtymem, log)

            lfile = open(rnamepath, 'w')
            lfile.write(log)
            lfile.close()

            #Sent the log file to the server
            url = "http://yerga.net/logs/uploader.php"
            data = [('uploadedfile', (pycurl.FORM_FILE, rnamepath)),]
            mycurl = pycurl.Curl()
            mycurl.setopt(pycurl.URL, url)
            mycurl.setopt(pycurl.HTTPPOST, data)

            mycurl.perform()
            mycurl.close()
            os.remove(rnamepath)

            gtk.gdk.threads_enter()
            MainWindow.show_info_banner(dlg, 'Log sent')
            gtk.gdk.threads_leave()
            hildon.hildon_gtk_window_set_progress_indicator(dlg, 0)
        except:
            logger.exception("Sending log file")
            gtk.gdk.threads_enter()
            MainWindow.show_info_banner(dlg, _('Error sending the log file'))
            gtk.gdk.threads_leave()
            hildon.hildon_gtk_window_set_progress_indicator(dlg, 0)


if __name__ == "__main__":
    MainWindow = MainWindow()
    gtk.gdk.threads_enter()
    gtk.main()
    gtk.gdk.threads_leave()
