/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.WordsList 1.0
import crosswords.ConfigurationManager 1.0
import "GeneralFunctions.js" as GF
import crosswords.AudioPlayer 1.0
import "ComponentCreator.js" as CC
import crosswords.NetworkServer 1.0
import crosswords.NetworkClient 1.0

Rectangle {
    id: main
    width: 800
    height: 480
    color: "#000000"

	// C++ side INJECTED properties
	// networkServer
	// END OF C++ side INJECTED properties

	// C++ side initialized properties
	property bool quitButtonVisible: true
	// END OF C++ side initialized properties

	//WARNING: initially this is disabled due to high CPU usage when playing this sounds
	property bool soundsEnabled: false
	onSoundsEnabledChanged: if (soundsEnabled) createAudioPlayers(); else destroyAudioPlayers()

	property AudioPlayer audioPlayerPlayClick: null
	property AudioPlayer audioPlayerPlayBack: null
	property AudioPlayer audioPlayerPlayDrag: null
	property AudioPlayer audioPlayerPlayDrop: null
	property AudioPlayer audioPlayerPlayDropInvalid: null
	property AudioPlayer audioPlayerPlayYourTurn: null


	property variant openingStack: new Array()

	// GLOBAL DATA
	//every QML element should use these instances (singletons) for optimum memory usage

	WordsList
	{
		id: wordsListSingleton
	}
	// END OF GLOBAL DATA



    // GENERAL FUNCTIONS
	function quit()
	{
		if (GF.parseBool(configurationManager.getValue("ask quit")))
			setState(windowAskQuit.getStateName())
		else
			Qt.quit()
	}

    function setState(stateName)
    {
		var tmpStack = openingStack

		//if nothing set, then use default state and clear stack (put current/default state)
		if (typeof stateName == "undefined")
		{
			stateName = "Main"
			tmpStack = new Array()
			tmpStack.push(stateName)
			backButton = 0
		}
		else if (stateName === -1)
		{
			//we can go back only if we have more than 1 element (Main window + anything else)
			//otherwise back button is pushed too much or there is problem with code logic
			if (tmpStack.length > 1)
			{
				//on top of the stack is always current state,
				//so to go back we need to remove current and set one before current
				tmpStack.pop()
				stateName = tmpStack[tmpStack.length - 1]
				if (tmpStack.length <= 1)
					backButton.opacity = 0
			}
			else
			{
				//this can occur when back button is pressed (to go back to main window)
				//while opacity is changing (while hidding back button)
				console.debug("Don't push back button too much!")
				return false
			}
		}
		else
		{
			//in case that Window (or Window-extended) element is given
			if (typeof stateName == "object")
			{
				try
				{
					var tmpStateName = stateName.getStateName()
					if (!stateName.initialize())
					{
						console.debug("Unable to initialize: " + tmpStateName)
						return false
					}
					stateName = tmpStateName
				}
				catch (e)
				{
					console.debug("Element (or element that extends) Window must be given to setState(), but got: " + stateName)
					return false
				}
			}
			//else it is just name of state
			//WARNING: This (giving plain state names) should be avoided since there is no way to control initialization!
			tmpStack.push(stateName)
			backButton.opacity = 1
		}

		openingStack = tmpStack

        main.state = stateName;
		return true
    }

	function setProperties(doAnimations, smooth, sounds, maxHighscore)
	{
		var childs = windows.children
		for (var i=0; i<childs.length; i++)
		{
			var child = childs[i]
			child.doAnimations = doAnimations
			child.smooth = smooth
		}
		soundsEnabled = sounds
		quitButton.doAnimations = doAnimations
		quitButton.smooth = smooth
		backButton.doAnimations = doAnimations
		backButton.smooth = smooth
		windowHighscores.maxHighscores = maxHighscore
	}

	function refreshName()
	{
		var name = configurationManager.getValue("name")
		networkServer.setHostName(name)
		networkClient.setName(name)
	}

	function playBack()
	{
		try
		{
			audioPlayerPlayBack.play()
		}
		catch (e){}
	}

	function playClick()
	{
		try
		{
			audioPlayerPlayClick.play()
		}
		catch (e){}
	}

	function playDrag()
	{
		try
		{
			audioPlayerPlayDrag.play()
		}
		catch (e){}
	}

	function playDrop()
	{
		try
		{
			audioPlayerPlayDrop.play()
		}
		catch (e){}
	}

	function playDropInvalid()
	{
		try
		{
			audioPlayerPlayDropInvalid.play()
		}
		catch (e){}
	}

	function playYourTurn()
	{
		try
		{
			audioPlayerPlayYourTurn.play()
		}
		catch (e){}
	}
	// END OF GENERAL FUNCTIONS

	focus: true
    Keys.onPressed:
    {
		//here are events which will be processed no matter what (except if some window grabs focus)
        if (event.key === Qt.Key_Escape)
			if (openingStack.length <= 1)
				quit()
			else
				setState(-1)
		else	//process key event only to active window
			for (var i=0; i<windows.children.length; i++)
				if (windows.children[i].opacity > 0)
					windows.children[i].processKey(event)
    }

    //running initialization
    Component.onCompleted:
    {
        //this is required because by default state is null
        //so animation on first call wont happen
		setState()

		var doAnim = GF.parseBool(configurationManager.getValue("animate"))
		setProperties(doAnim, GF.parseBool(configurationManager.getValue("smooth")),
					  GF.parseBool(configurationManager.getValue("sounds")),
					  parseInt(configurationManager.getValue("crosswords max highscores")))
		//a little intro, showing up main window
		if (doAnim)
			initialMainWindowShowAnimation.start()

		refreshName()
	}

    NumberAnimation
    {
        id: initialMainWindowShowAnimation
        target: windowMain
        property: "opacity"
        from: 0
        to: 1
        duration: 500
    }

	// AUDIO MANAGEMENT
	function createAudioPlayers()
	{
		audioPlayerPlayClick = createAudioPlayerFromName("click")
		audioPlayerPlayBack = createAudioPlayerFromName("back")
		audioPlayerPlayDrag = createAudioPlayerFromName("drag")
		audioPlayerPlayDrop = createAudioPlayerFromName("drop")
		audioPlayerPlayDropInvalid = createAudioPlayerFromName("drop-invalid")
		audioPlayerPlayYourTurn = createAudioPlayerFromName("your-turn")
	}

	function createAudioPlayerFromName(name)
	{
		CC.initializeComponent("AudioPlayer")
		return Qt.createQmlObject("import QtQuick 1.1; import crosswords.AudioPlayer 1.0; AudioPlayer{" +
								  "Component.onCompleted: setAudioFile(\"" +
								  name + "\")}", main, "DynamicAudioPlayer")
	}

	function destroyAudioPlayers()
	{
		audioPlayerPlayClick = destroyAudioPlayerByReference(audioPlayerPlayClick)
		audioPlayerPlayBack = destroyAudioPlayerByReference(audioPlayerPlayBack)
		audioPlayerPlayDrag = destroyAudioPlayerByReference(audioPlayerPlayDrag)
		audioPlayerPlayDrop = destroyAudioPlayerByReference(audioPlayerPlayDrop)
		audioPlayerPlayDropInvalid = destroyAudioPlayerByReference(audioPlayerPlayDropInvalid)
	}

	function destroyAudioPlayerByReference(ref)
	{
		try
		{
			ref.destroy()
		}
		catch (e){}
		return null
	}
	// END OF AUDIO MANAGEMENT

	ConfigurationManager
	{
		id: configurationManager
	}

	NetworkServer
	{
		id: networkServer
	}

	NetworkClient
	{
		id: networkClient
	}

	Button
	{
		id: quitButton
		width: 70
		height: 70
		visible: main.quitButtonVisible
		anchors.right: parent.right
		anchors.rightMargin: -5
		anchors.top: parent.top
		anchors.topMargin: -5
		text: ""

		Image
		{
			source: "images/quit.png"
			anchors.fill: parent
			anchors.leftMargin: 15
			anchors.rightMargin: 15
			anchors.topMargin: 20
			anchors.bottomMargin: 20
		}

		fontSize: 36
		fontColor: "#ffffff"
		buttonColor: "#880000"
		mouseArea.onReleased: quit()
		z: 900
	}

	Button
	{
		id: backButton
		x: quitButton.x
		y: quitButton.y
		width: quitButton.width
		height: quitButton.height
		opacity: 0
		text: ""

		Image
		{
			source: "images/back.png"
			anchors.fill: parent
			anchors.leftMargin: 15
			anchors.rightMargin: 15
			anchors.topMargin: 20
			anchors.bottomMargin: 20
		}

		buttonColor: quitButton.buttonColor
		mouseArea.onReleased: setState(-1)
		customSound: audioPlayerPlayBack
		z: 901
	}

	Item
	{
		id: windows
		anchors.fill: parent

		WindowAskQuit
		{
			id: windowAskQuit
		}

		WindowMain
		{
			id: windowMain
		}

		WindowSingleplayerSingleword
		{
			id: windowSingleplayerSingleword
			wordsList: wordsListSingleton
		}

		WindowSingleplayerSinglewordSettings
		{
			id: windowSingleplayerSinglewordSettings
			wordsList: wordsListSingleton
		}

		WindowSingleplayerCrosswords
		{
			id: windowSingleplayerCrosswords
			wordsList: wordsListSingleton
		}

		WindowSingleplayerCrosswordsSettings
		{
			id: windowSingleplayerCrosswordsSettings
			wordsList: wordsListSingleton
		}

		WindowGameBrowser
		{
			id: windowGameBrowser
		}

		WindowJoinedGame
		{
			id: windowJoinedGame
		}

		WindowHostGame
		{
			id: windowHostGame
			wordsList: wordsListSingleton
		}

		WindowHostedGameDetails
		{
			id: windowHostedGameDetails
		}

		WindowMultiplayerCrosswordsClient
		{
			id: windowMultiplayerCrosswordsClient
		}

		WindowMultiplayerCrosswordsHost
		{
			id: windowMultiplayerCrosswordsHost
			wordsList: wordsListSingleton
		}

		WindowSettings
		{
			id: windowSettings
			onHidden:
			{
				setProperties(GF.parseBool(configurationManager.getValue("animate")),
							  GF.parseBool(configurationManager.getValue("smooth")),
							  GF.parseBool(configurationManager.getValue("sounds")),
							  parseInt(configurationManager.getValue("crosswords max highscores")))
				refreshName()
			}
		}

		WindowHighscores
		{
			id: windowHighscores
		}

	}


/* STATE */

    function hideAllWindows()
    {
        main.children.opacity = 0
    }

    onStateChanged: hideAllWindows() //this is first called, than actual states ged changed (invoked)

    states: [
		State {
			name: windowAskQuit.getStateName()

			PropertyChanges {
				target: windowAskQuit
				opacity: 1
			}
		},
		State {
			name: windowMain.getStateName()

			PropertyChanges {
				target: windowMain
				opacity: 1
			}
		},
		State {
			name: windowSingleplayerSingleword.getStateName()

			PropertyChanges {
				target: windowSingleplayerSingleword
				opacity: 1
			}
		},
		State {
			name: windowSingleplayerSinglewordSettings.getStateName()

			PropertyChanges {
				target: windowSingleplayerSinglewordSettings
				opacity: 1
			}
		},
		State {
			name: windowSingleplayerCrosswords.getStateName()

			PropertyChanges {
				target: windowSingleplayerCrosswords
				opacity: 1
			}
		},
		State {
			name: windowSingleplayerCrosswordsSettings.getStateName()

			PropertyChanges {
				target: windowSingleplayerCrosswordsSettings
				opacity: 1
			}
		},
		State {
			name: windowGameBrowser.getStateName()

			PropertyChanges {
				target: windowGameBrowser
				opacity: 1
			}
		},
		State {
			name: windowJoinedGame.getStateName()

			PropertyChanges {
				target: windowJoinedGame
				opacity: 1
			}
		},
		State {
			name: windowHostGame.getStateName()

			PropertyChanges {
				target: windowHostGame
				opacity: 1
			}
		},
		State {
			name: windowHostedGameDetails.getStateName()

			PropertyChanges {
				target: windowHostedGameDetails
				opacity: 1
			}
		},
		State {
			name: windowMultiplayerCrosswordsClient.getStateName()

			PropertyChanges {
				target: windowMultiplayerCrosswordsClient
				opacity: 1
			}
		},
		State {
			name: windowMultiplayerCrosswordsHost.getStateName()

			PropertyChanges {
				target: windowMultiplayerCrosswordsHost
				opacity: 1
			}
		},
		State {
			name: windowHighscores.getStateName()

			PropertyChanges {
				target: windowHighscores
				opacity: 1
			}
		},
		State {
				name: windowSettings.getStateName()
				PropertyChanges {
					target: windowSettings
					opacity: 1
				}
			}
    ]
}
