/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.WordsList 1.0

Window
{
	id: window

	function getStateName()
	{
		return "Singleplayer - Singleword - Game"
	}


	//this is used to ensure only one instance in whole app
	property WordsList wordsList

	property string letters

	property int lettersCount: 0
	property int roundTime: 0

	Component.onCompleted:
    {
		lettersPanel.solutionTextIinput = solutionTextInput
    }

	function initialize()
	{
		if (lettersCount <= 0)
		{
			if (lettersPanel.letterCount > 0)
				lettersCount = lettersPanel.letterCount
			else
				lettersCount = 10
		}
        timeProgressbar.setProgress(0)
		lettersPanel.letterCount = lettersCount
		letters = wordsList.makeLetters(lettersCount)
		if (!lettersPanel.setLetters(letters))
            return false;
		lettersPanel.setLetterButtonsEnabled(true)
        delButton.enabled = true
		solutionTextInput.text = ""
        doneButton.enabled = true
		bestSolutionResultText.opacity = 0
        bestSolutionText.opacity = 0
        statusText.opacity = 0
        timer.start()
        return true;
    }

	function processKey(event)
	{
		if (event.key === Qt.Key_Return || event.key === Qt.Key_Enter)
			stop(true)
		else if (event.key === Qt.Key_Plus)
			initialize()
		else if (timer.running)//after timer runs out, input can't be changed anymore
		{
			if (event.key === Qt.Key_Backspace)
				deleteLastLetter()
			else	//its some other key, so pass it to solutionTextInput
				solutionTextInput.text = solutionTextInput.text + event.text
		}
	}

    function stop(showResult)
    {
        var bestWord = wordsList.getBestWord()
		var givenWord = solutionTextInput.text
        doneButton.enabled = false
        lettersPanel.setLetterButtonsEnabled(false)
        delButton.enabled = false
        timer.stop()

        if (showResult)
        {
			if (wordsList.isValidWord(givenWord))
			{
				if (validateLettersInWord(givenWord))
				{
					if (givenWord == bestWord || givenWord.length > bestWord.length)
					{
						statusText.text = qsTr("You found best word!")

						//this is in case that user finds better word than best word,
						//so we don't want to make computer look like stupid :)
						bestWord = givenWord
					}
					else
						statusText.text = qsTr("Good found")
				}
				else
					statusText.text = qsTr("Used too many letters")
			}
            else
				statusText.text = qsTr("Invalid word")
            statusText.opacity = 1

            bestSolutionResultText.text = bestWord
            bestSolutionResultText.opacity = 1
            bestSolutionText.opacity = 1
        }
    }

	//checks does word contain more letters than is allowed
	function validateLettersInWord(word)
	{
		var indexArray = new Array()
		for (var i=0; i<word.length; i++)
		{
			var tmpChar = word.charAt(i)
			var tmpIndex = letters.indexOf(tmpChar, indexArray[tmpChar])
			if (tmpIndex < 0)
				return false
			indexArray[tmpChar] = tmpIndex + 1
		}
		return true
	}

	function isGameFinished()
	{
		return lettersPanel.getUnusedLettersCount() == 0 || lettersCount <= 0
	}

	function resume()
	{
		if (!isGameFinished())
			timer.start()
		return true
	}

	function pause()
	{
		timer.stop()
	}

	function deleteLastLetter()
	{
		solutionTextInput.text = solutionTextInput.text.substring(0, solutionTextInput.text.length - 1)
	}

	onHidden: pause();

	onOrientationLandscape: state = "Landscape"
	onOrientationPortrait: state = "Portrait"

	ProgressBar
	{
        id: timeProgressbar
        x: 15
        y: 332
        width: 770
        height: 22
		progress: 0
        anchors.horizontalCenterOffset: 0
        anchors.horizontalCenter: parent.horizontalCenter
        visible: true
        color: "#333333"
		anchors.bottom: lettersPanel.top
		anchors.bottomMargin: 10

		doAnimations: parent.doAnimations
		smooth: parent.smooth
    }

	Timer
	{
		//this is needed since time for vertical progress bar is lagging 1 cycle (if animations are enabled)
		property bool nextStop: false

		id: timer
		//don't forget to modify progress formula to match interval (1000 = 1, 500 = 2...)
		interval: 500
		repeat: true
		onTriggered:
		{
			if (nextStop)
				parent.stop(true)

			//progress bar required formula for proper time keeping
			timeProgressbar.increaseProgress(100 / roundTime / 2)
			if (timeProgressbar.getProgress() >= 100)
				if (window.doAnimations)
					nextStop = true
				else
					parent.stop(true)
		}
		onRunningChanged: nextStop = false
	}

    Rectangle {
        id: solutionRectangle
		x: 240
        y: 229
        width: 399
        height: 49
        color: "#161616"
		anchors.bottom: timeProgressbar.top
		anchors.bottomMargin: 90
		anchors.horizontalCenter: parent.horizontalCenter
        visible: true
        border.width: 2
        border.color: "#646464"

        TextInput {
			id: solutionTextInput
            color: "#ffffff"
            anchors.fill: parent
            font.bold: true
            horizontalAlignment: TextInput.AlignHCenter
			font.pixelSize: 36

			//this is to prevent focus capturing (which can make also blinking cursor in text field)
			enabled: false

			onTextChanged:
			{
				//from keyboard it can be typed in lower case, thats why all chars are uppercased
				text = text.toUpperCase()

				var tmp = lettersPanel.availableLetters
				var correctedText = ""
				//ensuring that given letters are valid (available)
				for (var i=0; i<text.length; i++)
				{
					var checkChar = text.charAt(i)
					var isValid = false
					for (var j=0; j<tmp.length; j++)
					{
						if (checkChar.match(tmp.charAt(j)))
							isValid = true
					}
					if (isValid)
						correctedText = correctedText + checkChar
				}

				//applying filtered word, no need here for uppercase since it is done at beginning
				text = correctedText
			}
        }
    }

	Button
	{
        id: doneButton
		x: 639
		y: 110
        width: 150
        height: 70
        buttonColor: "#880000"

		doAnimations: parent.doAnimations
		smooth: parent.smooth

		text: qsTr("Done")
		anchors.right: parent.right
		anchors.rightMargin: 10
        mouseArea.onReleased: stop(true)
    }

	Button
	{
        id: delButton
		x: 662
        y: 219
		width: 123
        height: 70
		text: qsTr("DEL")

		doAnimations: parent.doAnimations
		smooth: parent.smooth

		anchors.right: parent.right
		anchors.rightMargin: 15
        visible: true
		mouseArea.onReleased: deleteLastLetter()
    }

	Text
	{
        id: solutionText
        x: 8
        y: 232
		color: "#ffffff"
		text: qsTr("Solution:")
		visible: true
        anchors.horizontalCenterOffset: -282
		anchors.horizontalCenter: timeProgressbar.horizontalCenter
		anchors.bottom: timeProgressbar.top
		anchors.bottomMargin: 65
		verticalAlignment: Text.AlignVCenter
        horizontalAlignment: Text.AlignHCenter
		font.pixelSize: 36
	}

	TextFading
	{
        id: bestSolutionText
        y: 154
        color: "#ffffff"
        text: qsTr("Best solution:")
		anchors.left: parent.left
		anchors.leftMargin: 10
        opacity: 0
        font.pixelSize: 36
        visible: true
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
    }

	TextFading
	{
        id: bestSolutionResultText
        y: 154
        color: "#ffee00"
		text: ""
		anchors.left: parent.left
		anchors.leftMargin: 241
        opacity: 0
        style: Text.Normal
        font.bold: true
        font.pixelSize: 36
        visible: true
        horizontalAlignment: Text.AlignLeft
        verticalAlignment: Text.AlignVCenter
    }

	TextFading
	{
        id: statusText
        x: 10
        y: 13
        color: "#ffffff"
        text: ""
        opacity: 0
        font.bold: true
        font.pixelSize: 36
    }

    PanelButtonLetters
    {
        id: lettersPanel
		x: 5
		y: 400

		doAnimations: parent.doAnimations
		smooth: parent.smooth
	}

	 states: [
		 State {
			 name: "Landscape"

			 PropertyChanges {
				 target: timeProgressbar
				 x: 15
				 y: 337
				 anchors.horizontalCenterOffset: 0
				 anchors.bottomMargin: 42
			 }

			 PropertyChanges {
				 target: solutionRectangle
				 x: 241
				 y: 229
				 anchors.bottomMargin: 58
			 }

			PropertyChanges {
				target: doneButton
				x: 320
				y: 110
				width: 150
				anchors.rightMargin: 10
			}

			PropertyChanges {
				target: bestSolutionResultText
				y: 154
				anchors.leftMargin: 290
			}
		 },
		 State {
			 name: "Portrait"

			 PropertyChanges {
				 target: timeProgressbar
				 x: -145
				 y: 591
				 width: 450
				 height: 22
				 anchors.horizontalCenterOffset: 0
			 }

			 PropertyChanges {
				 target: solutionRectangle
				 x: 41
				 y: 544
				 anchors.bottomMargin: 15
			 }

			 PropertyChanges {
				 target: delButton
				 x: 347
				 y: 463
				 anchors.rightMargin: 10
			 }

			 PropertyChanges {
				 target: solutionText
				 x: 41
				 y: 476
				 anchors.horizontalCenterOffset: -89
			 }

			 PropertyChanges {
				 target: statusText
				 x: 10
				 y: 119
			 }

			 PropertyChanges {
				 target: bestSolutionText
				 y: 220
			 }

			 PropertyChanges {
				 target: bestSolutionResultText
				 y: 270
				 anchors.leftMargin: 10
			 }

			PropertyChanges {
				target: doneButton
				x: 56
				y: 328
				width: 370
				height: 70
				anchors.rightMargin: 54
			}
		 }
	 ]


}
