/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.ConfigurationManager 1.0
import crosswords.WordsList 1.0
import "GeneralFunctions.js" as GC

Window
{
	id: window

	property WordsList wordsList

	Component.onCompleted:
	{
		var dicts = wordsList.getDictionaries()
		for (var i=0; i<dicts.length; i++)
			listWidget.addItem(dicts[i])

		lettersValueBox.value = parseInt(configurationManager.getValue("crosswords letters count"))
		rowsValueBox.value = parseInt(configurationManager.getValue("crosswords rows"))
		columnsValueBox.value = parseInt(configurationManager.getValue("crosswords columns"))
		doubleScoreValueBox.value = parseInt(configurationManager.getValue("crosswords double score words count"))
		doubleScoreButton.isChecked = GC.parseBool(configurationManager.getValue("crosswords double score enabled"))
		penaltyScoreButton.isChecked = GC.parseBool(configurationManager.getValue("crosswords penalty score enabled"))
		guaranteeWordButton.isChecked = GC.parseBool(configurationManager.getValue("crosswords guarantee word"))
		expandableWordsButton.isChecked = GC.parseBool(configurationManager.getValue("crosswords expandable words"))

		roundTimeValueBox.value = parseInt(configurationManager.getValue("crosswords round time"))

		roundLimitButton.isChecked = GC.parseBool(configurationManager.getValue("crosswords round limit enabled"))
		roundLimitValueBox.value = parseInt(configurationManager.getValue("crosswords round limit"))
		scorelessRoundsButton.isChecked = GC.parseBool(configurationManager.getValue("crosswords scoreless rounds enabled"))
		scorelessRoundsValueBox.value = parseInt(configurationManager.getValue("crosswords scoreless rounds"))
	}

	function initialize()
	{
		wordsList.setDictionary(configurationManager.getValue("crosswords dictionary"))
		if (wordsList.getSelectedDictionary()!=dictionaryButton.dictionary)
			resumeButton.enabled = false
		//this makes sure that valid dictionary is set
		refreshDictionaryProperties(wordsList.getSelectedDictionary())

		loadingDone()
		return true
	}

	function getStateName()
	{
		return windowSingleplayerCrosswords.getStateName() + " - Settings"
	}

	onOrientationPortrait: state = "Portrait"
	onOrientationLandscape: state = "base_state"

	ConfigurationManager
	{
		id: configurationManager
	}

	function checkIsGuaranteedWordPossible()
	{
		var val = lettersValueBox.value
		if (val < wordsList.getShortestWordLength())
		{
			guaranteeWordButton.isChecked = false
			guaranteeWordButton.enabled = false
		}
		else
			guaranteeWordButton.enabled = true
	}

	function startNewGame()
	{
		//preparing properties for game
		windowSingleplayerCrosswords.rowsCount = rowsValueBox.value
		windowSingleplayerCrosswords.columnsCount = columnsValueBox.value
		windowSingleplayerCrosswords.lettersCount = lettersValueBox.value

		windowSingleplayerCrosswords.roundTime = roundTimeValueBox.value
		windowSingleplayerCrosswords.expandableWords = expandableWordsButton.isChecked
		windowSingleplayerCrosswords.guaranteeWord = guaranteeWordButton.isChecked

		windowSingleplayerCrosswords.doubleScoreEnabled = doubleScoreButton.isChecked
		windowSingleplayerCrosswords.doubleScore = doubleScoreValueBox.value
		windowSingleplayerCrosswords.penaltyScore = penaltyScoreButton.isChecked
		windowSingleplayerCrosswords.roundLimitEnabled = roundLimitButton.isChecked
		windowSingleplayerCrosswords.roundLimit = roundLimitValueBox.value
		windowSingleplayerCrosswords.scorelessRoundsEnabled = scorelessRoundsButton.isChecked
		windowSingleplayerCrosswords.scorelessRounds = scorelessRoundsValueBox.value

		var tmp = wordsList.getInfo().trim()
		if (tmp.length > 0)
			windowSingleplayerCrosswords.displayPopupInformation(tmp, 5)

		//if it can't run it, then last game will probably be messed up
		//so this boolean is perfectly fitting
		resumeButton.enabled = setState(windowSingleplayerCrosswords)
	}

	function refreshDictionaryProperties(dictionary)
	{
		dictionaryButton.dictionary = dictionary
		columnsValueBox.min = wordsList.getShortestWordLength()
		resumeButton.enabled = false
		checkIsGuaranteedWordPossible()
		listWidget.setSelectedValue(dictionary)
	}

	PopupInformation
	{
		id: popup

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	WidgetConfirm
	{
		id: confirmWidget
		text: qsTr("One game is\nalready in progress.\nAre you sure\nthat you want to\nstart new game?")

		onHidden: if (ok) startNewGame()

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	Button {
		id: resumeButton
		x: 580
		y: 410
		enabled: false
		text: qsTr("Resume")
		buttonColor: "#ff0000"
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 20
		anchors.right: parent.right
		anchors.rightMargin: 20

		mouseArea.onReleased:
		{
			if (windowSingleplayerCrosswords.resume())
				setState(windowSingleplayerCrosswords.getStateName())
			else
			{
				enabled = false
				console.debug("Unable to resume game")
			}
		}

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	Button {
		id: startButton
		y: 335
		width: 220
		text: qsTr("Start")
		buttonColor: "#ff0000"
		anchors.left: parent.left
		anchors.leftMargin: 20
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 20

		doAnimations: window.doAnimations
		smooth: window.smooth

		mouseArea.onReleased:
		{
			if (windowSingleplayerCrosswords.isGameFinished())
				startNewGame()
			else
				confirmWidget.opacity = 1
		}
	}

	WidgetList
	{
		id: listWidget
		listWidth: 300
		title: qsTr("Dictionary")
		onHidden:
		{
			if (wordsList.getSelectedDictionary() === selectedValue)
			{
				console.debug("Dictionary \"" + selectedValue + "\" is already set")
				informWordsCount()
			}
			else if(wordsList.setDictionary(selectedValue))
			{
				refreshDictionaryProperties(selectedValue)
				informWordsCount()
				configurationManager.setValue("crosswords dictionary", selectedValue)
			}
			else
				console.debug("Unable to set dictionary: " + selectedValue)
		}

		function informWordsCount()
		{
			popup.showText("<font color=\"#ffff00\">" +
						   qsTr("Dictionary have %n word(s)", "", wordsList.getWordCound()) +
						   "<br/>" +
						   qsTr("Shortest word length: %1").arg(wordsList.getShortestWordLength()) +
						   "<br/>" +
						   qsTr("Longest word length: %1").arg(wordsList.getLongestWordLength()) +
						   "</font>", 5)
		}

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	TextDefault {
		id: settingsText
		x: 274
		y: 15
		text: qsTr("Settings")
		font.bold: true
		anchors.horizontalCenter: parent.horizontalCenter
		anchors.top: parent.top
		anchors.topMargin: 5
		font.pointSize: 20
	}

	Rectangle
	{
		id: settingsRectangle
		anchors.top: settingsText.bottom
		anchors.topMargin: 3
		anchors.bottom: startButton.top
		anchors.bottomMargin: 10
		anchors.right: parent.right
		anchors.rightMargin: 100
		color: "#444444"
		radius: 15
		anchors.left: parent.left
		anchors.leftMargin: 100

		Flickable {
			id: settingsFlickable

			anchors.fill: parent
			contentWidth: settingsFlickable.width
			contentHeight: column.height
			clip: true

			Column {
				id: column

				spacing: 10

				anchors.horizontalCenter: parent.horizontalCenter

				Button {
					id: dictionaryButton

					property string dictionary: ""

					width: 400
					text: qsTr("Dictionary") + (dictionary.length > 0 ? " - " + dictionary : "")
					buttonColor: "#888800"
					anchors.horizontalCenter: parent.horizontalCenter

					doAnimations: window.doAnimations
					smooth: window.smooth

					mouseArea.onReleased: listWidget.opacity = 1
				}

				ValueBox {
					id: rowsValueBox
					x: 122
					text: qsTr("Rows")
					min: 1
					max: 20
					anchors.right: parent.right
					anchors.rightMargin: 0

					onValueChanged: configurationManager.setValue("crosswords rows", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: columnsValueBox
					x: 71
					y: 5
					text: qsTr("Columns")
					min: 2
					max: 20
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0

					onValueChanged: configurationManager.setValue("crosswords columns", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: lettersValueBox

					min: 1
					max: 10

					x: 98
					y: 85
					text: qsTr("Letters")
					anchors.right: columnsValueBox.right
					anchors.rightMargin: 0

					onValueChanged:
					{
						configurationManager.setValue("crosswords letters count", value)
						checkIsGuaranteedWordPossible()
					}

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ButtonCheckbox {
					id: guaranteeWordButton
					x: 148
					width: 340
					height: 70
					text: qsTr("Guarantee word")
					anchors.horizontalCenter: parent.horizontalCenter

					onIsCheckedChanged: configurationManager.setValue("crosswords guarantee word", isChecked)
				}

				ButtonCheckbox {
					id: expandableWordsButton
					x: 148
					width: 340
					height: 70
					text: qsTr("Expandable words")
					anchors.horizontalCenter: parent.horizontalCenter

					onIsCheckedChanged: configurationManager.setValue("crosswords expandable words", isChecked)
				}

				ButtonCheckbox {
					id: doubleScoreButton
					x: -25
					y: 187
					width: 340
					height: 70
					text: qsTr("Double score")
					anchors.horizontalCenter: parent.horizontalCenter

					onIsCheckedChanged: configurationManager.setValue("crosswords double score enabled", isChecked)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: doubleScoreValueBox
					min: 2
					max: 10
					text: qsTr("For")
					anchors.right: columnsValueBox.right
					anchors.rightMargin: 0

					onValueChanged: configurationManager.setValue("crosswords double score words cound", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				TextDefault {
					id: doubleScoreWordsText
					anchors.right: doubleScoreValueBox.right
					anchors.rightMargin: 10
					text: qsTr("(or more words)")
					font.pointSize: 18
				}

				ButtonCheckbox {
					id: penaltyScoreButton
					x: -25
					y: 187
					width: 340
					height: 70
					text: qsTr("Penalty score\nfor unused letters")
					anchors.horizontalCenter: parent.horizontalCenter

					onIsCheckedChanged: configurationManager.setValue("crosswords penalty score enabled", isChecked)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: roundTimeValueBox
					min: 5
					max: 300
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0
					text: qsTr("Round time")

					onValueChanged: configurationManager.setValue("crosswords round time", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ButtonCheckbox {
					id: roundLimitButton
					width: 340
					text: qsTr("Round limit")
					anchors.horizontalCenter: parent.horizontalCenter

					onIsCheckedChanged: configurationManager.setValue("crosswords round limit enabled", isChecked)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: roundLimitValueBox
					min: 1
					max: 100
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0
					text: qsTr("Round limit")

					onValueChanged: configurationManager.setValue("crosswords round limit", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ButtonCheckbox {
					id: scorelessRoundsButton
					width: 340
					text: qsTr("<font size=\"2\">Scoreless rounds</font>")
					anchors.horizontalCenter: parent.horizontalCenter

					onIsCheckedChanged: configurationManager.setValue("crosswords scoreless rounds enabled", isChecked)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

				ValueBox {
					id: scorelessRoundsValueBox
					min: 1
					max: 100
					anchors.right: rowsValueBox.right
					anchors.rightMargin: 0
					text: qsTr("<font size=\"2\">Scoreless rounds</font>")

					onValueChanged: configurationManager.setValue("crosswords scoreless rounds", value)

					doAnimations: window.doAnimations
					smooth: window.smooth
				}

			}
		}
	}



	states: [
		State {
			name: "Portrait"

			PropertyChanges {
				target: startButton
				width: 200
			}

			PropertyChanges {
				target: resumeButton
				width: 200
			}

			PropertyChanges {
				target: popup
				fontSize: 20
			}

			PropertyChanges {
				target: settingsRectangle
				x: 20
				y: 307
				width: 440
				height: 383
				anchors.bottomMargin: 20
				anchors.rightMargin: 20
				anchors.topMargin: 10
				anchors.leftMargin: 20
			}

			PropertyChanges {
				target: settingsText
				x: 20
				y: 274
				anchors.leftMargin: 0
			}

			PropertyChanges {
				target: dictionaryButton
				anchors.bottomMargin: startButton.height + 20
			}
		}
	]

}
