/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.WordsList 1.0

Window
{
	id: window

	function getStateName()
	{
		return "Singleplayer - Crosswords - Game"
	}

	property int lastRowsCount: 0
	property int lastColumnsCount: 0
	property int zoomStep: 5
	property int remainingFreeSlots: 0
	property bool isFinished: true
	//THIS MUST BE ONE CHAR
	property string wordDelimiter: "-"
	onWordDelimiterChanged: if (wordDelimiter.length > 1) wordDelimiter = wordDelimiter.charAt(0)

	//game settings
	property int rowsCount: 0
	property int columnsCount: 0
	property int lettersCount: 0

	property int roundTime: 0
	property bool guaranteeWord: true
	property bool expandableWords: true
	property bool doubleScoreEnabled: false
	property int doubleScore: 0
	property bool penaltyScore: false
	property int penaltyScoreMultiplier: 2
	property bool roundLimitEnabled: false
	property int roundLimit: 0
	property bool scorelessRoundsEnabled: false
	property int scorelessRounds: 0

	//game state
	property int currentRound: 0
	property int currentScorelessRounds: 0


	//this is used to ensure only one instance in whole app
	//behavior is undefined if changed more than once
	property WordsList wordsList
	onWordsListChanged:
	{
		wordsList.errorReadingFile.connect(function()
		{
			singlewordButton.enabled = false
		})
	}

	function initialize()
	{
		if (!crosswordsTable.initialize(rowsCount, columnsCount))
			return false

		if (!lettersCount)
			lettersCount = 10
		lettersPanel.letterCount = lettersCount

		isFinished = false
		currentRound = 0
		currentScorelessRounds = 0
		timeProgressbar.setProgress(0)
		popup.hideTextNow()
		popup.clearQueue()

		//we are getting values from grid directly since rowsOPTIONAL and columnsOPTIONAL
		//can be undefined or modified inside crosswordsTable
		lastRowsCount = crosswordsTable.getRowsCount()
		lastColumnsCount = crosswordsTable.getColumnsCount()
		remainingFreeSlots = lastRowsCount * lastColumnsCount
		if (!nextRound())	//depends on remainingFreeSlots property
			return false

		scoreWidget.opacity = 0
		timeProgressbar.enabled = true
		lettersPanel.setLetterButtonsEnabled(true)
		scoreWidget.reset()
		timer.start()
		return true;
	}

	function resume()
	{
		if (!isFinished && scoreWidget.opacity == 0)
			timer.start()
		return true
	}

	function pause()
	{
		dragger.opacity = 0

		//avoid clearing child, if it is not reseted it can make problem in next situation:
		// - put letter on table
		// - grab that letter and hold
		// - wait until round ends (keep holding)
		// - click to go on next round
		// - drag some letter from panel
		generalMouseArea.clearChild = false

		timer.stop()
	}

	function nextRound()
	{
		if (!isFinished)
		{
			currentRound++
			var letters
			var lettersPlaces = lettersPanel.getLetterPlaces()
			if (guaranteeWord)
				letters = wordsList.makeLetters(lettersPlaces, true)
			else
				letters = wordsList.makeLettersRandom(lettersPlaces)
			lettersPanel.setLetters(letters)
			timeProgressbar.setProgress(0)

			timeProgressbar.enabled = true
			timer.start()
			return true
		}
		return false
	}

	function stop(displayScore)
	{
		pause()
		timeProgressbar.enabled = false

		var scorePenalty = 0
		if (penaltyScore)
			scorePenalty = lettersPanel.getUnusedLettersCount() * penaltyScoreMultiplier
		lettersPanel.setLetterButtonsEnabled(false)

		var foundWords = findNewWords()
		var thisRoundScore = evaluateWords(foundWords)
		thisRoundScore -= scorePenalty
		scoreWidget.addScore(thisRoundScore)

		if (scorelessRoundsEnabled && (thisRoundScore + scorePenalty) <= 0)
			currentScorelessRounds++

		makeFinal()

		if (displayScore)
		{
			if (doubleScoreEnabled && foundWords.length >= doubleScore)
				popup.showText("<font color=\"#ffff00\">" + qsTr("DOUBLE SCORE") + "</font>")
			if (penaltyScore && scorePenalty > 0)
				popup.showText("<font color=\"#ffff00\">" + qsTr("Penalty: -%1").arg(scorePenalty) + "</font>")
			while (foundWords.length > 0)
			{
				var word = foundWords[0]
				var newArray = new Array()
				var count = 1
				for (var i=1; i<foundWords.length; i++)
				{
					var tmpWord = foundWords[i]
					if (tmpWord == word)
						count++
					else
						newArray.push(tmpWord)
				}
				var text = "<font color=\"#aaaaaa\">" + qsTr("Found word") + ":</font> " + word
				if (count > 1)
					text += " <font color=\"#ff0000\">x" + count + "</font>"
				popup.showText(text)
				foundWords = newArray
			}
			scoreWidget.opacity = 1
		}
		//check is game finished
		var reason = isFinished = isGameFinished()
		if (isFinished)
		{
			popup.showText("<font color=\"#ff0000\"><b>" + qsTr("GAME OVER") +
						   "</b><br/>" + reason + "</font>", 0)
			windowHighscores.addHighscore(scoreWidget.totalScore)
		}
	}

	function isGameFinished()
	{
		if (remainingFreeSlots < 0)
			console.debug("Error in isGameFinished(): remainingFreeSlots = " + remainingFreeSlots)

		if (remainingFreeSlots <= 0)
			return qsTr("Table is full")
		if (roundLimitEnabled && currentRound >= roundLimit)
			return qsTr("Round limit reached")
		if (scorelessRoundsEnabled && currentScorelessRounds >= scorelessRounds)
			return qsTr("Scoreless round limit reached")

		return false
	}

	function makeFinal()
	{
		var childs = crosswordsTable.gridChildren
		for (var i=0; i<childs.length; i++)
		{
			var tmp = childs[i]
			if (tmp.letter != "")
				tmp.isFinal = true
		}
	}

	function findNewWords()
	{
		var foundNewWords = new Array()
		var verticalWords = new Array()
		var verticalNew = new Array()
		for (var i=0; i<crosswordsTable.getColumnsCount(); i++)
		{
			verticalWords[i] = ""
			verticalNew[i] = false
		}
		for (var i=0; i<crosswordsTable.getRowsCount(); i++)
		{
			var horizontalWord = "", horizontalNew = false
			for (var j=0; j<crosswordsTable.getColumnsCount(); j++)
			{
				var index = i * crosswordsTable.getColumnsCount() + j
				var child = crosswordsTable.gridChildren[index]
				if (child && child.letter != "" && child.letter != wordDelimiter)
				{
					horizontalWord += child.letter
					verticalWords[j] += child.letter
					if (!child.isFinal)
					{
						remainingFreeSlots = remainingFreeSlots - 1
						horizontalNew = true
						verticalNew[j] = true
					}
				}
				else
				{
					if (horizontalNew && horizontalWord.length > 0 && wordsList.isValidWord(horizontalWord))
					{
						foundNewWords.push(horizontalWord)
						makeButtonUnacceptable(child)
						if (horizontalWord.length + 1 <= j)
							makeButtonUnacceptable(crosswordsTable.gridChildren[index - horizontalWord.length - 1])
					}
					if (verticalNew[j] && verticalWords[j].length > 0 && wordsList.isValidWord(verticalWords[j]))
					{
						foundNewWords.push(verticalWords[j])
						makeButtonUnacceptable(child)
						if (verticalWords[j].length + 1 <= i)
						{
							var vertIndex = (i - verticalWords[j].length - 1) * crosswordsTable.getColumnsCount() + j
							makeButtonUnacceptable(crosswordsTable.gridChildren[vertIndex])
						}
					}
					horizontalWord = ""
					horizontalNew = false
					verticalWords[j] = ""
					verticalNew[j] = false
				}
				if (j == crosswordsTable.getColumnsCount() - 1)
					if (horizontalNew && horizontalWord.length > 0 && wordsList.isValidWord(horizontalWord))
					{
						foundNewWords.push(horizontalWord)
						//we know for sure that last child is not empty in this row
						//(since index is now pointing to first child in next row)
						//in other words, this word is at the end of row
						if (horizontalWord.length <= j)
							makeButtonUnacceptable(crosswordsTable.gridChildren[index - horizontalWord.length])
					}
			}
		}
		for (var i=0; i<verticalWords.length; i++)
			if (verticalNew[i] && wordsList.isValidWord(verticalWords[i]))
			{
				foundNewWords.push(verticalWords[i])
				if (verticalWords[i].length + 1 <= crosswordsTable.getRowsCount())
				{
					var vertIndex = (crosswordsTable.getRowsCount() - verticalWords[i].length - 1) * crosswordsTable.getColumnsCount() + i
					makeButtonUnacceptable(crosswordsTable.gridChildren[vertIndex])
				}
			}
		return foundNewWords
	}

	function makeButtonUnacceptable(button)
	{
		if (!expandableWords && button.letter=="")
		{
			remainingFreeSlots = remainingFreeSlots - 1
			button.letter = wordDelimiter
			button.isFinal = true
		}
	}

	function evaluateWords(words)
	{
		var score = 0
		if (words)
		{
			var dinstinct = new Array()
			var distinctScoreMultiplier = new Array()
			for (var i=0; i<words.length; i++)
			{
				var tmp = words[i]
				//initialize multiplier
				distinctScoreMultiplier[i] = 1
				var haveDuplicate = false
				for (var j=0; j<dinstinct.length; j++)
					if (dinstinct[j] == tmp)
					{
						distinctScoreMultiplier[j] += 2
						haveDuplicate = true
						break
					}
				if (!haveDuplicate)
					dinstinct.push(tmp)
			}
			for (var i=0; i<dinstinct.length; i++)
			{
				var tmp = dinstinct[i]
				score += tmp.length * distinctScoreMultiplier[i]
				var tmpLetters = ""
				for (var j=0; j<tmp.length; j++)
					if (tmpLetters.match(tmp[j]))
						score++
					else
						tmpLetters += tmp[j]
			}
			if (doubleScoreEnabled && dinstinct.length >= doubleScore)
				score *= 2
		}
		return score
	}

	function processKey(event)
	{
		//FIXME: VolumeUp and VolumeDown doesn't work on n900
		if (event.key == Qt.Key_VolumeUp || event.key == Qt.Key_Plus)
			crosswordsTable.buttonSize += zoomStep
		else if (event.key == Qt.Key_VolumeDown || event.key == Qt.Key_Minus)
			crosswordsTable.buttonSize -= zoomStep
	}

	function displayPopupInformation(text, seconds)
	{
		popup.showText(text, seconds)
	}

	onHidden: pause()

	//it is used also to track score (not just to show)
	WidgetScore
	{
		id: scoreWidget
		onHidden: nextRound()

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	PopupInformation
	{
		id: popup

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	ProgressBarVertical
	{
        id: timeProgressbar
        color: "#333333"
		anchors.left: crosswordsTable.right
		anchors.top: totalScoreText.bottom
		anchors.right: parent.right
		anchors.bottom: lettersPanel.top
		anchors.leftMargin: 5
		anchors.rightMargin: 5
		anchors.topMargin: 2
		anchors.bottomMargin: 5

		updateInterval: timer.interval

		doAnimations: window.doAnimations
		smooth: window.smooth

		text: qsTr("DONE")
		textColor: "#22ffffff"
		textSize: 32

		MouseArea
		{
			id: mouseArea
			anchors.fill: parent
			onPressed: timeProgressbar.state = "Pressed"
			onReleased: stop(true)
		}

		states: [
			State {
				name: "Pressed"
				when: mouseArea.pressed

				PropertyChanges {
					target: timeProgressbar
					color: "#555555"
				}
			}
		]
    }

    Timer
    {
		//this is needed since time for vertical progress bar is lagging 1 cycle (if animations are enabled)
		property bool nextStop: false

        id: timer
		//don't forget to modify progress formula to match interval (1000 = 1, 500 = 2...)
		interval: 500
        repeat: true
        onTriggered:
        {
			if (nextStop)
				parent.stop(true)

			//progress bar required formula for proper time keeping
			timeProgressbar.increaseProgress(100 / roundTime / 2)
			if (timeProgressbar.getProgress() >= 100)
				if (window.doAnimations)
					nextStop = true
				else
					parent.stop(true)
		}
		onRunningChanged: nextStop = false
    }

	property int totalScoreCache: scoreWidget.totalScore
	Behavior on totalScoreCache
	{
		NumberAnimation
		{
			duration: (window.doAnimations ? 2500 : 0)
			easing.type: Easing.InOutQuad
		}
	}

	TextDefault
	{
		id: totalScoreText
		x: 695
		y: 86
		width: 70
		text: window.totalScoreCache

		color: "#ffff00"
		font.pixelSize: 20

		anchors.right: parent.right
		anchors.top: parent.top
		anchors.topMargin: 67
	}

	TextDefault
	{
		id: roundInfoText
		anchors.top: window.top
		anchors.left: window.left
		anchors.leftMargin: 5
		text: qsTr("Round") + " " + window.currentRound + (window.roundLimitEnabled ? "/" + window.roundLimit : "")

		color: "#ffff00"
		font.pixelSize: 20
	}

	TextDefault
	{
		id: scorelessRoundsInfoText
		anchors.top: roundInfoText.top
		anchors.left: roundInfoText.right
		anchors.leftMargin: 20
		opacity: (window.scorelessRoundsEnabled ? 1 : 0)
		text: qsTr("Scoreless round") + " " + window.currentScorelessRounds + "/" + window.scorelessRounds

		color: "#ffff00"
		font.pixelSize: 20
	}

	CrosswordsTable {
		id: crosswordsTable
		width: 685
		height: 470
		anchors.right: totalScoreText.left
		anchors.rightMargin: 0
		anchors.bottom: lettersPanel.top
		anchors.bottomMargin: 5
		anchors.left: parent.left
		anchors.leftMargin: 5
		anchors.top: roundInfoText.bottom
		anchors.topMargin: 0

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	PanelButtonLetters
	{
		id: lettersPanel
		anchors.bottomMargin: 5

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	ButtonLetter
	{
		id: dragger
		opacity: 0

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	MouseArea
	{
		id: generalMouseArea

		anchors.top: parent.top
		anchors.left: parent.left
		anchors.bottom: parent.bottom
		anchors.right: parent.right

		property ButtonLetter child
		property bool clearChild: false

		//used for speed optimization
		property int draggerWidth
		property int draggerHeight

		onPressed:
		{
			if (timer.running && dragger.opacity == 0)
			{
				child = lettersPanel.childAt(mouse.x - lettersPanel.x, mouse.y - lettersPanel.y)
				if (child && child.enabled)
				{
					dragger.letter = child.letter
					draggerWidth = dragger.width / 2
					draggerHeight = dragger.height / 2
					dragger.x = mouse.x - draggerWidth
					dragger.y = mouse.y - draggerHeight
					dragger.opacity = 1
					generalMouseArea.onPositionChangedState.connect(dragOnDemand)
					playDrag()
					return
				}
				//allow dragging/moving already putted letters on table
				child = crosswordsTable.getChildAt(mouse.x - crosswordsTable.x, mouse.y - crosswordsTable.y)
				if (child && child!=null && !child.isFinal && child.letter!=null && child.letter!="")
				{
					dragger.letter = child.letter
					draggerWidth = dragger.width / 2
					draggerHeight = dragger.height / 2
					dragger.x = mouse.x - draggerWidth
					dragger.y = mouse.y - draggerHeight
					dragger.opacity = 1
					generalMouseArea.onPositionChangedState.connect(dragOnDemand)
					clearChild = true
					playDrag()
					return
				}
			}
			//not event for lettersPanel so we are forwarding this event to lower mouseArea listeners
			mouse.accepted = false
		}

		//this is used to avoid constant mouse position event processing
		//and this is workaround for signals in javascript
		signal onPositionChangedState(variant mouse)
		onPositionChanged: onPositionChangedState(mouse)

		function dragOnDemand(mouse)
		{
			if (timer.running)
			{
				dragger.x = mouse.x - draggerWidth
				dragger.y = mouse.y - draggerHeight
			}
		}

		onReleased:
		{
			//if nothing is dragged then skip this signal
			if (timer.running && dragger.opacity > 0)
			{
				//check if letter should be dragged back to letters panel
				if (mouse.x > lettersPanel.x && mouse.y > lettersPanel.y)
				{
					playDropInvalid()
					if (clearChild)
					{
						lettersPanel.setOneLetterButtonEnabled(dragger.letter, true)
						child.letter = ""
					}
					//else source (dragger) is from letters panel and we don't want to remove letter there
				}
				else
				{
					//else do we have valid letter location
					var toDrop = crosswordsTable.getChildAt(mouse.x - crosswordsTable.x, mouse.y - crosswordsTable.y)
					if (toDrop && !toDrop.isFinal)
					{
						//it is pressed on same place (or same letter) so it means that we need to remove
						//letter that is clicked or letter from which was dragged
						if (toDrop.letter == dragger.letter)
						{
							lettersPanel.setOneLetterButtonEnabled(toDrop.letter, true)
						}
						else
						{
							//either we are returning letter to letters panel or we want to override some other letter
							if (clearChild || toDrop.letter!="")
								lettersPanel.setOneLetterButtonEnabled(toDrop.letter, true)
							toDrop.letter = dragger.letter
						}
						//in case of moving letter on table
						if (clearChild)
							child.letter = ""
						else	//else it is dragged from letters table
							child.enabled = false
						playDrop()
					}
					else
					{
						console.debug("Not valid place given for dropping button")
						playDropInvalid()
					}
				}
				//reset values
				clearChild = false
				dragger.opacity = 0
				generalMouseArea.onPositionChangedState.disconnect(dragOnDemand)
			}
		}
	}
}
