/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1

Window
{
	id: window

	property string gameName
	property int maxPlayers: 2
	property bool lock: false
	property bool startingGame: false

	onMaxPlayersChanged:
	{
		if (!lock)
		{
			lock = true
			var tmp = networkServer.setMaxPlayers(maxPlayers)
			if (tmp != maxPlayers)
				maxPlayers = tmp
			lock = false
		}
	}

	function getStateName()
	{
		return "Hosted Game Details"
	}

	function resumeToGame()
	{
		startingGame = true
		networkServer.gameResume()
		setState(windowMultiplayerCrosswordsHost.getStateName())
	}

	function initialize()
	{
		playersListScroll.clear()
		networkServer.setHostName(configurationManager.getValue("name"))
		networkServer.setMaxPlayers(maxPlayers)
		networkServer.announceGame(gameName)
		resumeButton.enabled = false
		return true
	}

	function sendParametersToPlayers()
	{
		networkServer.sendParameters(wordsListSingleton.getSelectedDictionary(),
													   windowMultiplayerCrosswordsHost.rowsCount,
													   windowMultiplayerCrosswordsHost.columnsCount,
													   windowMultiplayerCrosswordsHost.lettersCount,
													   windowMultiplayerCrosswordsHost.guaranteeWord,
													   windowMultiplayerCrosswordsHost.expandableWords,
													   windowMultiplayerCrosswordsHost.doubleScoreEnabled,
													   windowMultiplayerCrosswordsHost.doubleScore,
													   windowMultiplayerCrosswordsHost.penaltyScore,
													   windowMultiplayerCrosswordsHost.roundTime,
													   windowMultiplayerCrosswordsHost.roundLimitEnabled,
													   windowMultiplayerCrosswordsHost.roundLimit,
													   windowMultiplayerCrosswordsHost.scorelessRoundsEnabled,
													   windowMultiplayerCrosswordsHost.scorelessRounds)
	}

	onShow:
	{
		if (windowMultiplayerCrosswordsHost.isFinished)
			if (doAnimations)
				setState(-1)
			else
				stateDelay.start()
		else
		{
			resumeConfirm.opacity = 0
			startingGame = false
		}
	}

	Timer
	{
		id: stateDelay
		interval: 100
		repeat: false
		onTriggered: setState(-1)
	}

	onHidden:
	{
		networkServer.stopAnnouncingGame()
		if (!startingGame)
			networkServer.kick("")
	}

	onOrientationPortrait: state = "Portrait"
	onOrientationLandscape: state = "base_state"

	Connections
	{
		target: networkServer

		onPlayerConnected:
		{
			sendParametersToPlayers()
			playersListScroll.addText(playerName)
		}

		onPlayerDisconnected: playersListScroll.removeText(playerName)

		onResumeGame: if (opacity == 1 && resumeButton.enabled) resumeConfirm.opacity = 1
	}



	WidgetConfirm
	{
		id: resumeConfirm
		text: qsTr("Game is resumed!<br/>Do you want to join?")

		onHidden: if (ok) resumeToGame()
	}

	ListScroll
	{
		id: playersListScroll
		anchors.rightMargin: 20
		anchors.leftMargin: 20
		anchors.topMargin: 20
		anchors.bottom: startButton.top
		anchors.right: parent.right
		anchors.left: parent.left
		anchors.top: parent.top
		anchors.bottomMargin: 10

		selectable: false

		onSelectedValueChanged:
		{
			if (value != "")
			{
				networkServer.kick(value)
				playersListScroll.removeText(value)
			}
		}

		doAnimations: parent.doAnimations
		smooth: parent.smooth
	}

	PopupInformation
	{
		id: popup

		doAnimations: parent.doAnimations
		smooth: parent.smooth
	}

	Button
	{
		id: startButton
		text: qsTr("Start")
		buttonColor: "#ff0000"

		anchors.left: parent.left
		anchors.leftMargin: 20
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 10

		mouseArea.onReleased:
		{
			if (networkServer.startLoading())
			{
				startingGame = true
				resumeButton.enabled = true
				setState(windowMultiplayerCrosswordsHost)
			}
			else
			{
				popup.showText("<font color=\"#ffff00\">" +
							   qsTr("Parameters are not sent to<br/>all players or there are<br/>no enough players.") +
							   "</font>", 5)
				console.debug("Parameters are not sent to all clients or there are not enough players!");
				sendParametersToPlayers()
			}
		}

		doAnimations: parent.doAnimations
		smooth: parent.smooth
	}

	Button {
		id: resumeButton
		x: 580
		y: 410
		enabled: false
		text: qsTr("Resume")
		buttonColor: "#ff0000"
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 10
		anchors.right: parent.right
		anchors.rightMargin: 20

		mouseArea.onReleased: resumeToGame()

		doAnimations: window.doAnimations
		smooth: window.smooth
	}

	states: [
		State {
			name: "Portrait"

			PropertyChanges {
				target: startButton
				width: 200
			}

			PropertyChanges {
				target: resumeButton
				width: 200
			}
		}
	]

}
