/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import crosswords.ConfigurationManager 1.0

Window
{
	property int maxHighscores: 10
	property variant highscores: new Array()

	property int maxListWidth: 500

	function getStateName()
	{

		return "Highscores"
	}

	Component.onCompleted: loadHighscores()
	Component.onDestruction: saveHighscores()

	onHidden: confirmWidget.opacity = 0

	function initialize()
	{
		updateList()
		return true
	}

	function addHighscore(newHighscore)
	{
		if (typeof newHighscore != "number")
		{
			try
			{
				var tmp = newHighscore.trim().toLowerCase()
				if (tmp == "" || tmp == "nan")
					throw ""
				newHighscore = parseInt(newHighscore)
			}
			catch(e)
			{
				console.debug("Invalid new highscore given: " + newHighscore)
				return
			}
		}

		var tmpHighscores = highscores
		tmpHighscores.push(newHighscore)
		tmpHighscores.sort(function(x,y){return parseInt(y) - parseInt(x)})
		if (tmpHighscores.length > maxHighscores)
			tmpHighscores.length = maxHighscores
		highscores = tmpHighscores
	}

	function updateList()
	{
		listScroll.clear()
		if (highscores.length > 0)
			//i<maxHighscores - this is needed if maxHighscores is changed but not called loadHighscores
			//this can happen if changed in options, and then switched to highscores window
			for (var i=0; i<highscores.length && i<maxHighscores; i++)
				listScroll.addText(highscores[i])
		else
			console.debug("Unable to update list for highscores: " + highscores)
	}

	function loadHighscores()
	{
		highscores = new Array()
		var highscoresData = configurationManager.getValue("highscores")
		try
		{
			var highscoresArray = highscoresData.split(",")
			for (var i=0; i<highscoresArray.length; i++)
				addHighscore(highscoresArray[i])
		}
		catch (e)
		{
			console.debug("No previous highscores found")
		}
	}

	function saveHighscores()
	{
		configurationManager.setValue("highscores", highscores.join(","))
	}


	ConfigurationManager
	{
		id: configurationManager
	}

	WidgetConfirm
	{
		id: confirmWidget
		text: qsTr("Do you really\nwant to delete\nall highscores?")

		onHidden:
		{
			if (ok)
			{
				highscores = new Array()
				saveHighscores()
				updateList()
			}
		}
	}

	ListScroll
	{
		id: listScroll

		width: (parent.width - listScroll.anchors.bottomMargin - listScroll.anchors.topMargin < maxListWidth ?
					parent.width - listScroll.anchors.bottomMargin - listScroll.anchors.topMargin : maxListWidth)

		anchors.horizontalCenter: parent.horizontalCenter
		anchors.bottom: clearHighscoresButton.top
		anchors.top: parent.top
		anchors.bottomMargin: 10
		anchors.topMargin: 10

		selectable: false
	}

	Button {
		id: clearHighscoresButton
		x: 290
		y: 370
		width: 350
		height: 70
		text: qsTr("Clear highscores")
		anchors.bottom: parent.bottom
		anchors.bottomMargin: 10
		anchors.horizontalCenter: parent.horizontalCenter

		mouseArea.onReleased: confirmWidget.opacity = 1
	}
}
