#ifndef WORDSLIST_H
#define WORDSLIST_H

/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QObject>
#include <QFile>
#include <QDir>
#include <QTextStream>
#include <QApplication>
#include <qmath.h>
#include <QVariantList>

class WordsList : public QObject
{
		Q_OBJECT
	private:
		int longestWord, shortestWord;
		QVariantList dictionaries;
		QString dictionary, info;

	public:
		static const QString DEFAULT_DICTIONARY, PATH, EXTENSION;

		explicit WordsList():
			QObject()
		{
			words = new QList<QString>();
			info = "";

			QDir dictDir(PATH);
			if (dictDir.exists())
			{
				dictDir.setFilter(QDir::Files | QDir::Hidden | QDir::NoSymLinks);
				dictDir.setSorting(QDir::Name);
				QFileInfoList fileList = dictDir.entryInfoList();
				for (int i=0; i<fileList.size(); i++)
				{
					QString fileName = fileList.at(i).fileName();
					if (fileName.endsWith(EXTENSION))
						dictionaries.append(fileName.mid(0, fileName.lastIndexOf(EXTENSION)));
				}
			}
			else
			{
				error = true;
				emit errorReadingFile();
				qDebug("Dictionaries directory doesn't exist!");
			}
		}

		~WordsList()
		{
			try
			{
				words->clear();
				delete words;
			}
			catch (...)
			{
				qDebug("Error while destroying words list!");
			}
		}

		Q_INVOKABLE QString getInfo()
		{
			return info;
		}

		Q_INVOKABLE int getWordCound()
		{
			return words->size();
		}

		Q_INVOKABLE QString getSelectedDictionary()
		{
			return dictionary;
		}

		Q_INVOKABLE QVariantList getDictionaries()
		{
			return dictionaries;
		}

		Q_INVOKABLE bool setDictionary(QString dictionary)
		{
			if (dictionary.trimmed().isEmpty())
				return false;
			if (this->dictionary == dictionary)
			{
				qDebug("Same dictionary is already set (" + dictionary.toAscii() + "). Skipping...");
				return false;
			}

			this->dictionary = dictionary;
			error = true;
			pickedWord = QString("");
			words->clear();

			//this is loaded from resources.qrc file
			file = new QFile(PATH + dictionary + EXTENSION);
			if (!file->exists())
			{
				setDictionary(DEFAULT_DICTIONARY);
				return false;
			}
			if (file->open(QIODevice::ReadOnly))
			{
				QTextStream ts(file);

				longestWord = 0;
				shortestWord = 999999;
				QString word;
				if (!ts.atEnd())
					chars = ts.readLine().trimmed().toUpper();
				if (!ts.atEnd())
					info = ts.readLine().trimmed();
				while (!ts.atEnd())
				{
					word = ts.readLine().trimmed().toUpper();
					words->append(word);
					int len = word.length();
					if (len > longestWord)
						longestWord = len;
					if (len < shortestWord)
						shortestWord = len;
				}

				error = false;
				emit created();
			}
			else
			{
				qDebug("Error occured while reading words file!");
				qDebug(file->errorString().toAscii());
				emit errorReadingFile();

				//set default to avoid any further errors
				setDictionary(DEFAULT_DICTIONARY);
			}
			try
			{
				file->close();
				delete file;
			}
			catch (...)
			{
				qDebug("Error while closing words file!");
			}
			return !error;
		}

		Q_INVOKABLE int getLongestWordLength()
		{
			return longestWord;
		}

		Q_INVOKABLE int getShortestWordLength()
		{
			return shortestWord;
		}

		Q_INVOKABLE QString makeLetters(int amount = 10, bool trim = false)
		{
			if (error || words->size() <= 0)
			{
				emit errorReadingFile();
				return QString("");
			}
			try
			{
				pickedWord = words->at(qrand() % words->size());
				QString letters, tmpLetters;
				tmpLetters = pickedWord = pickWord(amount, trim);
				for (int i=0; i<amount - pickedWord.length(); i++)
					tmpLetters.append(randomChar());
				for (int i=0; i<amount; i++)
				{
					QChar tmp = randomChar(&tmpLetters);
					letters.append(tmp);
					int occurences = tmpLetters.count(tmp);
					tmpLetters.remove(tmp);
					if (occurences > 1)
						for (int j=0; j<occurences - 1; j++)
							tmpLetters.append(tmp);
				}
				return letters;
			}
			catch (...)
			{
				qDebug("Error occured while making letters!");
			}
			return "";
		}

		Q_INVOKABLE QString makeLettersRandom(int amount = 10)
		{
			if (error || words->size() <= 0)
			{
				emit errorReadingFile();
				return QString("");
			}
			try
			{
				QString letters;
				letters = "";
				for (int i=0; i<amount; i++)
					letters.append(randomChar());
				return letters;
			}
			catch (...)
			{
				qDebug("Error occured while making letters!");
			}
			return "";
		}

		Q_INVOKABLE QString getBestWord()
		{
			if (error)
			{
				emit errorReadingFile();
				return QString("");
			}
			return pickedWord;
		}

		Q_INVOKABLE bool isValidWord(QString word)
		{
			if (error)
			{
				emit errorReadingFile();
				return false;
			}
			return words->contains(word.toUpper());
		}

	signals:
		void created();
		void errorReadingFile();

	public slots:

	private:
		QFile* file;
		QList<QString>* words;
		QString pickedWord;
		QString chars;
		bool error;

		QChar randomChar(QString* charList = NULL)
		{
			if (charList == NULL)
				return chars.at(qrand() % chars.length());
			else
				return charList->at(qrand() % charList->length());
		}

		/**
		 * Picks word for max defined length, if max length is out of scope,
		 * it is trimmed to near lowest/highest possible value.
		 * @brief pickWord
		 * @param maxWordLength
		 * @return
		 */
		QString pickWord(int maxWordLength = -1, bool trim = false)
		{
			//in case that wirds list is empty
			if (words->size() <= 0)
				return "";

			if (maxWordLength < shortestWord)
			{
				if (trim)
					maxWordLength = shortestWord;
				else
					return "";
			}
			else if (maxWordLength > longestWord)
			{
				if (trim)
					maxWordLength = longestWord;
				else
					return "";
			}

			int minWordLength = maxWordLength - 3;
			if (minWordLength < shortestWord)
				minWordLength = shortestWord;
			int length;
			QString word;
			while ((length = (word = words->at(qrand() % words->length())).length()) > maxWordLength ||
				   length < minWordLength);
			return word;
		}

};

#endif // WORDSLIST_H
