#ifndef NETWORKCLIENT_H
#define NETWORKCLIENT_H

/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QObject>
#include <QVariantList>
#include <QStringList>
#include <QtNetwork/QHostAddress>
#include <QList>

#include "NetworkServer.h"
#include "HostedGameInfoHolder.h"
#include "NetworkFunctions.h"

class NetworkClient : public QObject
{
		Q_OBJECT
	private:
		QUdpSocket gameListener;
		QList<Network::HostedGameInfoHolder*> games;
		bool searchingHostedGames;

		//actual game
		QTcpSocket serverSocket;
		QString playerName;

	public:
		explicit NetworkClient():
			QObject()
		{
		}

		Q_INVOKABLE bool searchHostedGames()
		{
			if (gameListener.bind(Network::GAME_ANNOUNCING_PORT, QUdpSocket::ShareAddress))
			{
				clearGames();
				searchingHostedGames = true;
				connect(&gameListener, SIGNAL(readyRead()), this, SLOT(readData()));
				return true;
			}
			qDebug("Unable to bind client to listen game announcements!");
			return false;
		}

		Q_INVOKABLE void stopSearchingHostedGames()
		{
			gameListener.close();
			disconnect(&gameListener, SIGNAL(readyRead()), this, SLOT(readData()));
			searchingHostedGames = false;
		}

		Q_INVOKABLE bool isSearchingHostedGames()
		{
			return searchingHostedGames;
		}

		Q_INVOKABLE void setName(QString name)
		{
			qDebug("Name set for client: " + name.toAscii());
			this->playerName = name;
		}

		Q_INVOKABLE QVariantList getHostedGames()
		{
			QVariantList gameNames;
			gameNames.clear();
			for (int i=0; i<games.length(); i++)
				gameNames.append(games.at(i)->getGameName());
			return gameNames;
		}

		/**
		 * Used for refreshing game list (with newer content)
		 */
		Q_INVOKABLE void clearGames()
		{
			games.clear();
		}

		/**
		 * @return TRUE if connection is INITIAILIZING, FALSE if something gone wrong
		 * @return Please note that this is not returning TRUE if connection is successfull,
		 * @return instead it will emit signal connectedToHost()
		 */
		Q_INVOKABLE bool connectToGame(QString gameName)
		{
			for (int i=0; i<games.length(); i++)
			{
				Network::HostedGameInfoHolder* host = games.at(i);
				if (host->getGameName() == gameName)
				{
					connect(&serverSocket, SIGNAL(connected()), this, SLOT(processConnection()));
					connect(&serverSocket, SIGNAL(disconnected()), this, SLOT(processDisconnection()));
					serverSocket.connectToHost(host->getAddress(), host->getPort());
					return true;
				}
			}
			return false;
		}

		Q_INVOKABLE void closeConnection()
		{
			serverSocket.disconnectFromHost();
		}

		Q_INVOKABLE void requirePlayersList()
		{
			sendMessage(Network::GET_PLAYERS_LIST);
		}

		Q_INVOKABLE void loadingDone()
		{
			sendMessage(Network::GAME_LOADING_FINISHED);
		}

		Q_INVOKABLE void makeMoveInTable(qint8 fromIndex, qint8 toIndex = -1)
		{
			QString tmp = QString::number(fromIndex) + Network::CODE_SEPARATOR;
			tmp += QString::number(toIndex);
			sendMessage(Network::MOVE_IN_TABLE, tmp);
		}

		Q_INVOKABLE void makeMoveFromPanel(QString letter, qint8 toIndex)
		{
			QString tmp = letter + Network::CODE_SEPARATOR;
			tmp += QString::number(toIndex);
			sendMessage(Network::MOVE_FROM_PANEL, tmp);
		}

		Q_INVOKABLE void turnEnded()
		{
			sendMessage(Network::TURN_ENDED);
		}

		Q_INVOKABLE void requestPause()
		{
			sendMessage(Network::GAME_REQUEST_PAUSE);
		}

		Q_INVOKABLE void requestResume()
		{
			sendMessage(Network::GAME_REQUEST_RESUME);
		}

	private slots:
		//used only for game announcements
		void readData()
		{
			while (gameListener.hasPendingDatagrams())
			{
				QByteArray datagram;
				QHostAddress hostAddress;
				datagram.resize(gameListener.pendingDatagramSize());
				gameListener.readDatagram(datagram.data(), datagram.size(), &hostAddress);
				try
				{
					QString message(datagram);
					bool ok;
					qint8 code = Network::cutFirstParameter(message).toInt(&ok);
					if (!ok || code != Network::GAME_ANNOUNCE)
					{
						qDebug("UdpSocket: Unable to process code from message! (code: " + QString::number(code).toAscii() + ")");
						continue;
					}
					QString gameName = Network::cutFirstParameter(message);
					qint16 hostPort = Network::cutFirstParameter(message).toInt(&ok);
					if (!ok)
					{
						qDebug("Unable to process port from message!");
						return;
					}
					for (int i=0; i<games.length(); i++)
						if (games.at(i)->getGameName() == gameName)
							//Game is already in list - skipping...
							return;
					games.append(new Network::HostedGameInfoHolder(gameName, hostAddress, hostPort));
					emit newGameFound(gameName);
					qDebug("New game found: " + gameName.toAscii());

				}
				catch (...)
				{
					qDebug("Error while reading received data: " + QString(datagram).toAscii());
				}
			}
		}

		void processConnection()
		{
			disconnect(&serverSocket, SIGNAL(connected()), this, SLOT(processConnection()));
			connect(&serverSocket, SIGNAL(readyRead()), this, SLOT(processMessage()));
			connect(&serverSocket, SIGNAL(disconnected()), this, SLOT(processDisconnection()));
			sendMessage(Network::PLAYER_NAME, this->playerName);
			emit connectedToHost();
		}

		void processDisconnection()
		{
			disconnect(&serverSocket, SIGNAL(readyRead()), this, SLOT(processMessage()));
			disconnect(&serverSocket, SIGNAL(disconnected()), this, SLOT(processDisconnection()));
			emit disconnectedFromHost();
			qDebug("- - - Disconnected from host. - - -");
		}

		void processMessage()
		{
			QString allMessages = serverSocket.readAll();
			QStringList lines = allMessages.split(Network::END_OF_MESSAGE);
			//last item is empty since in every message at end we (must) have END_OF_MESSAGE character
			for (int i=0; i<lines.size() - 1; i++)
			{
				QString line = lines.at(i);
				if (line.trimmed().isEmpty())
					continue;
#if defined(QT_DEBUG)
				//TODO: REMOVE THIS - WHEN DEBUGGING FINISHED
				qDebug("CLIENT MESSAGE: " + line.toAscii());
#endif
				bool ok;
				qint8 code = Network::cutFirstParameter(line).toInt(&ok);
				if (!ok)
				{
					qDebug("TcpSocket: Unable to process code from message!");
					continue;
				}
				processCode(code, line);
			}
		}

		void processCode(qint8& code, QString& message)
		{
			switch (code)
			{
				case Network::GAME_FULL:
					emit gameFull();
					break;

				case Network::GAME_PARAMETERS:
				{
					//brackets are required when declaring new variables inside switch/case statement to define scope
					QString dictionary = Network::cutFirstParameter(message);
					qint8 maxPlayers = Network::cutFirstParameter(message).toInt();
					qint8 rows = Network::cutFirstParameter(message).toInt();
					qint8 columns = Network::cutFirstParameter(message).toInt();
					qint8 lettersCount = Network::cutFirstParameter(message).toInt();
					bool guaranteeWord = Network::parseBool(Network::cutFirstParameter(message));
					bool expandableWords = Network::parseBool(Network::cutFirstParameter(message));
					bool doubleScoreEnabled = Network::parseBool(Network::cutFirstParameter(message));
					qint8 doubleScore = Network::cutFirstParameter(message).toInt();
					bool penaltyForUnusedLetters = Network::parseBool(Network::cutFirstParameter(message));
					qint8 roundTime = Network::cutFirstParameter(message).toInt();
					bool roundLimitEnabled = Network::parseBool(Network::cutFirstParameter(message));
					qint8 roundLimit = Network::cutFirstParameter(message).toInt();
					bool scorelessRoundLimitEnabled = Network::parseBool(Network::cutFirstParameter(message));
					qint8 scorelessRoundLimit = Network::cutFirstParameter(message).toInt();

					emit gameParameters(dictionary, maxPlayers, rows, columns, lettersCount,
										guaranteeWord, expandableWords, doubleScoreEnabled,
										doubleScore, penaltyForUnusedLetters, roundTime,
										roundLimitEnabled, roundLimit,
										scorelessRoundLimitEnabled, scorelessRoundLimit);
					break;
				}

				case Network::PLAYERS_LIST:
				{
					QString playerName;
					QVariantList playerNames;
					while (!(playerName = Network::cutFirstParameter(message)).trimmed().isEmpty())
						playerNames.append(playerName);
					emit gotPlayerNames(playerNames);
					break;
				}

				case Network::GAME_START_LOADING:
					emit startLoading();
					break;

				case Network::CURRENT_TURN:
				{
					qDebug("CURRENT TURN: " + message.toAscii());
					QString playerName = Network::cutFirstParameter(message);
					QString letters = Network::cutFirstParameter(message);
					if (playerName == this->playerName)
						emit yourTurn(letters);
					else
						emit currentPlayerTurn(playerName, letters);
					break;
				}

				case Network::SET_LETTER:
				{
					QString letter = Network::cutFirstParameter(message);
					qint8 index = Network::cutFirstParameter(message).toInt();
					emit setLetterAt(letter, index);
					//NOTE: for some reason you can't make call like this:
//						emit setLetterAt(Network::cutFirstParameter(message),
//										 Network::cutFirstParameter(message).toInt());
					//because it sends instead "letter", "number", it sends like this: "number", "NONE"
					//it is still mystery why...

					break;
				}

				case Network::YOUR_SCORE:
				{
					qint16 score = Network::cutFirstParameter(message).toInt();
					bool doubleScore = Network::cutFirstParameter(message).toLower().trimmed() == "true";
					qint8 penaltyScore = Network::cutFirstParameter(message).toInt();
					emit addMyScore(score, doubleScore, penaltyScore);
					break;
				}

				case Network::GAME_FOUND_WORD:
				{
					QString word = Network::cutFirstParameter(message);
					qint8 count = Network::cutFirstParameter(message).toInt();
					emit foundWord(word, count);
					break;
				}

				case Network::PLAYER_KICKED:
					emit kicked(message);
					break;

				case Network::PLAYER_KICKED_NEED_DIFFERENT_PLAYER_NAME:
					emit serverRequiresDifferentPlayerName();
					break;

				case Network::PLAYER_LEFT_GAME:
					emit playerLeftGame(message);
					break;

				case Network::GAME_FINISHED_FULL_TABLE:
				{
					QString winner = Network::cutFirstParameter(message);
					qint16 winnerScore = Network::cutFirstParameter(message).toInt();
					emit gameFinishedFullTable(winner, winnerScore);
					break;
				}

				case Network::GAME_FINISHED_ROUND_LIMIT_REACHED:
				{
					QString winner = Network::cutFirstParameter(message);
					qint16 winnerScore = Network::cutFirstParameter(message).toInt();
					emit gameFinishedRoundLimitReached(winner, winnerScore);
					break;
				}

				case Network::GAME_FINISHED_SCORELESS_ROUND_LIMIT_REACHED:
				{
					QString winner = Network::cutFirstParameter(message);
					qint16 winnerScore = Network::cutFirstParameter(message).toInt();
					emit gameFinishedScorelessRoundLimitReached(winner, winnerScore);
					break;
				}

				case Network::GAME_PAUSED:
					emit pauseGame(message);
					break;

				case Network::GAME_RESUMED:
					emit resumeGame(message);
					break;


					//TODO: IMPLEMENT ALL CODES (that are needed)
				default:
					qDebug("Unknown code received");
			}
		}


	signals:
		void newGameFound(QString gameName);
		void connectedToHost();
		void disconnectedFromHost();
		void gameFull();
		void kicked(QString reason = "");
		void serverRequiresDifferentPlayerName();
		void playerLeftGame(QString playerName);
		void gameParameters(QString dictionary, qint8 maxPlayers, qint8 rows, qint8 columns, qint8 lettersCount,
							bool guaranteeWord, bool expandableWords, bool doubleScoreEnabled,
							qint8 doubleScore, bool penaltyForUnusedLetters, qint8 roundTime,
							bool roundLimitEnabled, qint8 roundLimit,
							bool scorelessRoundLimitEnabled, qint8 scorelessRoundLimit);
		void gotPlayerNames(QVariantList playerNames);
		void startLoading();
		void yourTurn(QString letters);
		void currentPlayerTurn(QString playerName, QString letters);
		//can also be used for unseting (just put empty "letter")
		void setLetterAt(QString letter, qint8 index);
		void addMyScore(qint16 score, bool doubleScore, qint8 penaltyScore);
		void foundWord(QString word, qint8 count);
		void pauseGame(QString playerName);
		void resumeGame(QString playerName);
		void gameFinishedFullTable(QString winner, qint16 winnerScore);
		void gameFinishedRoundLimitReached(QString winner, qint16 winnerScore);
		void gameFinishedScorelessRoundLimitReached(QString winner, qint16 winnerScore);

	private:
		void sendMessage(Network::NetworkDictionary code, QString message = "")
		{
			if (!serverSocket.isOpen() || !serverSocket.isWritable())
			{
				qDebug("Unable to write message to server!");
				return;
			}
			QString preparedMessage(QString::number(code).toAscii());
			if (!message.isEmpty())
				preparedMessage += Network::CODE_SEPARATOR + message;
			preparedMessage += Network::END_OF_MESSAGE;
#if defined(QT_DEBUG)
			//TODO: REMOVE THIS - WHEN DEBUGGING FINISHED
			qDebug("CLIENT SEND MESSAGE: " + preparedMessage.toAscii());
#endif
			serverSocket.write(QByteArray(preparedMessage.toAscii()));
		}

};

#endif // NETWORKCLIENT_H
