#ifndef LANGUAGELOADER_H
#define LANGUAGELOADER_H

/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QObject>
#include <QApplication>
#include <QTranslator>
#include <QDir>
#include <QVariantList>

/**
 * SINGLETON
 */
class LanguageLoader : public QObject
{
		Q_OBJECT
	private:
		static QTranslator* qtrans;
		static QString language;
	public:
		static const QString TRANSLATION_PATH;
		static const QString DEFAULT_LANGUAGE;

		/**
		 * Installs translator to QApplication, but it can be done only before app is created.
		 * @brief Can be called only once before application is loaded
		 * @param qtapp
		 */
		static void installTranslatorToApplication(QApplication* qtapp)
		{
			setLanguageStatic();
			qtapp->installTranslator(qtrans);
		}

		explicit LanguageLoader():
			QObject()
		{
		}

		/**
		 * It is wrapper for static method so it can be called from QML
		 */
		Q_INVOKABLE bool setLanguage(QString language = DEFAULT_LANGUAGE)
		{
			return LanguageLoader::setLanguageStatic(language);
		}

		static bool setLanguageStatic(QString language = DEFAULT_LANGUAGE)
		{
			try
			{
				if (qtrans->load(TRANSLATION_PATH + language))
				{
					LanguageLoader::language = language;
					return true;
				}
				else
				{
					LanguageLoader::language = DEFAULT_LANGUAGE;
					qtrans->load(TRANSLATION_PATH + DEFAULT_LANGUAGE);
					return false;
				}
			}
			catch (...)
			{
				qDebug("Error happended in class LanguageLoader, method setLanguage()");
			}
			return false;
		}

		Q_INVOKABLE QVariantList getAvailableLanguages()
		{
			QVariantList languages;
			try
			{
				languages.clear();
				QDir langDir(":/");
				langDir.setFilter(QDir::Files | QDir::Hidden | QDir::NoSymLinks);
				langDir.setSorting(QDir::Name);
				QFileInfoList fileList = langDir.entryInfoList();
				for (int i=0; i<fileList.size(); i++)
				{
					QString fileName = fileList.at(i).fileName();
					if (fileName.startsWith("translation_") && fileName.endsWith(".qm"))
						languages.append(fileName.mid(12, fileName.lastIndexOf(".qm") - 12));
				}
			}
			catch (...)
			{
				qDebug("Error while loading languages in class LanguageLoader, method getLanguages()");
			}
			return languages;
		}

		static QString getLanguageStatic()
		{
			return language;
		}

		Q_INVOKABLE QString getLanguage()
		{
			return getLanguageStatic();
		}
};

#endif // LANGUAGELOADER_H
