#ifndef CONFIGURATIONMANAGER_H
#define CONFIGURATIONMANAGER_H

/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QObject>
#include <QDir>
#include <QFile>
#include <QVariantList>
#include <QMap>
#include <QList>
#include <QTextStream>

#include "LanguageLoader.h"
#include "WordsList.h"

/**
 * SINGLETON
 *
 * DO NOT USE QMap<QString, QString> instead QList<QString>.
 * At first it is more logical to use, but for some reason it doesn't work under Maemo operating system.
 *
 * @brief The ConfigurationManager class
 */
class ConfigurationManager : public QObject
{
		Q_OBJECT
	private:
		static const QString directoryPath;
		static const QString fileName;

		static QList<QString> config;
		static QFile configFile;

		static bool isInitializationDone;

		static void initializeStatic()
		{
			if (isInitializationDone)
				return;

			if (!configFile.exists())
			{
				QDir* dir = new QDir(directoryPath);
				if (!dir->exists())
					dir->mkpath(directoryPath);
				if (configFile.open(QIODevice::ReadWrite))
				{
					configFile.close();

					//DEFAULT VALUES
					setValueStatic("language", LanguageLoader::DEFAULT_LANGUAGE);
					setValueStatic("animate", "true");
					setValueStatic("smooth", "true");
					setValueStatic("ask quit", "true");
					setValueStatic("name", "Player");
					setValueStatic("game name", "Game name");

					setValueStatic("crosswords letters count", "10");
					setValueStatic("crosswords rows", "10");
					setValueStatic("crosswords columns", "12");
					setValueStatic("crosswords double score words count", "5");
					setValueStatic("crosswords double score enabled", "true");
					setValueStatic("crosswords guarantee word", "true");
					setValueStatic("crosswords expandable words", "true");
					setValueStatic("crosswords dictionary", WordsList::DEFAULT_DICTIONARY);
					setValueStatic("crosswords round time", "60");
					setValueStatic("crosswords round limit", "5");
					setValueStatic("crosswords round limit enabled", "false");
					setValueStatic("crosswords scoreless rounds", "3");
					setValueStatic("crosswords scoreless rounds enabled", "false");

					setValueStatic("crosswords max highscores", "10");
					setValueStatic("crosswords penalty score enabled", "false");

					saveStatic();

					qDebug("New configuration file is created.");
				}
				else
					qDebug("Unable to create new configuration file!");
			}
			loadStatic();
			isInitializationDone = true;
		}

		static void loadStatic()
		{
			try
			{
				config.clear();
				configFile.open(QIODevice::ReadOnly);
				QTextStream qts(&configFile);
				qts.setCodec("UTF-8");
				QStringList data;
				while (!qts.atEnd())
				{
					data = qts.readLine().split("=");
					//empty data should be skipped
					if (data.size()!=2 || data.at(0).trimmed().isEmpty())
						continue;
					setValueStatic(data.at(0), data.at(1));
				}
				configFile.close();
			}
			catch (...)
			{
				qDebug("Unable to save configuration data!");
				configFile.close();
			}
		}

		static void saveStatic()
		{
			try
			{
				configFile.open(QIODevice::WriteOnly);
				QTextStream qts(&configFile);
				qts.setCodec("UTF-8");
				for (int i=0; i<config.size(); i++)
				{
					//for some ackward reason this can't be used directly
					//qts << config.at(i).append("\n");
					QString tmp = config.at(i);
					qts << tmp.append("\n");
				}
				configFile.close();
			}
			catch (...)
			{
				qDebug("Unable to save configuration data!");
				configFile.close();
			}
		}

		static void setValueStatic(QString fieldName, QString value)
		{
			if (fieldName.trimmed().isEmpty())
				return;
			for (int i=config.size() - 1; i>=0; i--)
				if (config.at(i).startsWith(fieldName + "="))
					config.removeAt(i);
			config.append(fieldName + "=" + value);
		}

		static QString getValueStatic(QString fieldName)
		{
			for (int i=config.size() - 1; i>=0; i--)
				if (config.at(i).startsWith(fieldName + "="))
					return config.at(i).mid(fieldName.length() + 1);
			return "";
		}


	public:
		//all these metheds are only reference to static ones

		explicit ConfigurationManager():
			QObject()
		{
			initializeStatic();
		}

		~ConfigurationManager()
		{
			save();
		}

		Q_INVOKABLE void setValue(QString fieldName, QString value)
		{
			setValueStatic(fieldName, value);
		}

		Q_INVOKABLE QString getValue(QString fieldName)
		{
			return getValueStatic(fieldName);
		}

		Q_INVOKABLE void save()
		{
			saveStatic();
		}

};

#endif // CONFIGURATIONMANAGER_H
