#ifndef AUDIOPLAYER_H
#define AUDIOPLAYER_H

/**
Copyright (c) 2012, DRAX <drax@drax.biz>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of the FreeBSD Project.
*/

#include <QObject>
#include <QTemporaryFile>
#include <QResource>

#if defined(SOUND_SAFEMODE)
//phonon
#include <phonon/MediaObject>
#include <phonon/AudioOutput>
#include "PlayThread.h"
#else
//qsound
#include <QSound>
#endif

/**
 * @brief Audio player - supports only WAV files at the moment!
 */
class AudioPlayer : public QObject
{
		Q_OBJECT
	private:
		static const QString PATH;
		static const QString EXTENSION;

		QTemporaryFile* tmpFile;
#if defined(SOUND_SAFEMODE)
		Phonon::MediaObject* mMediaObject;
		Phonon::AudioOutput* mAudioOutput;
		Phonon::MediaSource* mMediaSource;
#else
		QSound* sound;
#endif

	public:
		explicit AudioPlayer():
			QObject()
		{
			tmpFile = NULL;
#if defined(SOUND_SAFEMODE)
			mMediaObject = Phonon::createPlayer(Phonon::MusicCategory);
			mAudioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
			mAudioOutput->setVolume(0.5);
			Phonon::createPath(mMediaObject, mAudioOutput);

			mMediaSource = NULL;
#else
			sound = NULL;
#endif
		}

		virtual ~AudioPlayer()
		{
			if (tmpFile != NULL)
				delete tmpFile;
#if defined(SOUND_SAFEMODE)
			delete mMediaObject;
			delete mAudioOutput;
			if (mMediaSource!=NULL)
				delete mMediaSource;
#else
			delete sound;
#endif
		}

		Q_INVOKABLE void setAudioFile(QString audioFileName)
		{
			//this is needed since phonon is bugging with direct resource file streaming
			if (tmpFile != NULL)
				delete tmpFile;
			tmpFile = new QTemporaryFile("Crosswords_sound_temp");
			tmpFile->open();
			QResource res(PATH + audioFileName + EXTENSION);
			tmpFile->write((char*)res.data(),res.size());
			tmpFile->flush();
			tmpFile->setAutoRemove(true);
			tmpFile->close();

#if defined(SOUND_SAFEMODE)
			if (mMediaSource!=NULL)
				delete mMediaSource;
#else
			if (sound != NULL)
				delete sound;
			sound = new QSound(tmpFile->fileName());
#endif
		}

		Q_INVOKABLE void play()
		{
#if defined(SOUND_SAFEMODE)
			if (mMediaSource!=NULL)
				delete mMediaSource;
			mMediaSource = new Phonon::MediaSource(tmpFile);
			mMediaObject->setCurrentSource(*mMediaSource);
			new PlayThread(mMediaObject);
#else
			if (sound != NULL)
				sound->play();
			else
				qDebug("QSound not initialized!");
#endif
		}

		Q_INVOKABLE bool isPlayable()
		{
#if defined(SOUND_SAFEMODE)
			return tmpFile != NULL;
#else
			return sound != NULL && sound->isAvailable();
#endif
		}
};

#endif // AUDIOPLAYER_H
