/*
 *  Copyright (C) 2009 Ludovic Jacomme (ludovic.jacomme@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <fcntl.h>

#include <SDL/SDL.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "cap32.h"
#include "z80.h"
#include "psp_kbd.h"
#include "kbd.h"
#include "psp_menu.h"
#include "psp_sdl.h"
#include "psp_danzeff.h"

# define KBD_MIN_ANALOG_TIME      150000
# define KBD_MIN_START_TIME      1500000
# define KBD_MAX_EVENT_TIME       500000
# define KBD_MIN_PENDING_TIME     300000
# define KBD_MIN_DANZEFF_TIME     150000
# define KBD_MIN_COMMAND_TIME     100000
# define KBD_MIN_BATTCHECK_TIME 90000000
# define KBD_MIN_AUTOFIRE_TIME   1000000

 static gp2xCtrlData    loc_button_data;
 static unsigned int   loc_last_event_time = 0;
 static long           first_time_stamp = -1;
 static long           first_time_auto_stamp = -1;
 static char           loc_button_press[ KBD_MAX_BUTTONS ];
 static char           loc_button_release[ KBD_MAX_BUTTONS ];
 static unsigned int   loc_button_mask[ KBD_MAX_BUTTONS ] =
 {
   GP2X_CTRL_UP         , /*  KBD_UP         */
   GP2X_CTRL_RIGHT      , /*  KBD_RIGHT      */
   GP2X_CTRL_DOWN       , /*  KBD_DOWN       */
   GP2X_CTRL_LEFT       , /*  KBD_LEFT       */
   GP2X_CTRL_TRIANGLE   , /*  KBD_TRIANGLE   */
   GP2X_CTRL_CIRCLE     , /*  KBD_CIRCLE     */
   GP2X_CTRL_CROSS      , /*  KBD_CROSS      */
   GP2X_CTRL_SQUARE     , /*  KBD_SQUARE     */
   GP2X_CTRL_SELECT     , /*  KBD_SELECT     */
   GP2X_CTRL_START      , /*  KBD_START      */
   GP2X_CTRL_LTRIGGER   , /*  KBD_LTRIGGER   */
   GP2X_CTRL_RTRIGGER   , /*  KBD_RTRIGGER   */
   GP2X_CTRL_FIRE,        /*  KBD_FIRE       */
 };

 static char loc_button_name[ KBD_ALL_BUTTONS ][20] =
 {
   "UP",
   "RIGHT",
   "DOWN",
   "LEFT",
# if defined(DINGUX_MODE)
   "X",      // Triangle
   "A",      // Circle
   "B",      // Cross
   "Y",      // Square
# else
   "Y",      // Triangle
   "B",      // Circle
   "X",      // Cross
   "A",      // Square
# endif
   "SELECT",
   "START",
   "LTRIGGER",
   "RTRIGGER",
   "JOY_FIRE",
   "JOY_UP",
   "JOY_RIGHT",
   "JOY_DOWN",
   "JOY_LEFT"
 };

 static char loc_button_name_L[ KBD_ALL_BUTTONS ][20] =
 {
   "L_UP",
   "L_RIGHT",
   "L_DOWN",
   "L_LEFT",
# if defined(DINGUX_MODE)
   "L_X",      // Triangle
   "L_A",      // Circle
   "L_B",      // Cross
   "L_Y",      // Square
# else
   "L_Y",      // Triangle
   "L_B",      // Circle
   "L_X",      // Cross
   "L_A",      // Square
# endif
   "L_SELECT",
   "L_START",
   "L_LTRIGGER",
   "L_RTRIGGER",
   "L_JOY_FIRE",
   "L_JOY_UP",
   "L_JOY_RIGHT",
   "L_JOY_DOWN",
   "L_JOY_LEFT"
 };

  static char loc_button_name_R[ KBD_ALL_BUTTONS ][20] =
 {
   "R_UP",
   "R_RIGHT",
   "R_DOWN",
   "R_LEFT",
# if defined(DINGUX_MODE)
   "R_X",      // Triangle
   "R_A",      // Circle
   "R_B",      // Cross
   "R_Y",      // Square
# else
   "R_Y",      // Triangle
   "R_B",      // Circle
   "R_X",      // Cross
   "R_A",      // Square
# endif
   "R_SELECT",
   "R_START",
   "R_LTRIGGER",
   "R_RTRIGGER",
   "R_JOY_FIRE",
   "R_JOY_UP",
   "R_JOY_RIGHT",
   "R_JOY_DOWN",
   "R_JOY_LEFT"
 };

  static int loc_default_mapping[ KBD_ALL_BUTTONS ] = {
    CPC_CUR_UP          , /*  KBD_UP         */
    CPC_CUR_RIGHT       , /*  KBD_RIGHT      */
    CPC_CUR_DOWN        , /*  KBD_DOWN       */
    CPC_CUR_LEFT        , /*  KBD_LEFT       */
    CPC_ENTER           , /*  KBD_TRIANGLE   */
    CPC_J0_FIRE2        , /*  KBD_CIRCLE     */
    CPC_J0_FIRE1        , /*  KBD_CROSS      */
    CPC_DEL             , /*  KBD_SQUARE     */
    -1                  , /*  KBD_SELECT     */
    -1                  , /*  KBD_START      */
    KBD_LTRIGGER_MAPPING  , /*  KBD_LTRIGGER   */
    KBD_RTRIGGER_MAPPING  , /*  KBD_RTRIGGER   */
    CPC_J0_FIRE1        , /*  KBD_JOY_FIRE   */
    CPC_J0_UP           , /*  KBD_JOY_UP     */
    CPC_J0_RIGHT        , /*  KBD_JOY_RIGHT  */
    CPC_J0_DOWN         , /*  KBD_JOY_DOWN   */
    CPC_J0_LEFT           /*  KBD_JOY_LEFT   */
  };

  static int loc_default_mapping_L[ KBD_ALL_BUTTONS ] = {
    CPC_J0_UP           , /*  KBD_UP         */
    CPC_J0_RIGHT        , /*  KBD_RIGHT      */
    CPC_J0_DOWN         , /*  KBD_DOWN       */
    CPC_J0_LEFT         , /*  KBD_LEFT       */
    CAP32_LOAD          , /*  KBD_TRIANGLE   */
    CAP32_RENDER        , /*  KBD_CIRCLE     */
    CAP32_SAVE          , /*  KBD_CROSS      */
    CAP32_FPS           , /*  KBD_SQUARE     */
    -1                  , /*  KBD_SELECT     */
    -1                  , /*  KBD_START      */
    KBD_LTRIGGER_MAPPING  , /*  KBD_LTRIGGER   */
    KBD_RTRIGGER_MAPPING  , /*  KBD_RTRIGGER   */
    CPC_J0_FIRE1        , /*  KBD_JOY_FIRE   */
    CPC_CUR_UP          , /*  KBD_JOY_UP     */
    CPC_CUR_RIGHT       , /*  KBD_JOY_RIGHT  */
    CPC_CUR_DOWN        , /*  KBD_JOY_DOWN   */
    CPC_CUR_LEFT          /*  KBD_JOY_LEFT   */
 };

  static int loc_default_mapping_R[ KBD_ALL_BUTTONS ] = {
    CPC_J0_UP           , /*  KBD_UP         */
    CAP32_INCFIRE       , /*  KBD_RIGHT      */
    CPC_J0_DOWN         , /*  KBD_DOWN       */
    CAP32_DECFIRE       , /*  KBD_LEFT       */
    CPC_SPACE           , /*  KBD_TRIANGLE   */
    CPC_ESC             , /*  KBD_CIRCLE     */
    CAP32_AUTOFIRE      , /*  KBD_CROSS      */
    CPC_DEL             , /*  KBD_SQUARE     */
    -1                  , /*  KBD_SELECT     */
    -1                  , /*  KBD_START      */
    KBD_LTRIGGER_MAPPING  , /*  KBD_LTRIGGER   */
    KBD_RTRIGGER_MAPPING  , /*  KBD_RTRIGGER   */
    CPC_J0_FIRE1        , /*  KBD_JOY_FIRE   */
    CPC_CUR_UP          , /*  KBD_JOY_UP     */
    CPC_CUR_RIGHT       , /*  KBD_JOY_RIGHT  */
    CPC_CUR_DOWN        , /*  KBD_JOY_DOWN   */
    CPC_CUR_LEFT          /*  KBD_JOY_LEFT   */
 };

        int psp_kbd_mapping[ KBD_ALL_BUTTONS ];

 int psp_kbd_mapping[ KBD_ALL_BUTTONS ];
 int psp_kbd_mapping_L[ KBD_ALL_BUTTONS ];
 int psp_kbd_mapping_R[ KBD_ALL_BUTTONS ];
 int psp_kbd_presses[ KBD_ALL_BUTTONS ];
 int kbd_ltrigger_mapping_active;
 int kbd_rtrigger_mapping_active;

 static int danzeff_cpc_key     = 0;
 static int danzeff_cpc_pending = 0;
        int danzeff_mode        = 0;


       char command_keys[ 128 ];
 static int command_mode        = 0;
 static int command_index       = 0;
 static int command_size        = 0;
 static int command_cpc_pending = 0;
 static int command_cpc_key     = 0;

void
psp_kbd_run_command(char *Command, int wait)
{
  strncpy(command_keys, Command, 128);
  command_size  = strlen(Command);
  command_index = 0;

  command_cpc_key     = 0;
  command_cpc_pending = 0;
  command_mode        = wait;
}

void
psp_kbd_default_settings()
{
  memcpy(psp_kbd_mapping  , loc_default_mapping, sizeof(loc_default_mapping));
  memcpy(psp_kbd_mapping_L, loc_default_mapping_L, sizeof(loc_default_mapping_L));
  memcpy(psp_kbd_mapping_R, loc_default_mapping_R, sizeof(loc_default_mapping_R));
}

int
psp_kbd_reset_hotkeys(void)
{
  int index;
  int key_id;
  for (index = 0; index < KBD_ALL_BUTTONS; index++) {
    key_id = loc_default_mapping[index];
    if ((key_id >= CAP32_FPS) && (key_id <= CAP32_SCREEN)) {
      psp_kbd_mapping[index] = key_id;
    }
    key_id = loc_default_mapping_L[index];
    if ((key_id >= CAP32_FPS) && (key_id <= CAP32_SCREEN)) {
      psp_kbd_mapping_L[index] = key_id;
    }
    key_id = loc_default_mapping_R[index];
    if ((key_id >= CAP32_FPS) && (key_id <= CAP32_SCREEN)) {
      psp_kbd_mapping_R[index] = key_id;
    }
  }
  return 0;
}

int
psp_kbd_load_mapping_file(FILE *KbdFile)
{
  char     Buffer[512];
  char    *Scan;
  int      tmp_mapping[KBD_ALL_BUTTONS];
  int      tmp_mapping_L[KBD_ALL_BUTTONS];
  int      tmp_mapping_R[KBD_ALL_BUTTONS];
  int      cpc_key_id = 0;
  int      kbd_id = 0;

  memcpy(tmp_mapping  , loc_default_mapping  , sizeof(loc_default_mapping));
  memcpy(tmp_mapping_L, loc_default_mapping_L, sizeof(loc_default_mapping_R));
  memcpy(tmp_mapping_R, loc_default_mapping_R, sizeof(loc_default_mapping_R));

  while (fgets(Buffer,512,KbdFile) != (char *)0) {

      Scan = strchr(Buffer,'\n');
      if (Scan) *Scan = '\0';
      /* For this #@$% of windows ! */
      Scan = strchr(Buffer,'\r');
      if (Scan) *Scan = '\0';
      if (Buffer[0] == '#') continue;

      Scan = strchr(Buffer,'=');
      if (! Scan) continue;

      *Scan = '\0';
      cpc_key_id = atoi(Scan + 1);

      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name[kbd_id])) {
          tmp_mapping[kbd_id] = cpc_key_id;
          //break;
        }
      }
      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name_L[kbd_id])) {
          tmp_mapping_L[kbd_id] = cpc_key_id;
          //break;
        }
      }
      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name_R[kbd_id])) {
          tmp_mapping_R[kbd_id] = cpc_key_id;
          //break;
        }
      }
  }

  memcpy(psp_kbd_mapping, tmp_mapping, sizeof(psp_kbd_mapping));
  memcpy(psp_kbd_mapping_L, tmp_mapping_L, sizeof(psp_kbd_mapping_L));
  memcpy(psp_kbd_mapping_R, tmp_mapping_R, sizeof(psp_kbd_mapping_R));

  return 0;
}

int
psp_kbd_load_mapping(char *kbd_filename)
{
  FILE    *KbdFile;
  int      error = 0;

  KbdFile = fopen(kbd_filename, "r");
  error   = 1;

  if (KbdFile != (FILE*)0) {
    psp_kbd_load_mapping_file(KbdFile);
    error = 0;
    fclose(KbdFile);
  }

  kbd_ltrigger_mapping_active = 0;
  kbd_rtrigger_mapping_active = 0;

  return error;
}


int
psp_kbd_save_mapping(char *kbd_filename)
{
  FILE    *KbdFile;
  int      kbd_id = 0;
  int      error = 0;

  KbdFile = fopen(kbd_filename, "w");
  error   = 1;

  if (KbdFile != (FILE*)0) {

    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name[kbd_id], psp_kbd_mapping[kbd_id]);
    }
    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name_L[kbd_id], psp_kbd_mapping_L[kbd_id]);
    }
    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name_R[kbd_id], psp_kbd_mapping_R[kbd_id]);
    }
    error = 0;
    fclose(KbdFile);
  }

  return error;
}

int
psp_kbd_enter_command()
{
  gp2xCtrlData  c;

  unsigned int command_key = 0;
  int          cpc_key     = 0;

  gp2xCtrlPeekBufferPositive(&c, 1);

  if (command_cpc_pending)
  {
    if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_COMMAND_TIME) {
      loc_last_event_time = c.TimeStamp;
      command_cpc_pending = 0;
      cpc_key_event(command_cpc_key, 0);
    }

    return 0;
  }

  if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_COMMAND_TIME) {
    loc_last_event_time = c.TimeStamp;

    if (command_mode > 1) {
      command_mode--;
      return 0;
    }

    if (command_index >= command_size) {

      command_mode  = 0;
      command_index = 0;
      command_size  = 0;

      command_cpc_pending = 0;
      command_cpc_key     = 0;

      return 0;
    }

    command_key = command_keys[command_index++];
    cpc_key = cpc_get_key_from_ascii(command_key);

    if (cpc_key != -1) {
      command_cpc_key     = cpc_key;
      command_cpc_pending = 1;
      cpc_key_event(command_cpc_key, 1);
    }

    return 1;
  }

  return 0;
}

int
psp_kbd_is_danzeff_mode()
{
  return danzeff_mode;
}

int
psp_kbd_enter_danzeff()
{
  unsigned int danzeff_key = 0;
  int          cpc_key     = 0;
  gp2xCtrlData  c;

  if (! danzeff_mode) {
    psp_init_keyboard();
    danzeff_mode = 1;
  }

  gp2xCtrlPeekBufferPositive(&c, 1);

  if (danzeff_cpc_pending)
  {
    if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_PENDING_TIME) {
      loc_last_event_time = c.TimeStamp;
      danzeff_cpc_pending = 0;
      cpc_key_event(danzeff_cpc_key, 0);
    }
    return 0;
  }

  if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_DANZEFF_TIME) {
    loc_last_event_time = c.TimeStamp;

    gp2xCtrlPeekBufferPositive(&c, 1);
    danzeff_key = danzeff_readInput(c);
  }

  if (danzeff_key > DANZEFF_START) {
    cpc_key = cpc_get_key_from_ascii(danzeff_key);

    if (cpc_key != -1) {
      danzeff_cpc_key     = cpc_key;
      danzeff_cpc_pending = 1;
      cpc_key_event(danzeff_cpc_key, 1);
    }

    return 1;

  } else if (danzeff_key == DANZEFF_START) {
    danzeff_mode        = 0;
    danzeff_cpc_pending = 0;
    danzeff_cpc_key     = 0;

    psp_kbd_wait_no_button();

  } else if (danzeff_key == DANZEFF_SELECT) {
    danzeff_mode        = 0;
    danzeff_cpc_pending = 0;
    danzeff_cpc_key     = 0;
    psp_main_menu();
    psp_init_keyboard();

    psp_kbd_wait_no_button();
  }

  return 0;
}

int
cpc_decode_key(int psp_b, int button_pressed)
{
  int wake = 0;
  int reverse_cursor = ! CPC.psp_reverse_analog;

  if (reverse_cursor) {
    if ((psp_b >= KBD_JOY_UP  ) &&
        (psp_b <= KBD_JOY_LEFT)) {
      psp_b = psp_b - KBD_JOY_UP + KBD_UP;
    } else
    if ((psp_b >= KBD_UP  ) &&
        (psp_b <= KBD_LEFT)) {
      psp_b = psp_b - KBD_UP + KBD_JOY_UP;
    }
  }

  if (psp_b == KBD_START) {
     if (button_pressed) psp_kbd_enter_danzeff();
  } else
  if (psp_b == KBD_SELECT) {
    if (button_pressed) {
      psp_main_menu();
      psp_init_keyboard();
    }
  } else {

    if (psp_kbd_mapping[psp_b] >= 0) {
      wake = 1;
      if (button_pressed) {
        // Determine which buton to press first (ie which mapping is currently active)
        if (kbd_ltrigger_mapping_active) {
          // Use ltrigger mapping
          psp_kbd_presses[psp_b] = psp_kbd_mapping_L[psp_b];
          cpc_key_event(psp_kbd_presses[psp_b], button_pressed);
        } else
        if (kbd_rtrigger_mapping_active) {
          // Use rtrigger mapping
          psp_kbd_presses[psp_b] = psp_kbd_mapping_R[psp_b];
          cpc_key_event(psp_kbd_presses[psp_b], button_pressed);
        } else {
          // Use standard mapping
          psp_kbd_presses[psp_b] = psp_kbd_mapping[psp_b];
          cpc_key_event(psp_kbd_presses[psp_b], button_pressed);
        }
      } else {
          // Determine which button to release (ie what was pressed before)
          cpc_key_event(psp_kbd_presses[psp_b], button_pressed);
      }

    } else {
      if (psp_kbd_mapping[psp_b] == KBD_LTRIGGER_MAPPING) {
        kbd_ltrigger_mapping_active = button_pressed;
        kbd_rtrigger_mapping_active = 0;
      } else
      if (psp_kbd_mapping[psp_b] == KBD_RTRIGGER_MAPPING) {
        kbd_rtrigger_mapping_active = button_pressed;
        kbd_ltrigger_mapping_active = 0;
      }
    }
  }
  return 0;
}

void
kbd_change_auto_fire(int auto_fire)
{
  CPC.cpc_auto_fire = auto_fire;
  if (CPC.cpc_auto_fire_pressed) {
    if (CPC.psp_active_joystick) {
      cpc_key_event(CPC_J1_FIRE1, 0);
    } else {
      cpc_key_event(CPC_J0_FIRE1, 0);
    }
    CPC.cpc_auto_fire_pressed = 0;
  }
}

static int
kbd_reset_button_status(void)
{
  int b = 0;
  /* Reset Button status */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    loc_button_press[b]   = 0;
    loc_button_release[b] = 0;
  }
  psp_init_keyboard();
  return 0;
}

int
kbd_scan_keyboard(void)
{
  gp2xCtrlData c;
  long        delta_stamp;
  int         event;
  int         b;

  event = 0;
  gp2xCtrlPeekBufferPositive( &c, 1 );

  if (CPC.cpc_auto_fire) {
    delta_stamp = c.TimeStamp - first_time_auto_stamp;
    if ((delta_stamp < 0) ||
        (delta_stamp > (KBD_MIN_AUTOFIRE_TIME / (1 + CPC.cpc_auto_fire_period)))) {
      first_time_auto_stamp = c.TimeStamp;

      if (CPC.psp_active_joystick) {
        cpc_key_event(CPC_J1_FIRE1, CPC.cpc_auto_fire_pressed);
      } else {
        cpc_key_event(CPC_J0_FIRE1, CPC.cpc_auto_fire_pressed);
      }
      CPC.cpc_auto_fire_pressed = ! CPC.cpc_auto_fire_pressed;
    }
  }

  for (b = 0; b < KBD_MAX_BUTTONS; b++)
  {
    if (c.Buttons & loc_button_mask[b]) {
      if (!(loc_button_data.Buttons & loc_button_mask[b])) {
        loc_button_press[b] = 1;
        event = 1;
      }
    } else {
      if (loc_button_data.Buttons & loc_button_mask[b]) {
        loc_button_release[b] = 1;
        loc_button_press[b] = 0;
        event = 1;
      }
    }
  }
  memcpy(&loc_button_data,&c,sizeof(gp2xCtrlData));

  return event;
}

void
psp_kbd_wait_start(void)
{
  while (1)
  {
    gp2xCtrlData c;
    gp2xCtrlReadBufferPositive(&c, 1);
    if (c.Buttons & GP2X_CTRL_START) break;
  }
  psp_kbd_wait_no_button();
}

void
psp_init_keyboard(void)
{
  cpc_kbd_reset();
  kbd_ltrigger_mapping_active = 0;
  kbd_rtrigger_mapping_active = 0;
}

void
psp_kbd_wait_no_button(void)
{
  gp2xCtrlData c;

  do {
   gp2xCtrlPeekBufferPositive(&c, 1);
  } while (c.Buttons != 0);
}

void
psp_kbd_wait_button(void)
{
  gp2xCtrlData c;

  do {
   gp2xCtrlReadBufferPositive(&c, 1);
  } while (c.Buttons == 0);
}

int
psp_update_keys(void)
{
  int         b;

  static char first_time = 1;
  static int release_pending = 0;

  if (first_time) {

    gp2xCtrlData c;
    gp2xCtrlPeekBufferPositive(&c, 1);

    if (first_time_stamp == -1) first_time_stamp = c.TimeStamp;

    first_time      = 0;
    release_pending = 0;

    for (b = 0; b < KBD_MAX_BUTTONS; b++) {
      loc_button_release[b] = 0;
      loc_button_press[b] = 0;
    }
    gp2xCtrlPeekBufferPositive(&loc_button_data, 1);

    psp_main_menu();
    psp_init_keyboard();

    return 0;
  }

  if (command_mode) {
    return psp_kbd_enter_command();
  }

  if (danzeff_mode) {
    return psp_kbd_enter_danzeff();
  }

  if (release_pending)
  {
    release_pending = 0;
    for (b = 0; b < KBD_MAX_BUTTONS; b++) {
      if (loc_button_release[b]) {
        loc_button_release[b] = 0;
        cpc_decode_key(b, 0);
      }
    }
  }

  kbd_scan_keyboard();

  /* check press event */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    if (loc_button_press[b]) {
      loc_button_press[b] = 0;
      release_pending     = 0;
      cpc_decode_key(b, 1);
    }
  }
  /* check release event */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    if (loc_button_release[b]) {
      release_pending = 1;
      break;
    }
  }

  return 0;
}

extern SDL_Rect exit_rect;
static byte maemo_key_matrix[16];
static byte maemo_keys[0x1000];
static byte maemo_joy_keys[2];
static byte maemo_joy_ts[2];
static byte maemo_joy_acc[2];
static byte maemo_joy_map[2][6] = {
		{	0x95,	0x94,	0x90,	0x91,	0x92,	0x93	},
		{	0x65,	0x64,	0x60,	0x61,	0x62,	0x63	}
};


inline static void set_key( SDL_keysym code ) {
	int keyval;
	gdk_keymap_translate_keyboard_state( NULL, code.scancode, code.mod & KMOD_MODE ? GDK_MODIFIER_MASK : 0, 0, &keyval, NULL, NULL, NULL );
	unsigned char key = maemo_keys[ keyval & 0xFFF ];
	if( key == 0x97 && code.mod & KMOD_MODE ) key = 0x82; //Generate ESC
	else if( key == 0x22 && code .mod & KMOD_MODE ) key = 0x84; //Generate TAB
	maemo_key_matrix[ key >> 4 ] &= ~( 1 << ( key & 0x07 ) );
	if( key != 0xFF && key & 0x08 )
		maemo_key_matrix[ 0x02 ] &= 0xDF;
}

inline static void unset_key( SDL_keysym code ) {
	int keyval;
	gdk_keymap_translate_keyboard_state( NULL, code.scancode, code.mod & KMOD_MODE ? GDK_MODIFIER_MASK : 0, 0, &keyval, NULL, NULL, NULL );
	unsigned char key = maemo_keys[ keyval & 0xFFF ];
	maemo_key_matrix[ key >> 4 ] |= 1 << ( key & 0x07 );
	if( key == 0x97 ) maemo_key_matrix[ 0x08 ] |= 0x04; //Release ESC
	else if( key == 0x22 ) maemo_key_matrix[ 0x08 ] |= 0x10; //Release TAB
	if( key != 0xFF && key & 0x08 )
		maemo_key_matrix[ 0x02 ] |= 0x20;
}

inline static void set_joy( int code ) {
	int i;
	for( i = 0; i < 2; i++ )
		if( CPC.joy[i].key_enable ) maemo_joy_keys[i] |= CPC.joy[i].key_map[code];
}

inline static void unset_joy( int code ) {
	int i;
	for( i = 0; i < 2; i++ )
		if( CPC.joy[i].key_enable ) maemo_joy_keys[i] &= ~CPC.joy[i].key_map[code];
}

inline static void update_joy_ts( int x, int y, int mode ) {
	memset( maemo_joy_ts, 0, sizeof(maemo_joy_ts) );
	if( !mode ) return;
	unsigned char ts = 0;
	if( x >= 0 && x <= 300 && y >= 0 && y <= 480 ) {
		if( x >= 0 && x <= 100 ) ts |= CPCM_JOY_LEFT;
		if( x >= 200 && x <= 300 ) ts |= CPCM_JOY_RIGHT;
		if( y >= 0 && y <= 160 ) ts |= CPCM_JOY_UP;
		if( y >= 320 && y <= 480 ) ts |= CPCM_JOY_DOWN;
	} else if( x >= 700 && x <= 800 && y >= 0 && y <= 480) {
		if( y <= exit_rect.h && x >= exit_rect.x ) CPC.cpc_exit = 1;
		else if( y >= 160 && y <= 320 ) ts |= CPCM_JOY_F1;
		else ts |= CPCM_JOY_F2;
	}
	if( CPC.joy[0].ts_enable ) maemo_joy_ts[0] = ts;
	if( CPC.joy[1].ts_enable ) maemo_joy_ts[1] = ts;
}

inline static void update_joy_accel() {
	if( ! (CPC.joy[0].acc_enable || CPC.joy[1].acc_enable) ) return;
	memset( maemo_joy_acc, 0, sizeof(maemo_joy_acc) );
	int x, y, z, i;
	FILE* f = fopen( "/sys/class/i2c-adapter/i2c-3/3-001d/coord", "r" );
	if( !f ) return;
	fscanf( f, "%d %d %d", &x, &y, &z );
	fclose( f );
	for( i = 0; i < 2; i++ ) {
		if( CPC.joy[i].acc_enable ) {
			int sens = CPC.joy[i].acc_sens;
			if( x > sens ) maemo_joy_acc[i] |= CPCM_JOY_LEFT;
			else if( x < -sens ) maemo_joy_acc[i] |= CPCM_JOY_RIGHT;
			if( y > sens ) maemo_joy_acc[i] |= CPCM_JOY_UP;
			else if( y < -sens ) maemo_joy_acc[i] |= CPCM_JOY_DOWN;
		}
	}
}

inline static void update_input() {
	int i;
	memcpy( keyboard_matrix, maemo_key_matrix, sizeof(keyboard_matrix) );
	for( i = 0; i < 2; i++ ) {
		byte joy = maemo_joy_keys[i] | maemo_joy_ts[i] | maemo_joy_acc[i];
		int j;
		for( j = 0; j < 6; j++ ) {
			if( joy & ( 1 << j ) ) {
				byte key = maemo_joy_map[i][j];
				keyboard_matrix[ key >> 4 ] &= ~( 1 << ( key & 0x0F ) );
			}
		}
	}
}

void maemo_init_keys() {
	gdk_init(NULL,NULL);
	memset( maemo_key_matrix, -1, sizeof(maemo_keys) );
	memset( maemo_keys, -1, sizeof(maemo_keys) );
	//memset( maemo_keys_mod, -1, sizeof(maemo_keys_mod) );
	memset( maemo_joy_keys, 0, sizeof(maemo_joy_keys) );
	memset( maemo_joy_ts, 0, sizeof(maemo_joy_ts) );
	memset( maemo_joy_acc, 0, sizeof(maemo_joy_acc) );

#define XK( a, b ) { maemo_keys[ a & 0xFFF ] = b; }
#define XKA( a, b, c ) { maemo_keys[ a & 0xFFF ] = b; } // maemo_keys_mod[a] = c; }
#define XJ( a, b )
#include "keyboard.h"
#undef XJ
#undef XKA
#undef XK
}

int maemo_update_keys() {
#define set( key )
	SDL_Event event;
	int keyval;
	while( SDL_PollEvent(&event) ) {
		switch( event.type ) {
		case SDL_KEYDOWN:
			if( event.key.keysym.sym == SDLK_BACKSPACE && event.key.keysym.mod & KMOD_CTRL ) {
				CPC.cpc_exit = 1;
				return 0;
			}
			set_key( event.key.keysym );
			set_joy( event.key.keysym.scancode );
			break;
		case SDL_KEYUP:
			gdk_keymap_translate_keyboard_state( NULL, event.key.keysym.scancode, event.key.keysym.mod & KMOD_MODE ? GDK_MODIFIER_MASK : 0, 0, &keyval, NULL, NULL, NULL );
			unset_key( event.key.keysym );
			unset_joy( event.key.keysym.scancode );
			break;
		case SDL_MOUSEBUTTONDOWN:
			update_joy_ts( event.button.x, event.button.y, 1 );
			break;
		case SDL_MOUSEBUTTONUP:
			update_joy_ts( event.button.x, event.button.y, 0 );
			break;
		case SDL_MOUSEMOTION:
			update_joy_ts( event.button.x, event.button.y, 1 );
			break;
		case SDL_QUIT:
			CPC.cpc_exit = 1;
			return 0;
		}
	}
	update_joy_accel();
	update_input();
	psp_kbd_enter_command();
	return 0;
}
