/* CPCm - Amstrad CPC Emulator
   (c) Copyright 2010 javicq

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <glib.h>
#include <libosso.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <signal.h>

#include "cap32.h"
#include "menu/gconf.h"

static char osso_initialized;
static osso_context_t* osso = NULL;
static GMainContext* g_context = NULL;
static GMainLoop* g_loop = NULL;

static int cli_init( int argc, char **argv ) {
	int x, unrecognized = 0, use_gui = 0;

	for(x = 1; x < argc; x++)
	{
		if(argv[x][0] == '-')
		{
			if(strcasecmp(argv[x], "-gui") == 0) {
				use_gui = 1;
			}
			else {
				unrecognized = 1;
				break;
			}
		} else {
			/* External Frontend: ROM Name */
			cap32_disk_load( argv[x], 'A', 0 );
			break;
		}
	}

	if (unrecognized) {
		printf("\n\n\nCPCm for Maemo v" G_STRINGIFY(GAME_VERSION) " (c) javicq, 2010\n");
		printf("usage: %s [options] <disk1>\n", argv[0]);
		//printf("\nAvailable options:\n");
		//printf("\t-o <config_file>\tLoad options from specified config_file. Default is picodrive.ini\n\n");
	} else {
		//cli_config( conf_file );
	}
	return use_gui;
}

static gint dbus_callback( const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval ) {
	retval->type = DBUS_TYPE_BOOLEAN;

	if( !osso_initialized ) {
		// Only if we haven't received the startup command yet.
		printf("Osso: Startup method is: %s\n", method);

		if (strcmp(method, "game_run") == 0) {
			osso_initialized = 1;
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_continue") == 0) {
			CPC.cpc_continue = 1;
			osso_initialized = 1;
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_restart") == 0) {
			osso_initialized = 1;
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_autorun") == 0) {
			osso_initialized = 1;
			CPC.autorun = 1;
			retval->value.b = TRUE;
		} else if (strcmp(method, "game_close") == 0) {
			retval->value.b = TRUE;
		} else {
			osso_initialized = 1;
			retval->value.b = FALSE;
		}
	} else {
		if (strcmp(method, "game_close") == 0) {
			CPC.cpc_exit = 1;
			retval->value.b = TRUE;
		} else {
			retval->value.b = FALSE;
		}
	}

	return OSSO_OK;
}

static void hw_callback( osso_hw_state_t *state, gpointer data )
{
	if( state->shutdown_ind ) {
		// Shutting down. Try to quit gracefully.
		CPC.cpc_exit = 1;
	}
	if( state->system_inactivity_ind && CPC.cpc_screensaver ) {
		// Screen went off, and power saving is active.
		CPC.cpc_exit = 1;
	}
}

static void disp_callback( osso_display_state_t state, gpointer data ) {
	if( state != OSSO_DISPLAY_ON ) {
		if( CPC.cpc_screensaver ) {
			CPC.cpc_exit = 1;
		} else {
			osso_display_state_on( osso );
		}
	}
}

static void osso_config() {
	GConfClient* gcc = gconf_client_get_default();
	char* rom = gconf_client_get_string( gcc, G_CPCM_DRIVE_A, NULL );
	strcpy( CPC.cpc_drivea, rom ? rom : "" );
	rom = gconf_client_get_string( gcc, G_CPCM_DRIVE_B, NULL );
	strcpy( CPC.cpc_driveb, rom ? rom : "" );
	CPC.cpc_screensaver = gconf_client_get_bool( gcc, G_CPCM_SCREENSAVER, NULL );
	CPC.cpc_view_fps = gconf_client_get_bool( gcc, G_CPCM_DISP_FRAMERATE, NULL );
	CPC.model = gconf_client_get_int( gcc, G_CPCM_MACHINE, NULL );
	CPC.cpc_green = gconf_client_get_bool( gcc, G_CPCM_MONOCHROME, NULL );
	CPC.snd_enabled = gconf_client_get_bool( gcc, G_CPCM_SOUND, NULL );
	CPC.ts_disp_buttons = gconf_client_get_bool( gcc, G_CPCM_JOY1_TS_SHOW, NULL ) || gconf_client_get_bool( gcc, G_CPCM_JOY2_TS_SHOW, NULL );
	int sensitivities[] = { 300, 150, 75 }, i;
	for( i = 0; i < 2; i++ ) {
		CPC.joy[i].key_enable = gconf_client_get_bool( gcc, i ? G_CPCM_JOY2_KEY_ENABLE : G_CPCM_JOY1_KEY_ENABLE, NULL );
		CPC.joy[i].ts_enable = gconf_client_get_bool( gcc, i ? G_CPCM_JOY2_TS_ENABLE : G_CPCM_JOY1_TS_ENABLE, NULL );
		CPC.joy[i].acc_enable = gconf_client_get_bool( gcc, i ? G_CPCM_JOY2_ACC_ENABLE : G_CPCM_JOY1_ACC_ENABLE, NULL );
		CPC.joy[i].acc_sens = sensitivities[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_ACC_SENS : G_CPCM_JOY1_ACC_SENS, NULL ) ];
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "f1" : G_CPCM_JOY1_KEY "f1", NULL ) ] = CPCM_JOY_F1;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "f2" : G_CPCM_JOY1_KEY "f2", NULL ) ] = CPCM_JOY_F2;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "dn" : G_CPCM_JOY1_KEY "dn", NULL ) ] = CPCM_JOY_UP;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "ds" : G_CPCM_JOY1_KEY "ds", NULL ) ] = CPCM_JOY_DOWN;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "de" : G_CPCM_JOY1_KEY "de", NULL ) ] = CPCM_JOY_RIGHT;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "dw" : G_CPCM_JOY1_KEY "dw", NULL ) ] = CPCM_JOY_LEFT;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "dne" : G_CPCM_JOY1_KEY "dne", NULL ) ] = CPCM_JOY_UP | CPCM_JOY_RIGHT;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "dsw" : G_CPCM_JOY1_KEY "dsw", NULL ) ] = CPCM_JOY_DOWN | CPCM_JOY_LEFT;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "dse" : G_CPCM_JOY1_KEY "dse", NULL ) ] = CPCM_JOY_DOWN | CPCM_JOY_RIGHT;
		CPC.joy[i].key_map[ gconf_client_get_int( gcc, i ? G_CPCM_JOY2_KEY "dnw" : G_CPCM_JOY1_KEY "dnw", NULL ) ] = CPCM_JOY_UP | CPCM_JOY_LEFT;
	}
/*
	if( gconf_client_get_bool( gcc, kGConfSound, NULL ) )
		currentConfig.EmuOpt |= 0x04;
	else
		currentConfig.EmuOpt &= ~0x04;

	if( gconf_client_get_bool( gcc, kGConfDisplayFramerate, NULL ) )
		currentConfig.EmuOpt |= 0x02;
	else
		currentConfig.EmuOpt &= ~0x02;

	currentConfig.Frameskip = gconf_client_get_int( gcc, kGConfFrameskip, NULL ) - 1;
	currentConfig.scaling = gconf_client_get_int( gcc, kGConfScaler, NULL );

	currentConfig.screensaver = gconf_client_get_bool( gcc, kGConfSaver, NULL );
	currentConfig.displayTS = gconf_client_get_bool( gcc, GPLAYER1_K kGConfPlayerTouchscreenShow, NULL ) || gconf_client_get_bool( gcc, GPLAYER2_K kGConfPlayerTouchscreenShow, NULL );

	unsigned short* key_map = currentConfig.player[0].keymap;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "a", NULL) & 0xFF ] = MDB_A;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "b", NULL) & 0xFF ] = MDB_B;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "c", NULL) & 0xFF ] = MDB_C;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "x", NULL) & 0xFF ] = MDB_X;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "y", NULL) & 0xFF ] = MDB_Y;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "z", NULL) & 0xFF ] = MDB_Z;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dn", NULL) & 0xFF ] = MDB_UP;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "ds", NULL) & 0xFF ] = MDB_DOWN;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "de", NULL) & 0xFF ] = MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dw", NULL) & 0xFF ] = MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "start", NULL) & 0xFF ] = MDB_START;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dne", NULL) & 0xFF ] = MDB_UP | MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dnw", NULL) & 0xFF ] = MDB_UP | MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dse", NULL) & 0xFF ] = MDB_DOWN | MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "dsw", NULL) & 0xFF ] = MDB_DOWN | MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER1_KEY "quit", NULL) & 0xFF ] = CMD_QUIT;
	int i;
	for( i = 0; i < 4; i++ ) {
		char key[256];
		sprintf( key, "%sqs%d", GPLAYER1_KEY, i+1 );
		key_map[ gconf_client_get_int( gcc, key, NULL) & 0xFF ] = CMD_QUICKSAVE + i;
		sprintf( key, "%sql%d", GPLAYER1_KEY, i+1 );
		key_map[ gconf_client_get_int( gcc, key, NULL) & 0xFF ] = CMD_QUICKLOAD + i;
	}

	int accelSensitivities[] = { 300, 150, 75 };
	currentConfig.player[0].inputFlags =
		( gconf_client_get_bool( gcc, GPLAYER1_K kGConfPlayerAccelerometerEnable, NULL ) ? _P_ENABLE_ACC : 0 ) |
		( gconf_client_get_bool( gcc, GPLAYER1_K kGConfPlayerTouchscreenEnable, NULL ) ? _P_ENABLE_TS : 0 ) |
		( gconf_client_get_bool( gcc, GPLAYER1_K kGConfPlayerKeyboardEnable, NULL ) ? _P_ENABLE_KEY : 0 );
	currentConfig.player[0].accelSensitivity = accelSensitivities[ gconf_client_get_int( gcc, GPLAYER1_K kGConfPlayerAccelerometerSensitivity, NULL ) ];

	key_map = currentConfig.player[1].keymap;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "a", NULL) & 0xFF ] = MDB_A;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "b", NULL) & 0xFF ] = MDB_B;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "c", NULL) & 0xFF ] = MDB_C;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "x", NULL) & 0xFF ] = MDB_X;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "y", NULL) & 0xFF ] = MDB_Y;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "z", NULL) & 0xFF ] = MDB_Z;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "dn", NULL) & 0xFF ] = MDB_UP;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "ds", NULL) & 0xFF ] = MDB_DOWN;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "de", NULL) & 0xFF ] = MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "dw", NULL) & 0xFF ] = MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "start", NULL) & 0xFF ] = MDB_START;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "dne", NULL) & 0xFF ] = MDB_UP | MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "dnw", NULL) & 0xFF ] = MDB_UP | MDB_LEFT;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "dse", NULL) & 0xFF ] = MDB_DOWN | MDB_RIGHT;
	key_map[ gconf_client_get_int( gcc, GPLAYER2_KEY "dsw", NULL) & 0xFF ] = MDB_DOWN | MDB_LEFT;
	currentConfig.player[1].inputFlags =
		( gconf_client_get_bool( gcc, GPLAYER2_K kGConfPlayerAccelerometerEnable, NULL ) ? _P_ENABLE_ACC : 0 ) |
		( gconf_client_get_bool( gcc, GPLAYER2_K kGConfPlayerTouchscreenEnable, NULL ) ? _P_ENABLE_TS : 0 ) |
		( gconf_client_get_bool( gcc, GPLAYER2_K kGConfPlayerKeyboardEnable, NULL ) ? _P_ENABLE_KEY : 0 );
	currentConfig.player[1].accelSensitivity = accelSensitivities[ gconf_client_get_int( gcc, GPLAYER2_K kGConfPlayerAccelerometerSensitivity, NULL ) ];*/
}

gboolean osso_events( gboolean block ) {
	static unsigned char frames;
	if( !++frames && !CPC.cpc_screensaver ) osso_display_blanking_pause( osso );
	if( g_context ) return g_main_context_iteration( g_context, block );
	return FALSE;
}

static void sig_handler( int signal ) {
	printf( "Caught signal %d, exiting...\n", signal );
	osso_deinit();
	exit(1);
}

void osso_deinit() {
	if( !osso ) return;

	// Send a goodbye message to the launcher
	osso_return_t ret;
	osso_rpc_t retval = { 0 };
	ret = osso_rpc_run(osso, "com.jcq.cpcm.startup",
				"/com/jcq/cpcm/startup", "com.jcq.cpcm.startup",
				osso_initialized ? "game_pause" : "game_close", &retval, DBUS_TYPE_INVALID);

	if (ret != OSSO_OK) {
		printf("Osso: failed to notify launcher\n");
	}
	osso_rpc_free_val(&retval);

	osso_deinitialize(osso);
	g_main_loop_unref(g_loop);
	g_main_context_unref(g_context);

	osso = NULL;
}

void osso_init( int argc, char **argv ) {
	if( !cli_init( argc, argv ) ) return;

	signal( SIGSEGV, sig_handler );
	signal( SIGBUS, sig_handler );
	signal( SIGILL, sig_handler );
	signal( SIGFPE, sig_handler );
	signal( SIGINT, sig_handler );
	signal( SIGABRT, sig_handler );
	signal( SIGHUP, sig_handler );
	signal( SIGTERM, sig_handler );

	g_type_init();
	g_set_prgname( "cpcm" );
	g_set_application_name( "CPCm" );
	g_context = g_main_context_default();
	g_loop = g_main_loop_new( g_context, FALSE );

	osso = osso_initialize( "com.jcq.cpcm", "1.0", FALSE, NULL );
	if( !osso ) {
		fprintf( stderr, "Error initializing osso\n" );
		exit(1);
	}
	osso_rpc_set_default_cb_f( osso, dbus_callback, NULL );

	osso_hw_state_t hwStateFlags = { FALSE };
	hwStateFlags.shutdown_ind = TRUE;
	hwStateFlags.system_inactivity_ind = TRUE;
	osso_hw_set_event_cb( osso, &hwStateFlags, hw_callback, NULL );
	osso_hw_set_display_event_cb( osso, disp_callback, NULL );

	osso_initialized = 0;
	int timeout = 3000;
	for( ; !osso_initialized && timeout > 0; timeout -= 50 ) {
		osso_events( FALSE );
		usleep( 50000 );
	}
	if( !osso_initialized ) {
		fprintf( stderr, "No osso message received\n" );
		osso_deinit();
		exit(1);
	}

	osso_config();

	printf( "libosso initialized\n" );
}

