/* countdown-home-widget.c */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "countdown-home-widget.h"
#include <hildon/hildon.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <gconf/gconf-client.h>

#define HOME_DIR g_get_home_dir()

#define _(String) dgettext("hildon-libs", String)

#define CLOCK_GCONF_PATH "/apps/clock"
#define CLOCK_GCONF_IS_24H_FORMAT CLOCK_GCONF_PATH  "/time-format"

#define COUNTDOWN_HOME_WIDGET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,\
							                         COUNTDOWN_TYPE_HOME_WIDGET,\
							                         CountdownHomeWidgetPrivate))

struct _CountdownHomeWidgetPrivate
{
    GtkWidget *homeWidget;
    GtkWidget *contents;
	GtkWidget *buttons;
	GtkWidget *upButton, *downButton;
    GtkListStore *entries;
    guint refresh;
	gchar *titleColor, *valueColor;
	gchar *titleFont, *valueFont;
	gboolean systemFont, systemBG;
	guint bgOpacity, buttonAreaOpacity;
	gchar *bgColor, *buttonAreaColor;
	guint radius;
	gchar *yearsText, *weeksText, *daysText;
	gchar *minutesText, *hoursText;
	gchar *leftText, *agoText;
	guint offset;
    guint eventID;
};

HD_DEFINE_PLUGIN_MODULE (CountdownHomeWidget, countdown_home_widget,
  HD_TYPE_HOME_PLUGIN_ITEM)

gboolean countdown_home_widget_update_content (CountdownHomeWidget *self);
static gboolean countdown_home_widget_expose_button_area (GtkWidget *widget, GdkEventExpose *event, CountdownHomeWidget *self);

gboolean check_ampm_format ()
{
	//g_warning ("check_ampm_format");
	GConfClient *client = NULL;
	gboolean value = TRUE;
	GError *error = NULL;

	client = gconf_client_get_default ();
	value = gconf_client_get_bool (client, CLOCK_GCONF_IS_24H_FORMAT, &error);
	if (error != NULL) {
		g_warning
			("Error trying to get gconf variable %s, using 24h format by default",
			CLOCK_GCONF_IS_24H_FORMAT);
		g_error_free (error);
	}

	return !value;
}

void countdown_home_widget_write_settings (CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_write_settings");
	GKeyFile *keyFile;
	gchar *fileData;
	FILE *iniFile;
	gsize size;
	GtkTreeIter iter;
	gboolean exists;
	gchar *filename;
	gint i=0;
	
	keyFile = g_key_file_new();
	exists = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (self->priv->entries), &iter);
	while (exists) {
		gchar *title, *date, *time;
		gboolean showWhich[5];
		gboolean removePassed;
		gchar entry[5];
		snprintf(entry, 5, "%d", i++);
		gtk_tree_model_get(GTK_TREE_MODEL (self->priv->entries),&iter,0,&title,1,&date,2,&time,4,&showWhich[0],5,&showWhich[1],6,&showWhich[2],7,&showWhich[3],8,&showWhich[4],9,&removePassed,-1);
		g_key_file_set_string(keyFile,entry,"title",title);
		g_key_file_set_string(keyFile,entry,"date",date);
		g_key_file_set_string(keyFile,entry,"time",time);
		g_key_file_set_boolean_list(keyFile,entry,"show",showWhich,5);
		g_key_file_set_boolean (keyFile, entry, "removePassed", removePassed);
		exists=gtk_tree_model_iter_next(GTK_TREE_MODEL (self->priv->entries),&iter);
		g_free(title);
		g_free(date);
		g_free(time);
	}
	
	g_key_file_set_integer (keyFile, "config", "refresh", self->priv->refresh);
	g_key_file_set_boolean (keyFile, "config", "systemBG", self->priv->systemBG);
	g_key_file_set_string (keyFile, "config", "bgColor", self->priv->bgColor);
	g_key_file_set_integer (keyFile, "config", "bgOpacity", self->priv->bgOpacity);
	g_key_file_set_integer (keyFile, "config", "radius", self->priv->radius);
	g_key_file_set_string (keyFile, "config", "buttonAreaColor", self->priv->buttonAreaColor);
	g_key_file_set_integer (keyFile, "config", "buttonAreaOpacity", self->priv->buttonAreaOpacity);
	g_key_file_set_boolean (keyFile, "config", "systemFont", self->priv->systemFont);
	g_key_file_set_string (keyFile, "config", "titleColor", self->priv->titleColor);
	g_key_file_set_string (keyFile, "config", "titleFont", self->priv->titleFont);
	g_key_file_set_string (keyFile, "config", "valueColor", self->priv->valueColor);
	g_key_file_set_string (keyFile, "config", "valueFont", self->priv->valueFont);
	g_key_file_set_string (keyFile, "labels", "yearsText", self->priv->yearsText);
	g_key_file_set_string (keyFile, "labels", "weeksText", self->priv->weeksText);
	g_key_file_set_string (keyFile, "labels", "daysText", self->priv->daysText);
	g_key_file_set_string (keyFile, "labels", "hoursText", self->priv->hoursText);
	g_key_file_set_string (keyFile, "labels", "minutesText", self->priv->minutesText);
	g_key_file_set_string (keyFile, "labels", "leftText", self->priv->leftText);
	g_key_file_set_string (keyFile, "labels", "agoText", self->priv->agoText);
	
	filename = g_strconcat (HOME_DIR, "/.countdown_home.ini", NULL);
	fileData = g_key_file_to_data (keyFile, &size, NULL);
	iniFile = fopen (filename, "w");
	fputs (fileData, iniFile);
	fclose (iniFile);
	g_key_file_free (keyFile);
	g_free (fileData);
	g_free (filename);
}

void countdown_home_widget_select_date (GtkButton *button, GtkWidget *dialog)
{	
	//g_warning ("countdown_home_widget_select_date");
	guint year, month, day;
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	
	gtk_dialog_run (GTK_DIALOG (dialog));
	hildon_date_selector_get_date (HILDON_DATE_SELECTOR (selector), &year, &month, &day);
	GDate *date = g_date_new_dmy (day, month+1, year);
	gchar dateString[100];
	g_date_strftime (dateString, 99, _("wdgt_va_date_long"), date);
	gtk_button_set_label (button, dateString);
	g_date_free (date);

	gtk_widget_hide (dialog);
}

void countdown_home_widget_select_time (GtkButton *button, GtkWidget *dialog)
{
	//g_warning ("countdown_home_widget_select_time");
	guint hour, minute;
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	
	gtk_dialog_run (GTK_DIALOG (dialog));
	hildon_time_selector_get_time (HILDON_TIME_SELECTOR (selector), &hour, &minute);
	struct tm tm = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
	tm.tm_min = minute;
	tm.tm_hour = hour;
	gchar timeString[255];
	
	if (check_ampm_format()) {
		if (hour >= 12) {
			strftime (timeString, 255, _("wdgt_va_12h_time_pm"), &tm);
		} else {
			strftime (timeString, 255, _("wdgt_va_12h_time_am"), &tm);
		}
	} else {
		strftime (timeString, 255, _("wdgt_va_24h_time"), &tm);
	}
	gtk_button_set_label (button, timeString);
	
	gtk_widget_hide (dialog);
}

void countdown_home_widget_select_position (GtkButton *button, GtkWidget *dialog)
{
	//g_warning ("countdown_home_widget_select_position");
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	
	gtk_dialog_run (GTK_DIALOG (dialog));
	gint current = hildon_touch_selector_get_active (selector, 0);
	gchar *currentText = g_strdup_printf ("%d", current+1);
	gtk_button_set_label (button, currentText);
	g_free (currentText);
	gtk_widget_hide (dialog);
}

void countdown_home_widget_add_event(GtkWidget *button, GdkEventButton *event, CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_add_event");
	GDate *date = g_date_new();
    GtkWidget *dialog, *content_area;
    GtkWidget *titleEntry, *dateButton, *timeButton;
    GtkWidget *titleCaption, *dateCaption, *timeCaption;
	GtkWidget *datePicker, *timePicker;
	GtkWidget *dateSelector, *timeSelector;
    GtkWidget *toggleYears, *toggleWeeks, *toggleDays, *toggleHours, *toggleMinutes;
    GtkWidget *displayCaption;
	GtkWidget *removePassedButton, *removePassedCaption;
    GtkSizeGroup *group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

    dialog = gtk_dialog_new_with_buttons ("Add Countdown Event",
        NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
        "Save", GTK_RESPONSE_ACCEPT, NULL);
    content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
    
    titleEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    hildon_entry_set_placeholder (HILDON_ENTRY (titleEntry), "What do you want to call this?");
    titleCaption = hildon_caption_new (group, "Title", titleEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (titleCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), titleCaption);

	datePicker = hildon_picker_dialog_new (GTK_WINDOW (dialog));
	gtk_window_set_title (GTK_WINDOW (datePicker), "Select date");
	dateSelector = hildon_date_selector_new ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (datePicker), HILDON_TOUCH_SELECTOR (dateSelector));
	g_date_set_time_t (date, time(NULL));
	gchar dateString[100];
	g_date_strftime (dateString, 99, _("wdgt_va_date_long"), date);
	
	dateButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_date_selector_select_current_date (HILDON_DATE_SELECTOR (dateSelector), g_date_get_year (date), g_date_get_month (date) - 1, g_date_get_day (date));
	gtk_button_set_label (GTK_BUTTON (dateButton), dateString);
	g_signal_connect (G_OBJECT (dateButton), "clicked", G_CALLBACK (countdown_home_widget_select_date), datePicker);
	
    dateCaption = hildon_caption_new (group, "Date", dateButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (dateCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), dateCaption);

	timePicker = hildon_picker_dialog_new (GTK_WINDOW (dialog));
	gtk_window_set_title (GTK_WINDOW (timePicker), "Select time");
	timeSelector = hildon_time_selector_new ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (timePicker), HILDON_TOUCH_SELECTOR (timeSelector));
	time_t now = time (NULL);
	struct tm *timeinfo;
	timeinfo = localtime(&now);
	gchar timeString[255];
	if (check_ampm_format()) {
		if (timeinfo->tm_hour >= 12) {
			strftime (timeString, 255, _("wdgt_va_12h_time_pm"), timeinfo);
		} else {
			strftime (timeString, 255, _("wdgt_va_12h_time_am"), timeinfo);
		}
	} else {
		strftime (timeString, 255, _("wdgt_va_24h_time"), timeinfo);
	}
	
	timeButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_time_selector_set_time (HILDON_TIME_SELECTOR (timeSelector), timeinfo->tm_hour, timeinfo->tm_min);
	gtk_button_set_label (GTK_BUTTON (timeButton), timeString);

	g_signal_connect (G_OBJECT (timeButton), "clicked", G_CALLBACK (countdown_home_widget_select_time), timePicker);
	
    timeCaption = hildon_caption_new (group, "Time", timeButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (timeCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), timeCaption);

    GtkHBox *hbox = GTK_HBOX (gtk_hbox_new (FALSE, 0));
    toggleYears = gtk_toggle_button_new_with_label ("Year");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleYears), TRUE);
    toggleWeeks = gtk_toggle_button_new_with_label ("Week");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleWeeks), TRUE);
    toggleDays = gtk_toggle_button_new_with_label ("Day");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleDays), TRUE);
    toggleHours = gtk_toggle_button_new_with_label ("Hour");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleHours), TRUE);
    toggleMinutes = gtk_toggle_button_new_with_label ("Minute");
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleMinutes), TRUE);
    gtk_container_add (GTK_CONTAINER (hbox), toggleYears);
    gtk_container_add (GTK_CONTAINER (hbox), toggleWeeks);
    gtk_container_add (GTK_CONTAINER (hbox), toggleDays);
    gtk_container_add (GTK_CONTAINER (hbox), toggleHours);
    gtk_container_add (GTK_CONTAINER (hbox), toggleMinutes);
    displayCaption = hildon_caption_new (group, "Display", GTK_WIDGET (hbox), NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (displayCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), displayCaption);
	g_date_free (date);
	
	removePassedButton = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (removePassedButton), "Delete after passed");
	removePassedCaption = hildon_caption_new (group, "", removePassedButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (removePassedCaption), "");
	gtk_container_add (GTK_CONTAINER (content_area), removePassedCaption);
    
    gtk_widget_show_all (dialog);

    int response=gtk_dialog_run (GTK_DIALOG (dialog));
    if (response == GTK_RESPONSE_ACCEPT) {
        const gchar *title = hildon_entry_get_text (HILDON_ENTRY (titleEntry));
        guint year, month, day;
        guint hours, minutes;
        gboolean tYears, tWeeks, tDays, tHours, tMinutes;
		gboolean removePassed;
        gchar *printDate, *printTime, *compTime;
		GtkTreeIter iter;

        hildon_date_selector_get_date (HILDON_DATE_SELECTOR (dateSelector), &year, &month, &day);
        hildon_time_selector_get_time (HILDON_TIME_SELECTOR (timeSelector), &hours, &minutes);

        printDate = g_strdup_printf ("%02d/%02d/%02d",year,month+1,day);
        printTime = g_strdup_printf ("%02d:%02d", hours, minutes);
        compTime = g_strdup_printf ("%d", hours*(60*60) + minutes*60);

        tYears = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleYears));
        tWeeks = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleWeeks));
        tDays = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleDays));
        tMinutes = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleMinutes));
        tHours = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleHours));
		removePassed = hildon_check_button_get_active (HILDON_CHECK_BUTTON (removePassedButton));

		gtk_list_store_append (GTK_LIST_STORE (self->priv->entries), &iter);
		gtk_list_store_set (GTK_LIST_STORE (self->priv->entries), &iter, 0, title, 1, printDate, 2, compTime, 3, printTime, -1);
		gtk_list_store_set (GTK_LIST_STORE (self->priv->entries), &iter, 4, tYears, 5, tWeeks, 6, tDays, 7, tHours, 8, tMinutes, 9, removePassed, -1);

        g_free (printDate);
        g_free (printTime);
        g_free (compTime);
		
		countdown_home_widget_write_settings (self);
		countdown_home_widget_update_content (self);
    }
    gtk_widget_destroy (dialog);
}

void countdown_home_widget_edit_event (GtkWidget *button, GdkEventButton *event, CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_edit_event");
	gint index = g_ascii_strtoll (gtk_widget_get_name (button), NULL, 10);
	GtkWidget *dialog, *content_area;
    GtkWidget *titleEntry, *dateButton, *timeButton, *positionButton;
    GtkWidget *titleCaption, *dateCaption, *timeCaption, *positionCaption;
	GtkWidget *datePicker, *timePicker, *positionPicker;
	GtkWidget *dateSelector, *timeSelector, *positionSelector;
    GtkWidget *toggleYears, *toggleWeeks, *toggleDays, *toggleHours, *toggleMinutes;
    GtkWidget *displayCaption;
	GtkWidget *removePassedButton, *removePassedCaption;
	GtkTreeIter iter;
    GtkSizeGroup *group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

    dialog = gtk_dialog_new_with_buttons ("Edit Countdown Event",
        NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
        "Save", GTK_RESPONSE_ACCEPT, "Delete", GTK_RESPONSE_NO, NULL);
	GtkWidget *editArea = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
	GtkWidget *scroll = hildon_pannable_area_new ();
	g_object_set (scroll, "hscrollbar-policy", GTK_POLICY_NEVER, NULL);
	content_area = gtk_vbox_new (FALSE, 0);
	hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (scroll), content_area);
	gtk_container_add (GTK_CONTAINER (editArea), scroll);
	gtk_window_set_default_size (GTK_WINDOW (dialog), -1, 350);
    
    titleEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    hildon_entry_set_placeholder (HILDON_ENTRY (titleEntry), "What do you want to call this?");
    titleCaption = hildon_caption_new (group, "Title", titleEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (titleCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), titleCaption);

	datePicker = hildon_picker_dialog_new (GTK_WINDOW (dialog));
	gtk_window_set_title (GTK_WINDOW (datePicker), "Select date");
	dateSelector = hildon_date_selector_new ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (datePicker), HILDON_TOUCH_SELECTOR (dateSelector));
	
	dateButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (dateButton), "clicked", G_CALLBACK (countdown_home_widget_select_date), datePicker);
	
    dateCaption = hildon_caption_new (group, "Date", dateButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (dateCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), dateCaption);

    timePicker = hildon_picker_dialog_new (GTK_WINDOW (dialog));
	gtk_window_set_title (GTK_WINDOW (timePicker), "Select time");
	timeSelector = hildon_time_selector_new ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (timePicker), HILDON_TOUCH_SELECTOR (timeSelector));

	timeButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (timeButton), "clicked", G_CALLBACK (countdown_home_widget_select_time), timePicker);
	
    timeCaption = hildon_caption_new (group, "Time", timeButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (timeCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), timeCaption);

    GtkHBox *hbox = GTK_HBOX (gtk_hbox_new (FALSE, 0));
    toggleYears = gtk_toggle_button_new_with_label ("Year");
    toggleWeeks = gtk_toggle_button_new_with_label ("Week");
    toggleDays = gtk_toggle_button_new_with_label ("Day");
    toggleHours = gtk_toggle_button_new_with_label ("Hour");
    toggleMinutes = gtk_toggle_button_new_with_label ("Minute");
    gtk_container_add (GTK_CONTAINER (hbox), toggleYears);
    gtk_container_add (GTK_CONTAINER (hbox), toggleWeeks);
    gtk_container_add (GTK_CONTAINER (hbox), toggleDays);
    gtk_container_add (GTK_CONTAINER (hbox), toggleHours);
    gtk_container_add (GTK_CONTAINER (hbox), toggleMinutes);
    displayCaption = hildon_caption_new (group, "Display", GTK_WIDGET (hbox), NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (displayCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), displayCaption);

	removePassedButton = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (removePassedButton), "Delete after passed");
	removePassedCaption = hildon_caption_new (group, "", removePassedButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (removePassedCaption), "");
	gtk_container_add (GTK_CONTAINER (content_area), removePassedCaption);
	
	positionPicker = hildon_picker_dialog_new (GTK_WINDOW (dialog));
	gtk_window_set_title (GTK_WINDOW (positionPicker), "Select position");
	positionSelector = hildon_touch_selector_new_text ();
	
	gint num = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (self->priv->entries), NULL);
	gint i;
	for (i=1; i<=num; i++) {
		gchar *pos = g_strdup_printf ("%d", i);
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (positionSelector), pos);
		g_free (pos);
	}
	
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (positionPicker), HILDON_TOUCH_SELECTOR (positionSelector));
	
	positionButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (positionButton), "clicked", G_CALLBACK (countdown_home_widget_select_position), positionPicker);
	
	positionCaption = hildon_caption_new (group, "Position", positionButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (positionCaption), "");
	if (num >1) {
		gtk_container_add (GTK_CONTAINER (content_area), positionCaption);
	}
    
	if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (self->priv->entries), &iter, NULL, index)) {
		gchar *titleText, *dateText, *timeText;
		gboolean tYears, tWeeks, tDays;
		gboolean tHours, tMinutes;
		gboolean removePassed;
		gint year, month, day;
		gint hour, minute;
		GDate date;
		gint time;
		
		gtk_tree_model_get (GTK_TREE_MODEL (self->priv->entries), &iter, 0, &titleText, 1, &dateText, 2, &timeText, 4, &tYears, 5, &tWeeks, 6, &tDays, 7, &tHours, 8, &tMinutes, 9, &removePassed, -1);
		
		g_date_set_parse (&date, dateText);
		year = g_date_get_year (&date);
		month = g_date_get_month (&date) - 1;
		day = g_date_get_day (&date);
		time = g_ascii_strtoll (timeText, NULL, 10);
		hour = time/(60 *60);
		minute = time/60 - hour*60;
		
		gchar dateString[100];
		g_date_strftime (dateString, 99, _("wdgt_va_date_long"), &date);
		
		struct tm tm = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
		tm.tm_min = minute;
		tm.tm_hour = hour;
		gchar timeString[255];
		if (check_ampm_format()) {
			if (hour >= 12) {
				strftime (timeString, 255, _("wdgt_va_12h_time_pm"), &tm);
			} else {
				strftime (timeString, 255, _("wdgt_va_12h_time_am"), &tm);
			}
		} else {
			strftime (timeString, 255, _("wdgt_va_24h_time"), &tm);
		}
		
		gchar *positionText = g_strdup_printf ("%d", index+1);
		
		hildon_entry_set_text (HILDON_ENTRY (titleEntry), titleText);
		hildon_date_selector_select_current_date (HILDON_DATE_SELECTOR (dateSelector), year, month, day);
		gtk_button_set_label (GTK_BUTTON (dateButton), dateString);
		hildon_time_selector_set_time (HILDON_TIME_SELECTOR (timeSelector), hour, minute);
		gtk_button_set_label (GTK_BUTTON (timeButton), timeString);
		hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (positionSelector), 0, index);
		gtk_button_set_label (GTK_BUTTON (positionButton), positionText);
		g_free (positionText);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleYears), tYears);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleWeeks), tWeeks);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleDays), tDays);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleHours), tHours);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (toggleMinutes), tMinutes);
		hildon_check_button_set_active (HILDON_CHECK_BUTTON (removePassedButton), removePassed);
	}
	
    gtk_widget_show_all (dialog);

    int response=gtk_dialog_run (GTK_DIALOG (dialog));
    if (response == GTK_RESPONSE_ACCEPT) {
        const gchar *title = hildon_entry_get_text (HILDON_ENTRY (titleEntry));
        guint year, month, day;
        guint hours, minutes;
        gboolean tYears, tWeeks, tDays, tHours, tMinutes;
		gboolean removePassed;
        gchar *printDate, *printTime, *compTime;
		GtkTreeIter iter;
		GtkTreeIter pos;
		gint position = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (positionSelector), 0);
		gint total = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (self->priv->entries), NULL) - 1;

        hildon_date_selector_get_date (HILDON_DATE_SELECTOR (dateSelector), &year, &month, &day);
        hildon_time_selector_get_time (HILDON_TIME_SELECTOR (timeSelector), &hours, &minutes);

        printDate = g_strdup_printf ("%02d/%02d/%02d",year,month+1,day);
        printTime = g_strdup_printf ("%02d:%02d", hours, minutes);
        compTime = g_strdup_printf ("%d", hours*(60*60) + minutes*60);

        tYears = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleYears));
        tWeeks = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleWeeks));
        tDays = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleDays));
        tMinutes = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleMinutes));
        tHours = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (toggleHours));
		removePassed = hildon_check_button_get_active (HILDON_CHECK_BUTTON (removePassedButton));

		if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (self->priv->entries), &iter, NULL, index)) {
			gtk_list_store_set (GTK_LIST_STORE (self->priv->entries), &iter, 0, title, 1, printDate, 2, compTime, 3, printTime, -1);
			gtk_list_store_set (GTK_LIST_STORE (self->priv->entries), &iter, 4, tYears, 5, tWeeks, 6, tDays, 7, tHours, 8, tMinutes, 9, removePassed, -1);
			
			if (position < total) {
				gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (self->priv->entries), &pos, NULL, position);
				gtk_list_store_move_before (self->priv->entries, &iter, &pos);
			} else {
				gtk_list_store_move_before (self->priv->entries, &iter, NULL);
			}
		}

        g_free (printDate);
        g_free (printTime);
        g_free (compTime);
		
		countdown_home_widget_write_settings (self);
		countdown_home_widget_update_content (self);
    } else if (response == GTK_RESPONSE_NO) {
		if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (self->priv->entries), &iter, NULL, index)) {
			gtk_list_store_remove (self->priv->entries, &iter);
		}
		countdown_home_widget_write_settings (self);
		countdown_home_widget_update_content (self);
	}
    gtk_widget_destroy (dialog);
}

void countdown_home_widget_select_refresh (GtkButton *button, GtkWidget *dialog)
{
	//g_warning ("countdown_home_widget_select_refresh");
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_button_set_label (button, hildon_touch_selector_get_current_text (selector));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_select_bg_opacity (GtkButton *button, GtkWidget *dialog)
{
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_button_set_label (button, hildon_touch_selector_get_current_text (selector));
	gtk_widget_hide (dialog);	
}

void countdown_home_widget_select_radius (GtkButton *button, GtkWidget *dialog)
{
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_button_set_label (button, hildon_touch_selector_get_current_text (selector));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_select_buttonArea_opacity (GtkButton *button, GtkWidget *dialog)
{
	HildonTouchSelector *selector = hildon_picker_dialog_get_selector (HILDON_PICKER_DIALOG (dialog));
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_button_set_label (button, hildon_touch_selector_get_current_text (selector));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_title_settings (GtkButton *button, GtkWidget *dialog)
{
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_value_settings (GtkButton *button, GtkWidget *dialog)
{
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_bg_settings (GtkButton *button, GtkWidget *dialog)
{
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_system_settings (HildonCheckButton *button, GtkWidget *target)
{
	if (hildon_check_button_get_active (button)) {
		gtk_button_set_label (GTK_BUTTON (target), "System style");
	} else {
		gtk_button_set_label (GTK_BUTTON (target), "User style");
	}
}

void countdown_home_widget_trans_settings (GtkButton *button, GtkWidget *dialog)
{
	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_hide (dialog);
}

void countdown_home_widget_global_options (HDHomePluginItem *hitem, CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_global_options");
    GtkWidget *dialog, *content_area;
    GtkWidget *refreshButton, *fontSettingsButton;
	GtkWidget *bgSettingsButton, *transSettingsButton;
    GtkWidget *refreshCaption, *fontSettingsCaption;
	GtkWidget *bgSettingsCaption, *transSettingsCaption;
	GtkWidget *fontDialog, *bgDialog, *transDialog;
	GtkWidget *fontContentArea, *bgContentArea, *transContentArea;
    GtkSizeGroup *group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	GtkSizeGroup *fontGroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	GtkSizeGroup *bgGroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	GtkSizeGroup *transGroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	GdkColor color;
	
	fontSettingsButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	bgSettingsButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	transSettingsButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);

    dialog = gtk_dialog_new_with_buttons ("Countdown Settings",
        NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
        "Save", GTK_RESPONSE_ACCEPT, NULL);
    content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
    
	GtkWidget *picker = hildon_picker_dialog_new (GTK_WINDOW (dialog));
	gtk_window_set_title (GTK_WINDOW (picker), "Update Every");
	GtkWidget *selector = hildon_touch_selector_new_text ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (picker), HILDON_TOUCH_SELECTOR (selector));
	
	int i;
	for (i=0; i<=60; i++) {
		gchar *text = g_strdup_printf ("%d minutes(s)", i);
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), text);
		g_free (text);
	}
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, self->priv->refresh);
	
	fontDialog = gtk_dialog_new_with_buttons ("Font Settings", dialog,
		GTK_DIALOG_DESTROY_WITH_PARENT, "Done", GTK_RESPONSE_ACCEPT, NULL);
	fontContentArea = gtk_dialog_get_content_area (GTK_DIALOG (fontDialog));
	
	GtkWidget *fontSystemStyle = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (fontSystemStyle), "Use system style");
	hildon_check_button_set_active (HILDON_CHECK_BUTTON (fontSystemStyle), self->priv->systemFont);
	g_signal_connect (fontSystemStyle, "toggled", G_CALLBACK (countdown_home_widget_system_settings), fontSettingsButton);
	GtkWidget *fontSystemStyleCaption = hildon_caption_new (fontGroup, "", fontSystemStyle, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (fontSystemStyleCaption), "");
	gtk_container_add (GTK_CONTAINER (fontContentArea), fontSystemStyleCaption);
	
	GtkWidget *titleColorButton = gtk_color_button_new ();
	gdk_color_parse (self->priv->titleColor, &color);
	gtk_color_button_set_color (GTK_COLOR_BUTTON (titleColorButton), &color);
	hildon_gtk_widget_set_theme_size (titleColorButton, HILDON_SIZE_FINGER_HEIGHT);
	GtkWidget *titleColorCaption = hildon_caption_new (fontGroup, "Title color", titleColorButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (titleColorCaption), "");
	gtk_container_add (GTK_CONTAINER (fontContentArea), titleColorCaption);
	
	GtkWidget *titleFontButton = gtk_font_button_new ();
	gtk_font_button_set_font_name (GTK_FONT_BUTTON (titleFontButton), self->priv->titleFont);
	hildon_gtk_widget_set_theme_size (titleFontButton, HILDON_SIZE_FINGER_HEIGHT);
	GtkWidget *titleFontCaption = hildon_caption_new (fontGroup, "Title font", titleFontButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (titleFontCaption), "");
	gtk_container_add (GTK_CONTAINER (fontContentArea), titleFontCaption);
	
	GtkWidget *valueColorButton = gtk_color_button_new ();
	gdk_color_parse (self->priv->valueColor, &color);
	gtk_color_button_set_color (GTK_COLOR_BUTTON (valueColorButton), &color);
	hildon_gtk_widget_set_theme_size (valueColorButton, HILDON_SIZE_FINGER_HEIGHT);
	GtkWidget *valueColorCaption = hildon_caption_new (fontGroup, "Value color", valueColorButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (valueColorCaption), "");
	gtk_container_add (GTK_CONTAINER (fontContentArea), valueColorCaption);
	
	GtkWidget *valueFontButton = gtk_font_button_new ();
	gtk_font_button_set_font_name (GTK_FONT_BUTTON (valueFontButton), self->priv->valueFont);
	hildon_gtk_widget_set_theme_size (valueFontButton, HILDON_SIZE_FINGER_HEIGHT);
	GtkWidget *valueFontCaption = hildon_caption_new (fontGroup, "Value font", valueFontButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (valueFontCaption), "");
	gtk_container_add (GTK_CONTAINER (fontContentArea), valueFontCaption);
	
	bgDialog = gtk_dialog_new_with_buttons ("Background Settings", dialog,
		GTK_DIALOG_DESTROY_WITH_PARENT, "Done", GTK_RESPONSE_ACCEPT, NULL);
	bgContentArea = gtk_dialog_get_content_area (GTK_DIALOG (bgDialog));
	
	GtkWidget *bgSystemStyle = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (bgSystemStyle), "Use system style");
	hildon_check_button_set_active (HILDON_CHECK_BUTTON (bgSystemStyle), self->priv->systemBG);
	g_signal_connect (bgSystemStyle, "toggled", G_CALLBACK (countdown_home_widget_system_settings), bgSettingsButton);
	GtkWidget *bgSystemStyleCaption = hildon_caption_new (bgGroup, "", bgSystemStyle, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (bgSystemStyleCaption), "");
	gtk_container_add (GTK_CONTAINER (bgContentArea), bgSystemStyleCaption);	
	
	GtkWidget *bgColorButton = gtk_color_button_new ();
	gdk_color_parse (self->priv->bgColor, &color);
	gtk_color_button_set_color (GTK_COLOR_BUTTON (bgColorButton), &color);
	hildon_gtk_widget_set_theme_size (bgColorButton, HILDON_SIZE_FINGER_HEIGHT);
	GtkWidget *bgColorCaption = hildon_caption_new (bgGroup, "Background color", bgColorButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (bgColorCaption), "");
	gtk_container_add (GTK_CONTAINER (bgContentArea), bgColorCaption);
	
	GtkWidget *bgOpacityPicker = hildon_picker_dialog_new (GTK_WINDOW (bgDialog));
	gtk_window_set_title (GTK_WINDOW (bgOpacityPicker), "Background opacity");
	GtkWidget *bgOpacitySelector = hildon_touch_selector_new_text ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (bgOpacityPicker), HILDON_TOUCH_SELECTOR (bgOpacitySelector));
	
	for (i=0; i<=100; i++) {
		gchar *text = g_strdup_printf ("%d%%", i);
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (bgOpacitySelector), text);
		g_free (text);
	}
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (bgOpacitySelector), 0, self->priv->bgOpacity);
	GtkWidget *bgOpacityButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (bgOpacityButton), "clicked", G_CALLBACK (countdown_home_widget_select_bg_opacity), bgOpacityPicker);
	gchar *name = g_strdup_printf ("%d%%", self->priv->bgOpacity);
	gtk_button_set_label (GTK_BUTTON (bgOpacityButton), name);
	g_free (name);
	GtkWidget *bgOpacityCaption = hildon_caption_new (bgGroup, "Background opacity", bgOpacityButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (bgOpacityCaption), "");
	gtk_container_add (GTK_CONTAINER (bgContentArea), bgOpacityCaption);
		
	GtkWidget *buttonAreaColorButton = gtk_color_button_new ();
	gdk_color_parse (self->priv->buttonAreaColor, &color);
	gtk_color_button_set_color (GTK_COLOR_BUTTON (buttonAreaColorButton), &color);
	hildon_gtk_widget_set_theme_size (buttonAreaColorButton, HILDON_SIZE_FINGER_HEIGHT);
	GtkWidget *buttonAreaColorCaption = hildon_caption_new (bgGroup, "Button area color", buttonAreaColorButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (buttonAreaColorCaption), "");
	gtk_container_add (GTK_CONTAINER (bgContentArea), buttonAreaColorCaption);
	
	GtkWidget *buttonAreaOpacityPicker = hildon_picker_dialog_new (GTK_WINDOW (bgDialog));
	gtk_window_set_title (GTK_WINDOW (buttonAreaOpacityPicker), "Button area opacity");
	GtkWidget *buttonAreaOpacitySelector = hildon_touch_selector_new_text ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (buttonAreaOpacityPicker), HILDON_TOUCH_SELECTOR (buttonAreaOpacitySelector));
	
	for (i=0; i<=100; i++) {
		gchar *text = g_strdup_printf ("%d%%", i);
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (buttonAreaOpacitySelector), text);
		g_free (text);
	}
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (buttonAreaOpacitySelector), 0, self->priv->buttonAreaOpacity);
	GtkWidget *buttonAreaOpacityButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (buttonAreaOpacityButton), "clicked", G_CALLBACK (countdown_home_widget_select_buttonArea_opacity), buttonAreaOpacityPicker);
	name = g_strdup_printf ("%d%%", self->priv->buttonAreaOpacity);
	gtk_button_set_label (GTK_BUTTON (buttonAreaOpacityButton), name);
	g_free (name);
	GtkWidget *buttonAreaOpacityCaption = hildon_caption_new (bgGroup, "Button area opacity", buttonAreaOpacityButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (buttonAreaOpacityCaption), "");
	gtk_container_add (GTK_CONTAINER (bgContentArea), buttonAreaOpacityCaption);
	
	GtkWidget *radiusPicker = hildon_picker_dialog_new (GTK_WINDOW (bgDialog));
	gtk_window_set_title (GTK_WINDOW (radiusPicker), "Corner radius");
	GtkWidget *radiusSelector = hildon_touch_selector_new_text ();
	hildon_picker_dialog_set_selector (HILDON_PICKER_DIALOG (radiusPicker), HILDON_TOUCH_SELECTOR (radiusSelector));
	
	for (i=0; i<=25; i++) {
		gchar *text = g_strdup_printf ("%d", i);
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (radiusSelector), text);
		g_free (text);
	}
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (radiusSelector), 0, self->priv->radius);
	GtkWidget *radiusButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (radiusButton), "clicked", G_CALLBACK (countdown_home_widget_select_radius), radiusPicker);
	name = g_strdup_printf ("%d", self->priv->radius);
	gtk_button_set_label (GTK_BUTTON (radiusButton), name);
	g_free (name);
	GtkWidget *radiusCaption = hildon_caption_new (bgGroup, "Corner radius", radiusButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (radiusCaption), "");
	gtk_container_add (GTK_CONTAINER (bgContentArea), radiusCaption);
	
	transDialog = gtk_dialog_new_with_buttons ("Change how terms display", dialog,
		GTK_DIALOG_DESTROY_WITH_PARENT, "Done", GTK_RESPONSE_ACCEPT, NULL);
	GtkWidget *transMain = gtk_dialog_get_content_area (GTK_DIALOG (transDialog));
	GtkWidget *scroll = hildon_pannable_area_new ();
	transContentArea = gtk_vbox_new (TRUE, 0);
	hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (scroll), transContentArea);
	gtk_container_add (GTK_CONTAINER (transMain), scroll);
	gtk_window_set_default_size (GTK_WINDOW (transDialog), -1, 350);
	
	GtkWidget *yearsEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (yearsEntry), self->priv->yearsText);
	hildon_entry_set_placeholder (HILDON_ENTRY (yearsEntry), self->priv->yearsText);
	GtkWidget *yearsCaption = hildon_caption_new (transGroup, "Years", yearsEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (yearsCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), yearsCaption);
	
	GtkWidget *weeksEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (weeksEntry), self->priv->weeksText);
	hildon_entry_set_placeholder (HILDON_ENTRY (weeksEntry), self->priv->weeksText);
	GtkWidget *weeksCaption = hildon_caption_new (transGroup, "Weeks", weeksEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (weeksCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), weeksCaption);
	
	GtkWidget *daysEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (daysEntry), self->priv->daysText);
	hildon_entry_set_placeholder (HILDON_ENTRY (daysEntry), self->priv->daysText);
	GtkWidget *daysCaption = hildon_caption_new (transGroup, "Days", daysEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (daysCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), daysCaption);

	GtkWidget *hoursEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (hoursEntry), self->priv->hoursText);
	hildon_entry_set_placeholder (HILDON_ENTRY (hoursEntry), self->priv->hoursText);
	GtkWidget *hoursCaption = hildon_caption_new (transGroup, "Hours", hoursEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (hoursCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), hoursCaption);
	
	GtkWidget *minutesEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (minutesEntry), self->priv->minutesText);
	hildon_entry_set_placeholder (HILDON_ENTRY (minutesEntry), self->priv->minutesText);
	GtkWidget *minutesCaption = hildon_caption_new (transGroup, "Minutes", minutesEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (minutesCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), minutesCaption);
	
	GtkWidget *leftEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (leftEntry), self->priv->leftText);
	hildon_entry_set_placeholder (HILDON_ENTRY (leftEntry), self->priv->leftText);
	GtkWidget *leftCaption = hildon_caption_new (transGroup, "Left", leftEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (leftCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), leftCaption);
	
	GtkWidget *agoEntry = hildon_entry_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	hildon_entry_set_text (HILDON_ENTRY (agoEntry), self->priv->agoText);
	hildon_entry_set_placeholder (HILDON_ENTRY (agoEntry), self->priv->agoText);
	GtkWidget *agoCaption = hildon_caption_new (transGroup, "Ago", agoEntry, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (agoCaption), "");
	gtk_container_add (GTK_CONTAINER (transContentArea), agoCaption);
	
	refreshButton = hildon_gtk_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	g_signal_connect (G_OBJECT (refreshButton), "clicked", G_CALLBACK (countdown_home_widget_select_refresh), picker);
	
	gchar *text = g_strdup_printf ("%d minute(s)", self->priv->refresh);
	gtk_button_set_label (GTK_BUTTON (refreshButton), text);
	g_free (text);
	
	refreshCaption = hildon_caption_new (group, "Update every", refreshButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (refreshCaption), "");
    gtk_container_add (GTK_CONTAINER (content_area), refreshCaption);
	
	GtkWidget *refreshLabel = gtk_label_new ("'0' will refresh only when switching to the desktop");
	gtk_label_set_markup (GTK_LABEL (refreshLabel), "<small>'0' will refresh only when switching to the desktop</small>");
	gtk_container_add (GTK_CONTAINER (content_area), refreshLabel);
	
	if (self->priv->systemFont) {
		gtk_button_set_label (GTK_BUTTON (fontSettingsButton), "System style");
	} else {
		gtk_button_set_label (GTK_BUTTON (fontSettingsButton), "User style");
	}
	g_signal_connect (fontSettingsButton, "clicked", G_CALLBACK (countdown_home_widget_title_settings), fontDialog);
	
	fontSettingsCaption = hildon_caption_new (group, "Fonts", fontSettingsButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (fontSettingsCaption), "");
	gtk_container_add (GTK_CONTAINER (content_area), fontSettingsCaption);

	if (self->priv->systemBG) {
		gtk_button_set_label (GTK_BUTTON (bgSettingsButton), "System style");
	} else {
		gtk_button_set_label (GTK_BUTTON (bgSettingsButton), "User style");
	}
	g_signal_connect (bgSettingsButton, "clicked", G_CALLBACK (countdown_home_widget_bg_settings), bgDialog);

	bgSettingsCaption = hildon_caption_new (group, "Background", bgSettingsButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (bgSettingsCaption), "");
	gtk_container_add (GTK_CONTAINER (content_area), bgSettingsCaption);
	
	gtk_button_set_label (GTK_BUTTON (transSettingsButton), "Change how terms display");
	g_signal_connect (transSettingsButton, "clicked", G_CALLBACK (countdown_home_widget_trans_settings), transDialog);
	
	transSettingsCaption = hildon_caption_new (group, "Translate", transSettingsButton, NULL, HILDON_CAPTION_MANDATORY);
	hildon_caption_set_separator (HILDON_CAPTION (transSettingsCaption), "");
	gtk_container_add (GTK_CONTAINER (content_area), transSettingsCaption);
	
    gtk_widget_show_all (dialog);

    int response=gtk_dialog_run (GTK_DIALOG (dialog));
    if (response == GTK_RESPONSE_ACCEPT) {
		GdkColor color;
		self->priv->refresh = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
		
		self->priv->systemFont = hildon_check_button_get_active (HILDON_CHECK_BUTTON (fontSystemStyle));
		gtk_color_button_get_color (GTK_COLOR_BUTTON (titleColorButton), &color);
		self->priv->titleColor = gdk_color_to_string (&color);
		self->priv->titleFont = g_markup_escape_text(gtk_font_button_get_font_name (GTK_FONT_BUTTON (titleFontButton)), -1);
		
		gtk_color_button_get_color (GTK_COLOR_BUTTON (valueColorButton), &color);
		self->priv->valueColor = gdk_color_to_string (&color);
		self->priv->valueFont = g_markup_escape_text(gtk_font_button_get_font_name (GTK_FONT_BUTTON (valueFontButton)), -1);
		
		self->priv->systemBG = hildon_check_button_get_active (HILDON_CHECK_BUTTON (bgSystemStyle));
		gtk_color_button_get_color (GTK_COLOR_BUTTON (bgColorButton), &color);
		self->priv->bgColor = gdk_color_to_string (&color);
		self->priv->bgOpacity = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (bgOpacitySelector), 0);
		self->priv->radius = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (radiusSelector), 0);
		
		gtk_color_button_get_color (GTK_COLOR_BUTTON (buttonAreaColorButton), &color);
		self->priv->buttonAreaColor = gdk_color_to_string (&color);
		self->priv->buttonAreaOpacity = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (buttonAreaOpacitySelector), 0);
		
		self->priv->yearsText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (yearsEntry)),-1);
		self->priv->weeksText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (weeksEntry)),-1);
		self->priv->daysText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (daysEntry)),-1);
		self->priv->hoursText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (hoursEntry)),-1);
		self->priv->minutesText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (minutesEntry)),-1);
		self->priv->leftText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (leftEntry)),-1);
		self->priv->agoText = g_markup_escape_text(hildon_entry_get_text (HILDON_ENTRY (agoEntry)),-1);
		
		if (self->priv->eventID != 0) {
			g_source_remove (self->priv->eventID);
			if (self->priv->refresh > 0) {
				self->priv->eventID = g_timeout_add (self->priv->refresh*1000*60, (GSourceFunc)countdown_home_widget_update_content, self);
			}
		}
		
		countdown_home_widget_write_settings (self);
		countdown_home_widget_update_content (self);
		gtk_widget_queue_draw (self);
    }
    gtk_widget_destroy (dialog);
}

void countdown_home_widget_make_view ( CountdownHomeWidget *self )
{
	//g_warning("setup_make_view");	
    gchar *filename;
    gboolean fileExists;
    GKeyFile *keyFile;
    GtkTreeIter iter;

	self->priv->entries = gtk_list_store_new (10, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);

    keyFile = g_key_file_new();
    filename = g_strconcat (HOME_DIR, "/.countdown_home.ini", NULL);
    fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);

    if (fileExists) {
        int i=0;
        gchar *entry;
        gboolean found;
        GError *error=NULL;
        entry = g_strdup_printf ("%d", i);
        found = g_key_file_has_group (keyFile, entry);
        while (found) {
            gchar *title, *date, *time, *readableTime;
            guint seconds, minutes, hours;
            gboolean *showWhich;
			gboolean removePassed;
            gsize length;

            sprintf (entry, "%d", i);
			title=g_key_file_get_string(keyFile,entry,"title",&error);
			if (error) {
				title="TITLE MISSING!";
				g_error_free(error);
				error=NULL;
			}
			date=g_key_file_get_string(keyFile,entry,"date",&error);
			if (error) {
				date="INCORRECT DATE!";
				g_error_free(error);
				error=NULL;
			}
			time=g_key_file_get_string(keyFile,entry,"time",&error);
			if (error) {
				time="0";
				g_error_free(error);
				error=NULL;
			}
			showWhich=g_key_file_get_boolean_list(keyFile,entry,"show",&length,&error);
			if (error) {
				gboolean show[5]={TRUE,TRUE,TRUE,TRUE,TRUE};
				showWhich=show;
				g_error_free(error);
				error=NULL;
			}
			removePassed=g_key_file_get_boolean (keyFile, entry, "removePassed", &error);
			if (error) {
				removePassed = FALSE;
				g_error_free (error);
				error = NULL;
			}

			seconds=g_ascii_strtoll(time,NULL,10);
			hours=seconds/(60*60);
			minutes=seconds/(60)-(hours*60);
			readableTime=g_strdup_printf("%02d:%02d",hours,minutes);
						
			gtk_list_store_append(self->priv->entries, &iter);
			gtk_list_store_set(self->priv->entries, &iter,0,title,1,date,2,time,3,readableTime,4,showWhich[0],5,showWhich[1],6,showWhich[2],7,showWhich[3],8,showWhich[4],9,removePassed,-1);
			snprintf(entry,5,"%d",++i);
			found=g_key_file_has_group(keyFile,entry);
			g_free(readableTime);
			g_free(title);
			g_free(date);
			g_free(time);
			g_free(showWhich);
        }
        self->priv->refresh = g_key_file_get_integer (keyFile, "config", "refresh", &error);
        if (error) {
            self->priv->refresh = 1;
            g_error_free (error);
            error = NULL;
        }
		self->priv->systemBG = g_key_file_get_boolean (keyFile, "config", "systemBG", &error);
		if (error) {
			self->priv->systemBG = TRUE;
			g_error_free (error);
			error = NULL;
		}
		self->priv->bgColor = g_key_file_get_string (keyFile, "config", "bgColor", &error);
		if (error) {
			self->priv->bgColor = "#000000";
			g_error_free (error);
			error = NULL;
		}
		self->priv->bgOpacity = g_key_file_get_integer (keyFile, "config", "bgOpacity", &error);
		if (error) {
			self->priv->bgOpacity = 50;
			g_error_free (error);
			error = NULL;
		}
		self->priv->radius = g_key_file_get_integer (keyFile, "config", "radius", &error);
		if (error) {
			self->priv->radius = 10;
			g_error_free (error);
			error = NULL;
		}
		self->priv->buttonAreaColor = g_key_file_get_string (keyFile, "config", "buttonAreaColor", &error);
		if (error) {
			self->priv->buttonAreaColor = "#000000";
			g_error_free (error);
			error = NULL;
		}
		self->priv->buttonAreaOpacity = g_key_file_get_integer (keyFile, "config", "buttonAreaOpacity", &error);
		if (error) {
			self->priv->buttonAreaOpacity = 75;
			g_error_free (error);
			error = NULL;
		}
		self->priv->systemFont = g_key_file_get_boolean (keyFile, "config", "systemFont", &error);
		if (error) {
			self->priv->systemFont = TRUE;
			g_error_free (error);
			error = NULL;
		}
		self->priv->titleColor = g_key_file_get_string (keyFile, "config", "titleColor", &error);
		if (error) {
			self->priv->titleColor = "#ffffff";
			g_error_free (error);
			error = NULL;
		}
		self->priv->titleFont = g_key_file_get_string (keyFile, "config", "titleFont", &error);
		if (error) {
			self->priv->titleFont = "Sans 18";
			g_error_free (error);
			error = NULL;
		}
		self->priv->valueColor = g_key_file_get_string (keyFile, "config", "valueColor", &error);
		if (error) {
			self->priv->valueColor = "#ababab";
			g_error_free (error);
			error = NULL;
		}
		self->priv->valueFont = g_key_file_get_string (keyFile, "config", "valueFont", &error);
		if (error) {
			self->priv->valueFont = "Sans 16";
			g_error_free (error);
			error = NULL;
		}
		self->priv->yearsText = g_key_file_get_string (keyFile, "labels", "yearsText", &error);
		if (error) {
			self->priv->yearsText = "years";
			g_error_free (error);
			error = NULL;
		}
		self->priv->weeksText = g_key_file_get_string (keyFile, "labels", "weeksText", &error);
		if (error) {
			self->priv->weeksText = "weeks";
			g_error_free (error);
			error = NULL;
		}
		self->priv->daysText = g_key_file_get_string (keyFile, "labels", "daysText", &error);
		if (error) {
			self->priv->daysText = "days";
			g_error_free (error);
			error = NULL;
		}
		self->priv->hoursText = g_key_file_get_string (keyFile, "labels", "hoursText", &error);
		if (error) {
			self->priv->hoursText = "hours";
			g_error_free (error);
			error = NULL;
		}
		self->priv->minutesText = g_key_file_get_string (keyFile, "labels", "minutesText", &error);
		if (error) {
			self->priv->minutesText = "mins";
			g_error_free (error);
			error = NULL;
		}
		self->priv->leftText = g_key_file_get_string (keyFile, "labels", "leftText", &error);
		if (error) {
			self->priv->leftText = "left";
			g_error_free (error);
			error = NULL;
		}
		self->priv->agoText = g_key_file_get_string (keyFile, "labels", "agoText", &error);
		if (error) {
			self->priv->agoText = "ago";
			g_error_free (error);
			error = NULL;
		}
    } else {
        self->priv->refresh = 1;
		self->priv->systemBG = TRUE;
		self->priv->bgOpacity = 50;
		self->priv->bgColor = "#000000";
		self->priv->radius = 10;
		self->priv->buttonAreaOpacity = 75;
		self->priv->buttonAreaColor = "#000000";
		self->priv->systemFont = TRUE;
		self->priv->titleColor = "#ffffff";
		self->priv->titleFont = "Sans 18";
		self->priv->valueColor = "#ababab";
		self->priv->valueFont = "Sans 16";
		self->priv->yearsText = "years";
		self->priv->weeksText = "weeks";
		self->priv->daysText = "days";
		self->priv->hoursText = "hours";
		self->priv->minutesText = "mins";
		self->priv->leftText = "left";
		self->priv->agoText = "ago";
    }

    g_key_file_free (keyFile);
    g_free (filename);
}

void countdown_home_widget_scroll_up (GtkWidget *button, GdkEventButton *event, CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_scroll_up");
	self->priv->offset--;
	countdown_home_widget_update_content (self);
}

void countdown_home_widget_scroll_down (GtkWidget *button, GdkEventButton *event, CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_scroll_down");
	self->priv->offset++;
	countdown_home_widget_update_content (self);
}

void countdown_home_widget_content_create (CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_content_create");
	GtkWidget *align = gtk_alignment_new (1, 1, 0, 0);
    self->priv->contents = gtk_vbox_new (TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->contents), 0);

    GtkWidget *addNewCountdown = gtk_image_new_from_icon_name ("general_add", HILDON_ICON_SIZE_FINGER);
	GtkWidget *addEvent = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (addEvent), FALSE);
	gtk_container_add (GTK_CONTAINER (addEvent), addNewCountdown);

    self->priv->buttons = GTK_HBOX (gtk_hbox_new (TRUE, 0));
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->buttons), 0);
	
	self->priv->upButton = gtk_image_new_from_icon_name ("rss_reader_move_up", HILDON_ICON_SIZE_FINGER);
	GtkWidget *upEvent = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (upEvent), FALSE);
	gtk_container_add (GTK_CONTAINER (upEvent), self->priv->upButton);
	self->priv->downButton = gtk_image_new_from_icon_name ("rss_reader_move_down", HILDON_ICON_SIZE_FINGER);
	GtkWidget *downEvent = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (downEvent), FALSE);
	gtk_container_add (GTK_CONTAINER (downEvent), self->priv->downButton);
	g_signal_connect (upEvent, "button-release-event", G_CALLBACK (countdown_home_widget_scroll_up), self);
	g_signal_connect (downEvent, "button-release-event", G_CALLBACK (countdown_home_widget_scroll_down), self);
	
	
	gtk_container_add (GTK_CONTAINER (self->priv->buttons), GTK_WIDGET (addEvent));
	gtk_container_add (GTK_CONTAINER (self->priv->buttons), GTK_WIDGET (upEvent));
	gtk_container_add (GTK_CONTAINER (self->priv->buttons), GTK_WIDGET (downEvent));
    
    g_signal_connect (addEvent, "button-release-event", G_CALLBACK (countdown_home_widget_add_event), self);
    
    gtk_box_pack_start (GTK_BOX (self->priv->homeWidget), GTK_WIDGET (self->priv->contents), FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (self->priv->homeWidget), align);
    gtk_box_pack_start (GTK_BOX (self->priv->homeWidget), GTK_WIDGET (self->priv->buttons), FALSE, FALSE, 0);
	
	g_signal_connect (self->priv->buttons, "expose-event", G_CALLBACK (countdown_home_widget_expose_button_area), self);
	
    gtk_widget_show_all (self->priv->homeWidget);
}

gboolean countdown_home_widget_update_content (CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_update_content");
	GDate *startDate=NULL, *endDate=NULL;
	gint startSec=0, endSec=0;
	time_t now;
	struct tm *timeinfo;
	GtkTreeIter iter;
	gint count=0, totalRows=0;
	gboolean exists=FALSE;
	GtkSizeGroup *titleGroup, *valueGroup;
	titleGroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	valueGroup = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);
	
	startDate = g_date_new();
	time (&now);
	g_date_set_time_t (startDate, now);
	timeinfo = localtime (&now);
	startSec = timeinfo->tm_hour*(60*60) + timeinfo->tm_min*60;
	
	endDate = g_date_new ();

	totalRows=gtk_tree_model_iter_n_children (GTK_TREE_MODEL (self->priv->entries), NULL);
	if (totalRows <= 3 || self->priv->offset < 0) {
		self->priv->offset = 0;
	} else if (totalRows > 3 && self->priv->offset > totalRows - 3) {
		self->priv->offset = totalRows -3;
	}
	
	gtk_container_foreach (GTK_CONTAINER (self->priv->contents), (GtkCallback) gtk_widget_destroy, NULL);
	
	exists = gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (self->priv->entries), &iter, NULL, self->priv->offset);
	if (exists) {
		while (exists) {
			GtkWidget *countdownButton;
			gchar *title="", *date="", *time="";
			gint years=0, weeks=0, days=0;
			gint hours=0, minutes=0, seconds=0;
			gint totDays=0;
			gboolean showYears, showWeeks, showDays;
			gboolean showHours, showMinutes, removePassed;
			gchar *tempContent="";
			gchar **content = g_new0 (gchar*, 6);
			gchar *contentLabelText="";
			gint index=0;
			gchar *tail="";
			gboolean firstShown;
			gboolean passed = FALSE;
			
			gtk_tree_model_get (GTK_TREE_MODEL (self->priv->entries), &iter, 0, &title, 1, &date, 2, &time, 4, &showYears, 5, &showWeeks, 6, &showDays, 7, &showHours, 8, &showMinutes, 9, &removePassed, -1);

			seconds = g_ascii_strtoll (time, NULL,  10);
			
			g_date_set_parse (endDate, date);
			if (!g_date_valid (endDate)) {
				title = g_strconcat (content[index++], "(INCORRECT DATE!)", NULL);
				totDays = 0;
			} else {
				totDays = g_date_days_between (startDate, endDate);
			}
			
			endSec = seconds;
			seconds = endSec-startSec;
			
			if (totDays > 0) {
				tail = g_strdup_printf (self->priv->leftText);
				if (seconds < 0) {
					totDays--;
					seconds = (60*60*24)+seconds;
				}
			} else if (totDays < 0) {
				tail = g_strdup_printf (self->priv->agoText);
				if (seconds > 0) {
					totDays++;
					seconds = (60*60*24)-seconds;
					passed = TRUE;
				}
			} else {
				if (seconds < 0) {
					tail = g_strdup_printf (self->priv->agoText);
					passed = TRUE;
				} else {
					tail = g_strdup_printf (self->priv->leftText);
				}
			}
			
			if (removePassed && passed) {
				exists = gtk_list_store_remove (self->priv->entries, &iter);
				g_free (title);
				g_free (date);
				g_free (time);
				g_free (tail);
				countdown_home_widget_write_settings (self);
				continue;
			}
			
			firstShown = FALSE;

			totDays=ABS(totDays);
			seconds=ABS(seconds);
			
			if (showYears) {
				years=totDays/365;
				if ((firstShown)||(years!=0)||((!showWeeks)&&(!showDays)&&(!showHours)&&(!showMinutes))) {
					firstShown = TRUE;
					tempContent = g_strdup_printf ("%d", ABS (years));
					content[index++] = g_strconcat (tempContent, " ",self->priv->yearsText," ", NULL);
					g_free (tempContent);
				}
			}
			if (showWeeks) {
				weeks = (totDays-(years*365))/7;
				if ((firstShown)||(weeks!=0)||((!showDays)&&(!showHours)&&(!showMinutes))) {
					firstShown = TRUE;
					tempContent = g_strdup_printf ("%d", ABS (weeks));
					content[index++] = g_strconcat (tempContent, " ",self->priv->weeksText," ", NULL);
					g_free (tempContent);
				}
			}
			if (showDays) {
				days = totDays-(weeks*7)-(years*365);
				if ((firstShown)||(days!=0)||((!showHours)&&(!showMinutes))) {
					firstShown = TRUE;
					tempContent = g_strdup_printf ("%d", ABS (days));
					content[index++] = g_strconcat (tempContent, " ",self->priv->daysText," ", NULL);
					g_free (tempContent);
				}
			}
			if (showHours) {
				hours = ((totDays-(days+(weeks*7)+(years*365)))*(24))+seconds/(60*60);
				if ((firstShown)||(hours!=0)||(!showMinutes)) {
					firstShown = TRUE;
					tempContent = g_strdup_printf ("%d", ABS (hours));
					content[index++] = g_strconcat (tempContent, " ",self->priv->hoursText," ", NULL);
					g_free (tempContent);
				}
			}
			if (showMinutes) {
				minutes=((totDays-(days+(weeks*7)+(years*365)))*(24*60))+((seconds/60)-(hours*60));
				firstShown = TRUE;
				tempContent = g_strdup_printf ("%d", ABS (minutes));
				content[index++] = g_strconcat (tempContent, " ",self->priv->minutesText," ", NULL);
				g_free (tempContent);
			}
			content[index] = g_strconcat (tail, NULL);
			
			contentLabelText = g_strjoinv (NULL, content);
			GtkWidget *event = gtk_event_box_new ();
			gtk_event_box_set_visible_window (GTK_EVENT_BOX (event), FALSE);
			gtk_container_set_border_width (GTK_CONTAINER (event), 5);
			GtkWidget *vbox = gtk_vbox_new (TRUE, 0);
			GtkWidget *titleLabel = gtk_label_new (title);
			gtk_label_set_ellipsize (GTK_LABEL (titleLabel), PANGO_ELLIPSIZE_END);
			if (self->priv->systemFont) {
				GdkColor color;
				gtk_style_lookup_color (gtk_rc_get_style (titleLabel), "DefaultTextColor", &color);
				gtk_label_set_markup (GTK_LABEL (titleLabel), g_strconcat ("<span color=\"", gdk_color_to_string (&color),"\">", title, "</span>", NULL));
			} else {
				gtk_label_set_markup (GTK_LABEL (titleLabel), g_strconcat ("<span font=\"", self->priv->titleFont,"\" color=\"",self->priv->titleColor, "\">", title, "</span>", NULL));
			}
			GtkWidget *contentLabel = gtk_label_new (contentLabelText);
			gtk_label_set_ellipsize (GTK_LABEL (contentLabel), PANGO_ELLIPSIZE_END);
			
			if (self->priv->systemFont) {
				GdkColor color;
				gtk_style_lookup_color (gtk_rc_get_style(contentLabel), "SecondaryTextColor", &color);
				gtk_label_set_markup (GTK_LABEL (contentLabel), g_strconcat ("<span size=\"smaller\" color=\"",gdk_color_to_string (&color),"\">", contentLabelText, "</span>", NULL));
			} else {
				gtk_label_set_markup (GTK_LABEL (contentLabel), g_strconcat ("<span font=\"", self->priv->valueFont,"\" color=\"",self->priv->valueColor, "\">", contentLabelText, "</span>", NULL));
			}

			gtk_container_add (GTK_CONTAINER (event), vbox);
			gtk_container_add (GTK_CONTAINER (vbox), titleLabel);
			gtk_container_add (GTK_CONTAINER (vbox), contentLabel);
			gtk_container_add (GTK_CONTAINER (self->priv->contents), event);
			
			g_signal_connect (event, "button-release-event", G_CALLBACK (countdown_home_widget_edit_event), self);
			
			gchar *name=g_strdup_printf("%d",self->priv->offset + count++);
			
			gtk_widget_set_name (event, name);
			
			g_free (contentLabelText);
			g_free (title);
			g_free (date);
			g_free (time);
			g_free (name);
			g_free (tail);
			g_strfreev (content);
			
			exists = gtk_tree_model_iter_next (GTK_TREE_MODEL (self->priv->entries), &iter);
			if (exists) {
				if (count >= 3) {
					exists = FALSE;
				}
			}
		}
	} else {
		GtkWidget *label = gtk_label_new ("You currently have no countdown events set up.  Press the '+' button below to start adding events!");
		gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
		gtk_widget_set_size_request (label, 352, -1);
		gtk_container_add (GTK_CONTAINER (self->priv->contents), label);
	}

	if (totalRows > 3) {
		if (self->priv->offset == 0) {
			gtk_widget_set_sensitive (self->priv->upButton, FALSE);
		} else {
			gtk_widget_set_sensitive (self->priv->upButton, TRUE);
		}
		
		if (self->priv->offset + 3 >= totalRows) {
			gtk_widget_set_sensitive (self->priv->downButton, FALSE);
		} else {
			gtk_widget_set_sensitive (self->priv->downButton, TRUE);
		}
	} else {
		gtk_widget_set_sensitive (self->priv->upButton, FALSE);
		gtk_widget_set_sensitive (self->priv->downButton, FALSE);
	}

	gtk_widget_show_all (self->priv->contents);
	
	g_date_free (startDate);
	g_date_free (endDate);
	
	return TRUE;
}

static void countdown_home_widget_check_desktop (GObject *gobject, GParamSpec *pspec, CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_check_desktop");
	gchar *name = pspec->name;
	gboolean status;
	g_object_get (gobject, name, &status, NULL);
	if (status) {
		countdown_home_widget_update_content (self);
		if (self->priv->eventID == 0) {
			if (self->priv->refresh > 0) {
				self->priv->eventID = g_timeout_add (self->priv->refresh*1000*60, (GSourceFunc)countdown_home_widget_update_content, self);
			}
		}
	} else if (self->priv->eventID != 0) {
		g_source_remove (self->priv->eventID);
		self->priv->eventID = 0;
	}
}

static void countdown_home_widget_dispose (GObject *object)
{
	//g_warning ("countdown_home_widget_dispose");
	CountdownHomeWidget *self = COUNTDOWN_HOME_WIDGET (object);

	G_OBJECT_CLASS (countdown_home_widget_parent_class)->dispose (object);
}

static void countdown_home_widget_finalize (GObject *object)
{
	//g_warning ("countdown_home_widget_finalize");
	CountdownHomeWidget *self = COUNTDOWN_HOME_WIDGET (object);

	if (self->priv->eventID) {
		g_source_remove (self->priv->eventID);
	}

	G_OBJECT_CLASS (countdown_home_widget_parent_class)->finalize (object);
}

static void countdown_home_widget_realize (GtkWidget *widget)
{
	//g_warning ("countdown_home_widget_realize");
	GdkScreen *screen;

	screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	GTK_WIDGET_CLASS (countdown_home_widget_parent_class)->realize (widget);
}

static gboolean countdown_home_widget_expose_button_area (GtkWidget *widget, GdkEventExpose *event, CountdownHomeWidget *self)
{
	cairo_t *cr;

    cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));
	gdk_cairo_region (cr, event->region);
	cairo_clip (cr);

    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	if (self->priv->systemBG) {
		GdkColor color;
		gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.8);
	} else {
		GdkColor color;
		gdk_color_parse (self->priv->buttonAreaColor, &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65535.0, color.blue/65535.0, self->priv->buttonAreaOpacity/100.0);
	}
    
	gint width, height, x, y;
	width = widget->allocation.width;
	height = widget->allocation.height;
	x = widget->allocation.x;
	y = widget->allocation.y;

	cairo_move_to(cr, x + self->priv->radius, y);
	cairo_line_to(cr, x + width, y);
	cairo_line_to(cr, x + width, y + height - self->priv->radius);
	cairo_curve_to(cr, x + width, y + height - self->priv->radius, x + width,
				y + height, x + width - self->priv->radius, y + height);
	cairo_line_to(cr, x + self->priv->radius, y + height);
	cairo_curve_to(cr, x + self->priv->radius, y + height, x, y + height, x,
				y + height - self->priv->radius);
	cairo_line_to(cr, x, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill(cr);

    cairo_destroy (cr);
	
	return FALSE;
}

static gboolean countdown_home_widget_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	//g_warning ("countdown_home_widget_expose_event");
	CountdownHomeWidget *self = COUNTDOWN_HOME_WIDGET (widget);
	cairo_t *cr;

    cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));

    cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
    cairo_set_source_rgba (cr, 0.0, 0.0, 0.0, 0.0);
    cairo_paint (cr);

    cairo_destroy (cr);
	
	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));
	if (self->priv->systemBG) {
		GdkColor color;
		gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	} else {
		GdkColor color;
		gdk_color_parse (self->priv->bgColor, &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65535.0, color.blue/65535.0, self->priv->bgOpacity/100.0);
	}
	
	gint width, height, x, y;
	width = widget->allocation.width;
	height = widget->allocation.height;
	x = widget->allocation.x;
	y = widget->allocation.y;

	cairo_move_to(cr, x + self->priv->radius, y);
	cairo_line_to(cr, x + width - self->priv->radius, y);
	cairo_curve_to(cr, x + width - self->priv->radius, y, x + width, y, x + width,
				y + self->priv->radius);
	cairo_line_to(cr, x + width, y + height - self->priv->radius);
	cairo_curve_to(cr, x + width, y + height - self->priv->radius, x + width,
				y + height, x + width - self->priv->radius, y + height);
	cairo_line_to(cr, x + self->priv->radius, y + height);
	cairo_curve_to(cr, x + self->priv->radius, y + height, x, y + height, x,
				y + height - self->priv->radius);
	cairo_line_to(cr, x, y + self->priv->radius);
	cairo_curve_to(cr, x, y + self->priv->radius, x, y, x + self->priv->radius, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill(cr);
	cairo_destroy(cr);

	return GTK_WIDGET_CLASS (countdown_home_widget_parent_class)->expose_event (widget, event);
}

static void countdown_home_widget_class_init (CountdownHomeWidgetClass *klass)
{
	//g_warning ("countdown_home_widget_class_init");
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = countdown_home_widget_dispose;
	object_class->finalize = countdown_home_widget_finalize;
	
	widget_class->realize = countdown_home_widget_realize;
	widget_class->expose_event = countdown_home_widget_expose_event;

	g_type_class_add_private (klass, sizeof (CountdownHomeWidgetPrivate));
}

static void countdown_home_widget_class_finalize (CountdownHomeWidgetClass *klass G_GNUC_UNUSED)
{
}

static void countdown_home_widget_init (CountdownHomeWidget *self)
{
	//g_warning ("countdown_home_widget_init");
	self->priv = COUNTDOWN_HOME_WIDGET_GET_PRIVATE (self);
	self->priv->offset = 0;
	self->priv->eventID = 0;
	
	gtk_window_set_default_size (GTK_WINDOW (self), 352, 252);
	
	hd_home_plugin_item_set_settings (&self->parent, TRUE);
	g_signal_connect (&self->parent, "show-settings", G_CALLBACK (countdown_home_widget_global_options), self);
	
	self->priv->homeWidget = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->homeWidget), 0);
	gtk_container_add (GTK_CONTAINER (self), self->priv->homeWidget);
	gtk_widget_show (self->priv->homeWidget);

	GdkGeometry hints;
	hints.min_width = 352;
	hints.min_height = 252;
	hints.max_width = 352;
	hints.max_height = 252;
	gtk_window_set_geometry_hints (GTK_WINDOW (self), self->priv->homeWidget, &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	
	countdown_home_widget_make_view (self);
	
	countdown_home_widget_content_create (self);
	
	g_signal_connect (self, "notify::is-on-current-desktop", G_CALLBACK (countdown_home_widget_check_desktop), self);
}

CountdownHomeWidget* countdown_home_widget_new (void)
{
  return g_object_new (COUNTDOWN_TYPE_HOME_WIDGET, NULL);
}
