/*
 * This file is part of contactinfos
 *
 * Copyright (C) 2007 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <glib.h>
#include <gmodule.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <backend/backend.h>

#include <dbus/dbus-error.h>
#include <contactinfos.h>

gboolean loadSymbol(GModule *module, gpointer *symbol, gchar *symbolName, GError ** error) {
#ifdef DEBUG
	g_printf("loadSymbol(\nGModule *module=%p, \ngpointer *symbol,\ngchar *symbolName=%s\n", module, symbolName);
#endif
	if (*symbol == NULL) {
		/* load our function in the module */
		if (!g_module_symbol (module, symbolName, symbol)) {
			g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_LOAD_SYMBOL_ERROR, "Backend loadSymbol : %s:%s: %s", g_module_name(module), symbolName, g_module_error ());
			if (!g_module_close (module)) {
				g_warning ("%s: %s", g_module_name(module), g_module_error ());
			}
			return FALSE;
		}
		if (*symbol == NULL) {
			g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_LOAD_SYMBOL_ERROR, "Backend loadSymbol : %s:%s: is NULL", g_module_name(module), symbolName);
			if (!g_module_close (module)) {
				g_warning ("%s: %s", g_module_name(module), g_module_error ());
			}
			return FALSE;
		}
	}
	return TRUE;
}

gboolean initBackend(gpointer *data, GError ** error) {

	gchar * message;
	GConfClient *gc_client;
	gint response;
	gchar *moduleName;
	gchar *libName;
	BackendData *pdata;

	(*data) = g_new0(BackendData, 1);
	pdata = (BackendData *) (*data);

	/* Load default GConf path */
	gc_client = gconf_client_get_default();

	if (gc_client == NULL) {
	    return FALSE;
	}

	/* Get string variable */
	moduleName = gconf_client_get_string(gc_client, CONTACTINFOS_GCONF_KEY, NULL);
	if (!moduleName) {
		moduleName = g_strdup(GPE_MODULE);
	}
	
	libName = g_strdup_printf(LIB_TEMPLATE, moduleName);
	g_free(moduleName);
	
	/* try gpe first */
	pdata->module = g_module_open (libName, G_MODULE_BIND_LAZY);
	if (!pdata->module) {
		message = g_strdup_printf("Backend loadModule : %s\n", g_module_error ());
		g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_LOAD_MODULE_ERROR,message);
		g_free(message);
		return FALSE;
	}

	if (!loadSymbol(pdata->module, (gpointer *)&(pdata->backendInit), INIT_BACKEND_SYMBOL, error)) {
		return FALSE;
	}

	return (pdata->backendInit)(&(pdata->backendData), error);
}

void freeBackendData(gpointer *data) {
	g_free(*data);
	*data=NULL;
}

gboolean terminateBackend(gpointer data, GError ** error) {
	
	gboolean ret;
	BackendData *pdata = (BackendData *) data;
	g_assert(pdata != NULL);
	
	if (!loadSymbol(pdata->module, (gpointer *)&(pdata->backendTerminate), TERMINATE_BACKEND_SYMBOL, error)) {
		return FALSE;
	}

	/* call our function in the module */
	ret = (pdata->backendTerminate)(pdata->backendData, error);

	if (!g_module_close (pdata->module)) {
		g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_UNLOAD_MODULE_ERROR,"Backend unloadModule %s : %s", g_module_name(pdata->module), g_module_error ());
		g_warning ("%s: %s", g_module_name(pdata->module), g_module_error ());
		freeBackendData(&data);
		return FALSE;
	}
	freeBackendData(&data);
	return ret;
}

gboolean backendGetAvailableContactInfos (gpointer data, guint requestedInfos, guint* availableInfos, GError ** error) {
	BackendData *pdata = (BackendData *) data;
	g_assert(pdata != NULL);
#ifdef DEBUG
	g_printf("backendGetAvailableContactInfos Before loading (\ngpointer data=%p, \nguint requestedInfos=%d\n", data, requestedInfos);
#endif

	if (!loadSymbol(pdata->module, (gpointer *)&(pdata->backendGetAvailableContactInfos), BACKEND_GET_AVAILABLE_CONTACTINFOS_SYMBOL, error)) {
		return FALSE;
	}
	/* call our function in the module */
	return (pdata->backendGetAvailableContactInfos)(pdata->backendData, requestedInfos, availableInfos, error);
}

gboolean backendGetContactInfos (gpointer data, guint requestedInfos, const gchar * searchText, GSList **response, GError ** error) {
	gboolean ret;
	BackendData *pdata = (BackendData *) data;
	g_assert(pdata != NULL);

	if (!loadSymbol(pdata->module, (gpointer *)&(pdata->backendGetContactInfos), BACKEND_GET_CONTACTINFOS_SYMBOL, error)) {
		return FALSE;
	}
	/* call our function in the module */
	ret = (pdata->backendGetContactInfos)(pdata->backendData, requestedInfos, searchText, response, error);
#ifdef DEBUG
	g_printf("backendGetContactInfos ret=%d before returning(\ngpointer data=%p, \nguint requestedInfos=%d,\ngchar *searchText=%s,\nGSList *(*response)=%p\n", ret, data, requestedInfos, searchText, *response);
#endif
	return ret;
}


