/*
 * This file is part of contactinfos
 *
 * Copyright (C) 2007 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <ctype.h>
#include <time.h>

#include "contactinfos.h"
#include "contacts-eds.h"

#include <libebook/e-book.h>

static EBook *book = NULL; /* Address Book from Evolution */

typedef struct _contactsDbPl ContactsDbPl;
struct _contactsDbPl {
	GSList *list;
	GSList *fields;
	guint requestedInfos;
	EContact *contact;
};

typedef struct _typeInfoField TypeInfoField;
struct _typeInfoField {
	EContactField field;
	guint typeInfo;
};

gboolean eds_open_pl (GError ** error) {
	GError *localError = NULL;
	/* open persistent connection */
	book = e_book_new_system_addressbook(&localError);
	if (book == NULL || localError) {
		if (localError) {
#ifdef DEBUG
			g_printf("eds_open_pl\n error=%d\n%s\n", localError->code, localError->message);
#endif
			g_propagate_error(error, localError);
		}
		return FALSE;
	}
	if (!e_book_open(book, FALSE, &localError)) {
		g_object_unref (book);
		return FALSE;
	}
	if (localError) {
#ifdef DEBUG
		g_printf("eds_open_pl book_open\n error=%d\t%s\n", localError->code, localError->message);
#endif
		g_propagate_error(error, localError);
	}
	return TRUE;
}

gboolean eds_close_pl (GError **error) {
	if (book != NULL) {
		g_object_unref(book);
	}
	return TRUE;
}

ContactInfosElement *newContactinfosElement (void) {
	ContactInfosElement *p = g_new0(ContactInfosElement, 1);
	p->valueArray = g_ptr_array_new();
	return p;
}

ContactInfosValue *newContactinfosValue (void) {
	ContactInfosValue *p = g_new0(ContactInfosValue, 1);
	return p;
}

TypeInfoField *newTypeInfoField (EContactField field, guint typeInfo) {
	TypeInfoField *p = g_new0(TypeInfoField, 1);
	p->field = field;
	p->typeInfo = typeInfo;
	return p;
}

void readValue (TypeInfoField *typeInfoField, ContactsDbPl *pdata) {

	ContactInfosValue *value;
	ContactInfosElement *record;
	EContact *contact;
	gchar * fieldValue;

	contact = (EContact *) (pdata->contact);
	record = (ContactInfosElement *) (pdata->list->data);
	
	fieldValue = (gchar *) e_contact_get (contact, typeInfoField->field);
	if (fieldValue) {
		value = newContactinfosValue();
		value->typeInfo = typeInfoField->typeInfo;
		value->type = g_strdup(e_contact_pretty_name (typeInfoField->field));
		value->value = fieldValue;
		g_ptr_array_add(record->valueArray, value);
	}
	
}

gchar *getNameForTypeInfo(guint typeInfo) {
	if (typeInfo == CI_PHONE_NUMBER) {
		return g_strdup("phone");
	}
	if (typeInfo == CI_EMAIL) {
		return g_strdup("email");
	}
	if (typeInfo == CI_WEB) {
		return g_strdup("URL");
	}
	return g_strdup("unknown");
}

/* Transform an (several) EContact Attributes into a (several) ContactInfosValue */
void addValue (ContactInfosElement *record, EVCardAttribute *attribute, guint typeInfo) {
	GList *values;
	GList *entry;
	ContactInfosValue *value;

#ifdef DEBUG
	g_printf("addValue (ContactInfosElement *record=%p, EVCardAttribute *attribute=%p, guint typeInfo=%d)\n", record, attribute, typeInfo);
#endif
	/* params = e_vcard_attribute_get_params(attribute); 
	*/
	if (attribute) {
		values = e_vcard_attribute_get_values (attribute);
		for (entry=values; entry; entry=g_list_next(entry)) {
			value = newContactinfosValue();
			value->typeInfo = typeInfo;
			value->type = getNameForTypeInfo(typeInfo);
			value->value = g_strdup(entry->data);
			g_ptr_array_add(record->valueArray, value);
		}
	} else {
#ifdef DEBUG
		g_printf("addValue %d is empty\n", typeInfo);
#endif
	}
}	

/* Transform an EContact into a ContactInfosElement */
void readContact (EContact *contact, ContactsDbPl *pdata) {

	gchar *familyName;
	gchar *givenName;
	ContactInfosElement *record;
	GList *attributes;
	GList *entry;
	EVCardAttribute *attribute;
	gchar *attributeName;

#ifdef DEBUG
	e_vcard_dump_structure(E_VCARD(contact));
#endif
	record = newContactinfosElement();

	if (pdata->requestedInfos & CI_NAME) {
		/*
		record->name = (gchar *) e_contact_get(contact, E_CONTACT_FULL_NAME);
		*/
		familyName = (gchar *) e_contact_get(contact, E_CONTACT_FAMILY_NAME);
		givenName = (gchar *) e_contact_get(contact, E_CONTACT_GIVEN_NAME);
		if (familyName && givenName) {
			record->name = g_strdup_printf("%s %s", givenName, familyName);
		} else if (familyName) {
			record->name = g_strdup(familyName);
		} else if (givenName) {
			record->name = g_strdup(givenName);
		}
		g_free(familyName);
		g_free(givenName);
	}
	if (!record->name) {
		record->name = g_strdup("TOTO");
	}
	if (pdata->requestedInfos & CI_COMPANY) {
		record->company = (gchar *) e_contact_get(contact, E_CONTACT_ORG);
	}
	
	attributes = e_vcard_get_attributes(E_VCARD(contact));
	
	for (entry=attributes; entry; entry=g_list_next(entry)) {
		attribute = (EVCardAttribute *) entry->data;
		attributeName = e_vcard_attribute_get_name(attribute);
#ifdef DEBUG
		g_printf("attributeName=%s\n", attributeName);
#endif
		if ((pdata->requestedInfos & CI_PHONE_NUMBER) && !strcmp(attributeName, EDS_VCARD_TYPE_TEL)) {
#ifdef DEBUG
		g_printf("PHONE\n");
#endif
			addValue(record, attribute, CI_PHONE_NUMBER);
		} else if ((pdata->requestedInfos & CI_EMAIL) && !strcmp(attributeName, EDS_VCARD_TYPE_EMAIL)) {
#ifdef DEBUG
		g_printf("EMAIL\n");
#endif
			addValue(record, attribute, CI_EMAIL);
		} else if ((pdata->requestedInfos & CI_WEB) && !strcmp(attributeName, EDS_VCARD_TYPE_URL)) {
#ifdef DEBUG
		g_printf("URL\n", attributeName);
#endif
			addValue(record, attribute, CI_WEB);
		}
	}
	
	pdata->list = g_slist_prepend(pdata->list, record);
	pdata->contact = contact;
/*
	g_slist_foreach(pdata->fields, (GFunc) readValue, (gpointer) pdata);
*/
}

gboolean eds_get_phone_list (const gchar *name, guint requestedInfos, GSList **listOut, GError ** error) {

	GList* list = NULL; /*List of contacts we shall return */
	EBookQuery* query; /*the resulting query*/
	GPtrArray * ptrArray;
	GPtrArray * ptrArrayField;
	ContactsDbPl *data;
	EContactField field;
	GError *localError=NULL;
#ifdef DEBUG
	gchar *request;
	GList *fields;
#endif

	*listOut = NULL;
	if (book == NULL)
		return FALSE;
		
#ifdef DEBUG
	g_printf("eds_get_phone_list (const gchar *name=%s,\n guint requestedInfos=%d\n",name, requestedInfos);
#endif
	data = g_new0(ContactsDbPl, 1 );
		      
	data->fields = NULL;
	data->requestedInfos = requestedInfos;
	data->list = NULL;
	
	ptrArray = g_ptr_array_new();
	ptrArrayField= g_ptr_array_new();
	
	
	if (requestedInfos & CI_PHONE_NUMBER) {
		/*
		for (field = E_CONTACT_FIRST_PHONE_ID; field<=E_CONTACT_LAST_PHONE_ID; field++) {
		*/
		for (field = E_CONTACT_PHONE_ASSISTANT; field<=E_CONTACT_PHONE_TTYTDD; field++) {
			data->fields = g_slist_prepend (data->fields, newTypeInfoField(field, CI_PHONE_NUMBER));
			g_ptr_array_add(ptrArrayField, e_book_query_field_exists (field));
		}
	}
	if (requestedInfos & CI_MOBILE_PHONE_NUMBER) {

		data->fields = g_slist_prepend (data->fields, newTypeInfoField(E_CONTACT_PHONE_MOBILE, CI_MOBILE_PHONE_NUMBER));
		g_ptr_array_add(ptrArrayField, e_book_query_field_exists (E_CONTACT_PHONE_MOBILE));
		data->fields = g_slist_prepend (data->fields, newTypeInfoField(E_CONTACT_PHONE_CAR, CI_MOBILE_PHONE_NUMBER));
		g_ptr_array_add(ptrArrayField, e_book_query_field_exists (E_CONTACT_PHONE_CAR));
	}
	if (requestedInfos & CI_FAX_PHONE_NUMBER) {
		data->fields = g_slist_prepend (data->fields, newTypeInfoField(E_CONTACT_PHONE_BUSINESS_FAX, CI_FAX_PHONE_NUMBER));
		g_ptr_array_add(ptrArrayField, e_book_query_field_exists (E_CONTACT_PHONE_BUSINESS_FAX));
		data->fields = g_slist_prepend (data->fields, newTypeInfoField(E_CONTACT_PHONE_HOME_FAX, CI_FAX_PHONE_NUMBER));
		g_ptr_array_add(ptrArrayField, e_book_query_field_exists (E_CONTACT_PHONE_HOME_FAX));
		data->fields = g_slist_prepend (data->fields, newTypeInfoField(E_CONTACT_PHONE_OTHER_FAX, CI_FAX_PHONE_NUMBER));
		g_ptr_array_add(ptrArrayField, e_book_query_field_exists (E_CONTACT_PHONE_OTHER_FAX));
	}
	if (requestedInfos & CI_FULL_ADDRESS) {
		for (field = E_CONTACT_ADDRESS_LABEL_HOME; field<=E_CONTACT_ADDRESS_LABEL_OTHER; field++) {
			data->fields = g_slist_prepend (data->fields, newTypeInfoField(field, CI_FULL_ADDRESS));
			g_ptr_array_add(ptrArrayField, e_book_query_field_exists (field));
		}
	}
	if (requestedInfos & CI_WEB) {
		for (field = E_CONTACT_HOMEPAGE_URL; field<=E_CONTACT_BLOG_URL; field++) {
			data->fields = g_slist_prepend (data->fields, newTypeInfoField(field, CI_FULL_ADDRESS));
			g_ptr_array_add(ptrArrayField, e_book_query_field_exists (field));
		}
	}
	if (requestedInfos & CI_EMAIL) {
		for (field = E_CONTACT_FIRST_EMAIL_ID; field<=E_CONTACT_LAST_EMAIL_ID; field++) {
			data->fields = g_slist_prepend (data->fields, newTypeInfoField(field, CI_EMAIL));
			g_ptr_array_add(ptrArrayField, e_book_query_field_exists (field));
		}
	}
	if (requestedInfos & CI_NAME) {
		for (field = E_CONTACT_FULL_NAME; field<=E_CONTACT_NICKNAME; field++) {
			data->fields = g_slist_prepend (data->fields, newTypeInfoField(field, CI_NAME));
			g_ptr_array_add(ptrArrayField, e_book_query_field_exists (field));
		}
	}
	if (requestedInfos & CI_COMPANY) {
		data->fields = g_slist_prepend (data->fields, newTypeInfoField(E_CONTACT_ORG, CI_COMPANY));
		g_ptr_array_add(ptrArrayField, e_book_query_field_exists (E_CONTACT_ORG));
	}
	if (ptrArrayField->len) {
		g_ptr_array_add(ptrArray, e_book_query_or (ptrArrayField->len, (EBookQuery**) (ptrArrayField->pdata), TRUE));
	}
	g_ptr_array_free(ptrArrayField, FALSE);
	data->fields = g_slist_reverse(data->fields);

/*
	query = e_book_query_any_field_contains(name);
	e_book_query_field_test (E_CONTACT_NAME_OR_ORG, E_BOOK_QUERY_CONTAINS, name)
*/
	g_ptr_array_add(ptrArray, e_book_query_any_field_contains(name));

	query = e_book_query_and (ptrArray->len, (EBookQuery**) (ptrArray->pdata), TRUE);
	g_ptr_array_free(ptrArray, FALSE);
	if (query == NULL)
		return FALSE;

#ifdef DEBUG
	request = e_book_query_to_string (query);
	g_printf("eds_get_phone_list \n strsearch=%s\n  request=%s\n", name, request);
	g_free(request);
#endif


	if (!e_book_get_contacts(book, query, &list, &localError)) {
		e_book_query_unref (query);
		return FALSE;
	}
	e_book_query_unref (query);
	
	if (localError) {
#ifdef DEBUG
		g_printf("eds_get_phone_list \n error=%d\t%s\n", localError->code, localError->message);
#endif
		g_propagate_error(error, localError);
	} else {
	
		g_list_foreach(list, (GFunc) readContact, (gpointer) data);
		
#ifdef DEBUG
		g_printf("eds_get_phone_list \n list from Query=%p\nlength=%d\n", list, g_list_length(list));
#endif
		/* Free GList */
		
		*listOut =data->list;

		g_free(data);
#ifdef DEBUG
		g_printf("eds_get_phone_list \n list=%p\nlength=%d\n", *listOut, g_slist_length(*listOut));
#endif
	}
	
	return TRUE;
}
