/*
 * This file is part of contactinfos
 *
 * Copyright (C) 2007 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <ctype.h>
#include <time.h>

#include "contactinfos.h"
#include "contacts-db-pl.h"

#define DB_NAME_CONTACTS "/.gpe/contacts"

#include <sqlite.h>

static sqlite *db = NULL; 

#ifdef DEBUG
static GIOChannel *fileOut = NULL;
void writeOut (gchar *line, ...) {
	va_list ap;
	gchar *l; 
	GError *le = NULL;
	if (!line) {
		l = g_strdup("\n");
	} else {
		va_start(ap, line);
		l = g_strdup_vprintf(line, ap);
		va_end(ap);
	}
	if (!fileOut) {
		fileOut = g_io_channel_new_file ("/tmp/contactinfos.log", "w", &le);
		if (le) {
			g_printf("g_io_channel_new_file code %d msg %s\n", le->code, le->message);
		}
		if (!fileOut) {
			g_printf("Cannot create file /tmp/contactinfos.log\n");
		}
	}
	if (fileOut) {
		g_io_channel_write_chars (fileOut, l, -1, NULL, &le);
		if (le) {
			g_printf("g_io_channel_write_chars code %d msg %s\n", le->code, le->message);
		}
		g_io_channel_flush(fileOut, &le);
		if (le) {
			g_printf("g_io_channel_flush code %d msg %s\n", le->code, le->message);
		}
		g_free(l);
	}
}
#endif

typedef struct _contactsDbPl ContactsDbPl;
struct _contactsDbPl {
	GSList *list;
	guint requestedInfos;
	gchar *request;
};

gboolean db_open_pl (GError ** error)
{
	/* open persistent connection */
	char *errmsg;
	char *buf;

	buf = g_strdup_printf("%s/%s", g_get_home_dir(), DB_NAME_CONTACTS);
        if (db) {
		sqlite_close(db);
	}
	db = sqlite_open (buf, 0, &errmsg);
	g_free (buf);
		    	
	if (db == NULL) {
		g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_INITIALIZE_ERROR,"Backend GPE : %s", errmsg);
		g_print (errmsg);
		free (errmsg);
		return FALSE;
	}

	/* if we can create this table, omething is wrong */
	if (sqlite_exec (db, "create table contacts (urn INTEGER NOT NULL, tag TEXT NOT NULL, value TEXT NOT NULL);", NULL, NULL, NULL) == SQLITE_OK) {
		sqlite_exec (db, "drop table contacts;", NULL, NULL, NULL);
		g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_ACCESS_ERROR,"The table contact doesn't seem to exist in the database ???");
		return FALSE;
	}

	return TRUE;
}

ContactInfosElement *newContactinfosElement (void) {
	ContactInfosElement *p = g_new0(ContactInfosElement, 1);
	p->valueArray = g_ptr_array_new();
	return p;
}

ContactInfosValue *newContactinfosValue (void) {
	ContactInfosValue *p = g_new0(ContactInfosValue, 1);
	return p;
}

static int read_phone (void *arg, int argc, char **argv, char **names) {

	ContactsDbPl *pdata = (ContactsDbPl *) arg;
	ContactInfosValue *value;
	ContactInfosElement *record;
	guint typeInfo = CI_NOTHING;
	gchar *type;

	record = (ContactInfosElement *)(pdata->list->data);

#ifdef DEBUG
	writeOut("read_phone name %s type %s  value %s\n",record->name, argv[0], argv[1]);
#endif	
	if (g_str_has_suffix(argv[0], ".TELEPHONE")) {
		typeInfo = CI_PHONE_NUMBER;
		if (g_str_has_prefix(argv[0], "HOME.")) {
			type=g_strdup(CI_HOME_PHONE);
		} else {
			type=g_strdup(CI_WORK_PHONE);
		}
	} else if (g_str_has_suffix(argv[0], ".MOBILE")) {
		typeInfo = CI_MOBILE_PHONE_NUMBER;
		if (g_str_has_prefix(argv[0], "HOME.")) {
			type=g_strdup(CI_HOME_MOBILE);
		} else {
			type=g_strdup(CI_WORK_MOBILE);
		}
	} else if (g_str_has_suffix(argv[0], ".FAX")) {
		typeInfo = CI_FAX_PHONE_NUMBER;
		if (g_str_has_prefix(argv[0], "HOME.")) {
			type=g_strdup(CI_HOME_FAX);
		} else {
			type=g_strdup(CI_WORK_FAX);
		}
	} else if (g_str_has_suffix(argv[0], ".ADDRESS")) {
		typeInfo = CI_FULL_ADDRESS;
		if (g_str_has_prefix(argv[0], "HOME.")) {
			type=g_strdup(CI_HOME_ADDRESS);
		} else {
			type=g_strdup(CI_WORK_ADDRESS);
		}
	} else if (g_str_has_suffix(argv[0], ".EMAIL")) {
		typeInfo = CI_EMAIL;
		if (g_str_has_prefix(argv[0], "HOME.")) {
			type=g_strdup(CI_HOME_EMAIL);
		} else {
			type=g_strdup(CI_WORK_EMAIL);
		}
	} else if (!strcmp(argv[0], "NAME")) {
		typeInfo = CI_NAME;
		type=g_strdup(CI_NAME_STR);
	} else if (!strcmp(argv[0], "COMPANY")) {
		typeInfo = CI_COMPANY;
		type=g_strdup(CI_COMPANY_STR);
	}

#ifdef DEBUG
/*
	writeOut("read_phone type %X  info %s value %s\n", typeInfo, type, argv[1]);
*/
#endif
	if (typeInfo != CI_NOTHING) {
		value = newContactinfosValue();
		value->typeInfo = typeInfo;
		value->type = type;
		value->value = g_strdup(argv[1]);
		g_ptr_array_add(record->valueArray, value);
	}

	return 0;
}


static int read_one_entry (void *arg, int argc, char **argv, char **names) {

	char *err;
	int r;
	gint urn = atoi (argv[0]);
	ContactsDbPl *pdata = (ContactsDbPl *) arg;
	ContactInfosElement *record = newContactinfosElement();

/*
	simplier now
	sqlite_exec_printf (db, "select tag,value,name,company from contacts,contacts_urn where (contacts.urn=%d) and (contacts_urn.urn=contacts.urn) and ((tag='HOME.TELEPHONE') or (tag='HOME.FAX') or (tag='HOME.MOBILE') or (tag='WORK.TELEPHONE') or (tag='WORK.FAX') or (tag='WORK.MOBILE'))", read_phone, arg, NULL, urn);
*/
	if ((pdata->requestedInfos & CI_NAME) && argv[1] && strlen(argv[1])!=0 && strncmp(argv[1],"(NULL)",6)) {
		record->name = g_strdup(argv[1]);
	}
	if ((pdata->requestedInfos & CI_COMPANY) && argv[2] && strlen(argv[2])!=0 && strncmp(argv[2],"(NULL)",6)) {
		record->company = g_strdup(argv[2]);
	}
	pdata->list = g_slist_prepend(pdata->list, record);
	r = sqlite_exec_printf (db, pdata->request, read_phone, arg, &err, urn);

	if (r) {
		g_print (err);
		free (err);
	}
        return r;
}

gboolean db_get_phone_list (const gchar *name, guint requestedInfos, GSList **listOut, GError ** error) {
	char *err;
	int r;
	gchar *strsearch;
	gchar* endRequest = NULL;
	ContactsDbPl *data;

	*listOut = NULL;
		
#ifdef DEBUG
	writeOut("db_get_phone_list (const gchar *name=%s,\n guint requestedInfos=%d\n",name, requestedInfos);
#endif
	data = g_new0(ContactsDbPl, 1 );
		      
	data->requestedInfos = requestedInfos;
	if (requestedInfos & CI_PHONE_NUMBER) {
		endRequest = g_strdup("(tag='HOME.TELEPHONE') or (tag='WORK.TELEPHONE')");
	}
	if (requestedInfos & CI_MOBILE_PHONE_NUMBER) {
		endRequest = g_strjoin(" or ", "(tag='HOME.MOBILE') or (tag='WORK.MOBILE')", endRequest, NULL);
	}
	if (requestedInfos & CI_FAX_PHONE_NUMBER) {
		endRequest = g_strjoin(" or ", "(tag='HOME.FAX') or (tag='WORK.FAX')", endRequest, NULL);
	}
	if (requestedInfos & CI_FULL_ADDRESS) {
		endRequest = g_strjoin(" or ", "(tag='HOME.ADDRESS') or (tag='WORK.ADDRESS')", endRequest, NULL);
	}
	if (requestedInfos & CI_EMAIL) {
		endRequest = g_strjoin(" or ", "(tag='HOME.EMAIL') or (tag='WORK.EMAIL')", endRequest, NULL);
	}
	if (requestedInfos & CI_NAME) {
		endRequest = g_strjoin(" or ", "tag='NAME'", endRequest, NULL);
	}
	if (requestedInfos & CI_COMPANY) {
		endRequest = g_strjoin(" or ", "tag='COMPANY'", endRequest, NULL);
	}
	if (endRequest && strlen(endRequest)>0) {
		endRequest = g_strconcat(" and (", endRequest, ")", NULL);
	}
	data->request = g_strconcat("select tag,value from contacts where urn=%d", endRequest, NULL);
	g_free(endRequest);

	strsearch = g_strdup_printf("%%%s%%", name);
	
#ifdef DEBUG
	writeOut("db_get_phone_list \n strsearch=%s\n  data->request=%s\n", strsearch, data->request);
#endif
	r = sqlite_exec_printf(db, "select distinct contacts.urn, name, company from contacts,contacts_urn where (contacts_urn.urn=contacts.urn) and (tag = 'NAME' or tag = 'GIVEN_NAME' or tag = 'FAMILY_NAME' or tag = 'COMPANY' or tag = 'MIDDLE_NAME') and value like '%q'", read_one_entry, data, &err, strsearch);

	*listOut = data->list;

	g_free(data->request);
	g_free(data);
	g_free(strsearch);
	if (r) {
		g_set_error (error, CONTACTINFOS_DBUS_ERROR, CI_BACKEND_ACCESS_ERROR,"Backend GPE : %s", err);
		g_print (err);
		free (err);
		return FALSE;
	}
	
#ifdef DEBUG
	writeOut("db_get_phone_list \n list=%p\n", *listOut);
#endif
	return TRUE;
}
