/*
 * Copyright (C) 2010 Collabora Ltd.
 *   @author Xavier Claessens <xavier.claessens@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <libosso-abook/osso-abook.h>

#include "contact-window.h"

typedef struct {
  OssoABookMenuExtension parent_instance;

  GtkWidget *view;
  GtkTreePath *path;
} MainViewPlugin;

typedef struct {
  OssoABookMenuExtensionClass parent_class;
} MainViewPluginClass;

G_GNUC_INTERNAL GType main_view_plugin_get_type (void) G_GNUC_CONST;
void main_view_plugin_register (GTypeModule *module);

G_DEFINE_DYNAMIC_TYPE (MainViewPlugin, main_view_plugin,
    OSSO_ABOOK_TYPE_MENU_EXTENSION);

static void
tap_and_hold_cb (GtkWidget *widget,
    MainViewPlugin *self)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  OssoABookContact *contact;
  GtkWidget *window;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (self->view));
  if (!gtk_tree_model_get_iter (model, &iter, self->path))
    return;

  gtk_tree_model_get (model, &iter,
      OSSO_ABOOK_LIST_STORE_COLUMN_CONTACT, &contact,
      -1);

  window = contact_window_new (contact);
  gtk_widget_show (window);

  g_object_unref (contact);
}

static gboolean
tap_and_hold_query_cb (GtkWidget *widget,
    GdkEvent *event,
    MainViewPlugin *self)
{
  gint x, y;
  gboolean success;

  if (self->path != NULL)
    gtk_tree_path_free (self->path);
  self->path = NULL;

  gtk_tree_view_convert_widget_to_bin_window_coords (GTK_TREE_VIEW (self->view),
      (gint) event->button.x, (gint) event->button.y,
      &x, &y);

  success = gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (self->view), x, y,
      &self->path, NULL, NULL, NULL);

  /* Prevent GtkTreeView from doing roughly the same and more */
  g_signal_stop_emission_by_name (widget, "tap-and-hold-query");

  /* If we didn't get a valid path, disable tap-and-hold */
  return !success;
}

static GtkWidget *
lookup_child_with_type (GtkWidget *widget,
    GType type)
{
  if (g_type_is_a (G_OBJECT_TYPE (widget), type))
    return widget;

  if (GTK_IS_CONTAINER (widget))
    {
      GList *childs;

      childs = gtk_container_get_children (GTK_CONTAINER (widget));
      while (childs != NULL)
        {
          GtkWidget *w;

          w = lookup_child_with_type (childs->data, type);
          if (w != NULL)
            {
              g_list_free (childs);
              return w;
            }

          childs = g_list_delete_link (childs, childs);
        }
    }

  return NULL;
}

static gboolean
register_tap_and_old_idle_cb (gpointer user_data)
{
  MainViewPlugin *self = user_data;
  GtkWindow *window;
  GtkWidget *menu;
  GtkWidget *item;

  window = osso_abook_menu_extension_get_parent (OSSO_ABOOK_MENU_EXTENSION (self));

  /* HACK: recurse through all its childs to find the treeview */
  self->view = lookup_child_with_type (GTK_WIDGET (window), GTK_TYPE_TREE_VIEW);
  g_assert (self->view != NULL);

  item = gtk_menu_item_new_with_label ("Zoom");
  g_signal_connect (item, "activate", G_CALLBACK (tap_and_hold_cb), self);
  gtk_widget_show (item);

  menu = gtk_menu_new ();
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_widget_tap_and_hold_setup (self->view, menu, NULL, GTK_TAP_AND_HOLD_NONE);

  /* HACK: OssoABookTreeView internally connect that signal and stops its
   * emition. We have to disconnect it to make our own. */
  g_signal_handlers_disconnect_matched (self->view,
      G_SIGNAL_MATCH_ID | G_SIGNAL_MATCH_DATA,
      g_signal_lookup ("tap-and-hold-query", GTK_TYPE_WIDGET),
      0, NULL, NULL, NULL);

  g_signal_connect (self->view, "tap-and-hold-query",
      G_CALLBACK (tap_and_hold_query_cb), self);

  return FALSE;
}

static void
main_view_plugin_init (MainViewPlugin *self)
{
  /* HACK: We want to override the long tap action of the main view. We know the
   * parent window of this menu extention is the main window. But the main
   * window's menu is created before its childs widgets, so we have to
   * introspect the window in idle callback to find the tree view.
   */
  g_idle_add (register_tap_and_old_idle_cb, self);
}

static OssoABookMenuEntry entries[] = {
};

static const OssoABookMenuEntry *
get_menu_entries (OssoABookMenuExtension *self)
{
  return entries;
}

static int
get_n_menu_entries (OssoABookMenuExtension *self)
{
  return G_N_ELEMENTS (entries);
}

void
main_view_plugin_register (GTypeModule *module)
{
  main_view_plugin_register_type (module);
}

static void
main_view_plugin_class_init (MainViewPluginClass *klass)
{
  OssoABookMenuExtensionClass *menu_extension_class;

  menu_extension_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (klass);
  menu_extension_class->get_n_menu_entries = get_n_menu_entries;
  menu_extension_class->get_menu_entries = get_menu_entries;
  menu_extension_class->name = OSSO_ABOOK_MENU_NAME_MAIN_VIEW;
}

static void
main_view_plugin_class_finalize (MainViewPluginClass *klass)
{
}
