#include "writewindow.h"
#include "ui_writewindow.h"

#include "processdialog.h"
#include "ui_processdialog.h"

#include "configreader.h"
#include "ui_configreader.h"

writeWindow::writeWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::writeWindow)
{
    // writeParent is a pointer to writeWindow's parent (configReader)
    writeParent = (configReader*) parent;

    ui->setupUi(this);
    // Hide the car, progress bar and open button at startup
    ui->carLbl->hide();
    ui->progBar->hide();
    ui->openBtn->hide();

    // Set the mainLabels text color to black
    ui->mainLbl1->setStyleSheet("color: black;");
    ui->mainLbl2->setStyleSheet("color: black;");
    ui->mainLbl3->setStyleSheet("color: black;");
    // Set the backgrounds
    this->setStyleSheet("QMainWindow { background-image: url(:/gfx/backsave.png) }");

    // If the dialog is cancelled, writeParent->saveDir is empty, but bkpDir isn't
    // saveDialog has bkpDir as default location, no empty path possible this way
    bkpDir.setPath(writeParent->saveDir.absolutePath());

    connect(ui->saveBtn, SIGNAL(clicked()),
            this, SLOT(writeFile()));
    connect(ui->openBtn, SIGNAL(clicked()),
            this, SLOT(openFile()));
}

writeWindow::~writeWindow()
{
    delete ui;
}

void writeWindow::writeFile() {
    // Get the save directory from the user
    writeParent->saveDir.setPath(QFileDialog::getExistingDirectory(this,
                                                      "Select directory",
                                                      bkpDir.absolutePath(),
                                                      QFileDialog::ShowDirsOnly));

    // Only save the file if the dialog hasn't been cancelled
    if(writeParent->saveDir.path().isEmpty() == false) {
        // Hide the saveBtn, show the progress bar and carLbl
        ui->saveBtn->hide();
        ui->progBar->show();
        ui->carLbl->show();

        // Set the background image
        this->setStyleSheet("QMainWindow { background-image: url(:/gfx/backdone.png) }");

        // Save the new selected directory to GConf
        system("gconftool -s /apps/maemo/configreader/save_dir -t string " + writeParent->saveDir.absolutePath().toAscii());

        curDate = QDate::currentDate();
        curTime = QTime::currentTime();

        // Set the filename string
        if(writeParent->cameFrom == "process") {
            // creadDDMMYYYY-HHMMSS.html
            preSaveFileName = "cread";
        }
        else if(writeParent->cameFrom == "compare") {
            // cread-diffDDMMYYYY-HHMMSS.html
            preSaveFileName = "cread-diff";
        }
        preSaveFileName.append(curDate.toString(writeParent->dateFormat)
                                + "-"
                                + curTime.toString("HHmmss")
                                + ".html");

        // Update main label
        if(writeParent->cameFrom == "process") {
            ui->mainLbl1->setText("");
            ui->mainLbl2->setText("Writing " + preSaveFileName);
            ui->mainLbl3->setText("");
            // Edit the output to HTML before saving
            editOutData();
        }
        else if(writeParent->cameFrom == "compare") {
            ui->mainLbl1->setText("Comparing files and writing");
            ui->mainLbl2->setText(preSaveFileName);
            ui->mainLbl3->setText("");
            // Read the difference between the files before saving
            outputDiff();
        }

        // Set filename and open it (truncate, write-only)
        saveFile.setFileName(writeParent->saveDir.absolutePath() + "/" + preSaveFileName);
        saveFile.open(QFile::Truncate | QFile::WriteOnly);

            // Write HTML
            writeStrm.setDevice(&saveFile);
            writeStrm << "<html>\n"
                    << "\t<head>\n"
                    << "\t\t<title>\n"
                    << "\t\t\t"
                    << curDate.toString(writeParent->dateFormatTitle) << ", "
                    << curTime.toString("HH:mm:ss");

            if(writeParent->cameFrom == "process")
                writeStrm << " (Config Reader)";
            else if(writeParent->cameFrom == "compare")
                writeStrm << " (Comparison, Config Reader)";

            writeStrm << "\n"
                    << "\t\t</title>\n"
                    << "\t</head>\n"
                    << "\t<body style='font-family: monospace'>\n";

            if(writeParent->cameFrom == "process")
                writeStrm << writeParent->outData;
            else if(writeParent->cameFrom == "compare")
                writeStrm << fileDiff;

            writeStrm << "\t</body>\n"
                    << "</html>";

        // Close the file
        saveFile.close();

        // Hide the empty label before showing "Done!" label
        ui->mainLbl1->setText("Done!");
        ui->mainLbl2->setText(writeParent->saveDir.absolutePath() + "/");
        ui->mainLbl3->setText(QFileInfo(saveFile).fileName());
        // Hide the progress bar
        ui->progBar->hide();

        // Show the open button
        ui->openBtn->show();
    }
}





void writeWindow::editOutData() {
    // Make outDataSplitted, split writeParent->outData by "\n"
    outDataSplitted = writeParent->outData.split("\n", QString::KeepEmptyParts, Qt::CaseInsensitive);

    // Empty writeParent->outData
    writeParent->outData.clear();

    // Read outDataSplitted line by line
    for(i=0; i<outDataSplitted.length(); i++) {
        // tempData = outDataSplitted[#], trimmed() (removes all whitespace and /n, /t etc.)
        tempData = outDataSplitted[i];
        tempData = tempData.trimmed();

        // Make al directory lines bold, make top directory bigger + linebreaks before it
        if(tempData.startsWith("/") && tempData.endsWith(":")) {
            // Top directories
            if(tempData.count("/", Qt::CaseInsensitive) == 1) {
                // Don't put linebreaks before first line
                if(i != 0)
                    outDataSplitted[i] = "<br /><br /><br /><font size='+5' color='#0000FF'><b>" + outDataSplitted[i] + "</b></font>";
                else
                    outDataSplitted[i] = "<font size='+5' color='#0000FF'><b>" + outDataSplitted[i] + "</b></font>";
            }
            // Subdirectories
            else
                outDataSplitted[i] = "<font color='#4D4DFF'><b>" + outDataSplitted[i] + "</b></font>";
        }

        // Start every line with "\t\t" (for HTML neatness)
        outDataSplitted[i] = "\t\t" + outDataSplitted[i];

        // Append every line with "<br />\n" (newline in HTML and C++)
        outDataSplitted[i] = outDataSplitted[i] + "<br />";

        // Replace all spaces with "&nbsp;", except for HTML tags
        tempData = outDataSplitted[i];
        tempData.replace(" ", "&nbsp;", Qt::CaseInsensitive);
        tempData.replace("<br&nbsp;/>", "<br />", Qt::CaseInsensitive);
        tempData.replace("<font&nbsp;size", "<font size", Qt::CaseInsensitive);
        tempData.replace("&nbsp;color='", " color='", Qt::CaseInsensitive);
        outDataSplitted[i] = tempData;

        // Refill writeParent->outData
        writeParent->outData.append(outDataSplitted[i] + "\n");

        // Show progress
        fI = i;
        fLen = outDataSplitted.length();
        percDone = ((fI + 1) / fLen) * 100;
        ui->progBar->setValue(percDone);

        ui->carLbl->show();
        fPercDone = (fI + 1) / fLen; // Can't devide 0
        carPos = 800 * fPercDone;
        ui->carLbl->setGeometry(carPos,
                                  ui->carLbl->geometry().y(),
                                  ui->carLbl->geometry().width(),
                                  ui->carLbl->geometry().height());
    } // End of for-loop

    // Hide the car after saving
    ui->carLbl->hide();

    // Add 'generated by' note
    writeParent->outData.prepend("\t\t<font color='#A6A6A6'><i>(This file is generated by Config Reader)</i></font><br /><br />\n");
}





void writeWindow::outputDiff() {
    // Copy the contents of the files to new strings, keep originals
    fileCopy1 = writeParent->fileCont1;
    fileCopy2 = writeParent->fileCont2;

    // Remove all preceding spaces for keys, example: "    /dir/dir/key..."
    while(fileCopy1.contains("\t\t&nbsp;"))
        fileCopy1.replace("\t\t&nbsp;", "\t\t", Qt::CaseSensitive);
    while(fileCopy2.contains("\t\t&nbsp;"))
        fileCopy2.replace("\t\t&nbsp;", "\t\t", Qt::CaseSensitive);
    // Remove all preceding spaces for directories, example: "<font><b>   /dir/dir"
    while(fileCopy1.contains("<b>&nbsp;"))
        fileCopy1.replace("<b>&nbsp;", "<b>", Qt::CaseSensitive);
    while(fileCopy2.contains("<b>&nbsp;"))
        fileCopy2.replace("<b>&nbsp;", "<b>", Qt::CaseSensitive);

    // Replace all other &nbsp;s with normal spaces, example: "key = value"
    fileCopy1.replace("&nbsp;", " ", Qt::CaseSensitive);
    fileCopy2.replace("&nbsp;", " ", Qt::CaseSensitive);

    // Replace big directory formatting with directory formatting (main dirs like "/apps" are bigger + different blue)
    fileCopy1.replace("<font size='+5' color='#0000FF'>", "<font color='#4D4DFF'>", Qt::CaseSensitive);
    fileCopy2.replace("<font size='+5' color='#0000FF'>", "<font color='#4D4DFF'>", Qt::CaseSensitive);

    /************* fileCopy's are now equal (except for differences, of course) ***************/

    // Search the indexposition of <body
    // Remove all content up to the <body> tag (focus on content only)
    // (only when it exists in the file, returns -1 when not found)
    bodyTagPos = fileCopy1.indexOf("\t<body", 0, Qt::CaseSensitive);
    if(bodyTagPos >= 0)
        fileCopy1.remove(0, bodyTagPos);

    bodyTagPos = fileCopy2.indexOf("\t<body", 0, Qt::CaseSensitive);
    if(bodyTagPos >= 0)
        fileCopy2.remove(0, bodyTagPos);

    // Make fileContSplitted's, split fileCopy's by "\n"
    fileContSplitted1 = fileCopy1.split("\n", QString::KeepEmptyParts, Qt::CaseInsensitive);
    fileContSplitted2 = fileCopy2.split("\n", QString::KeepEmptyParts, Qt::CaseInsensitive);

            // Check both ways
            sourceNum = first;
            saveDifference(fileContSplitted1, fileContSplitted2);
            fileDiffList.append("\t\t<br />"); // Create an empty line in between
            sourceNum = second;
            saveDifference(fileContSplitted2, fileContSplitted1);

    // Clear fileDiff before filling it with the differences
    fileDiff.clear();
    // Fill fileDiff with differences
    fileDiff = fileDiffList.join("\n");
    fileDiff.append("\n");

    // Remove all double newline characters
    while(fileDiff.contains("\n\n"))
        fileDiff.replace("\n\n", "\n", Qt::CaseInsensitive);

    // Show "(No differences)" when fileDiff is empty
    if(fileDiff == "\n")
        fileDiff = "\t\t<i>(No differences)</i>\n";

    // Prepend lines from last to first, because of prepending
    fileDiff.prepend("\t\t<hr /><br />\n");
    fileDiff.prepend("\t\t<font color='#0000FF'><b>File 2 =</b></font> " + QFileInfo(writeParent->fileWithPath2).absoluteFilePath() + "<br />\n");
    fileDiff.prepend("\t\t<font color='#0000FF'><b>File 1 =</b></font> " + QFileInfo(writeParent->fileWithPath1).absoluteFilePath() + "<br />\n");
    fileDiff.prepend("\t\t<font color='A6A6A6'><i>(This file is generated by Config Reader)</i></font><br /><br />\n");
}





void writeWindow::saveDifference(QStringList &source, QStringList &compare) {
    for(i=0; i<source.length(); i++) {
        // If the current line is a directory, store it temporarily
        preDirString = source[i];
        if(preDirString.startsWith("\t\t<font color='#4D4DFF'>"))
            dirString = preDirString;

        // Only edit and add when it doesn't exist in the compareFile
        if( ! compare.contains(source[i], Qt::CaseSensitive)) {
            // Store current entry in temporary string
            tempData = source[i];

            // If current entry is a directory:
            if(tempData.startsWith("\t\t<font color='#4D4DFF'>")) {
                // Remove ":" at the end
                tempData.replace(":</b>", "</b>", Qt::CaseSensitive);
                // Prepend the file number
                if(sourceNum == first)
                    tempData.replace("\t\t<font", "\t\tFile 1: <font", Qt::CaseSensitive);
                else if(sourceNum == second)
                    tempData.replace("\t\t<font", "\t\tFile 2: <font", Qt::CaseSensitive);

                // Store edited string in list again (same entry)
                source[i] = tempData;
                // Add the entry to fileDiffList
                fileDiffList.append(source[i]);
            }
            // If current entry is a key:
            else {
                // Make the key/value line indent 4 spaces (+ 8 spaces instead of "File #: ")
                // Also prepend the file number
                if(sourceNum == first) {
                    dirString.replace("\t\t<font", "\t\tFile 1:&nbsp;<font", Qt::CaseSensitive);
                    tempData.replace("\t\t", "\t\t&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;", Qt::CaseSensitive);
                }
                else if(sourceNum == second) {
                    dirString.replace("\t\t<font", "\t\tFile 2:&nbsp;<font", Qt::CaseSensitive);
                    tempData.replace("\t\t", "\t\t&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;", Qt::CaseSensitive);
                }

                // Store edited string in list again (same entry)
                source[i] = tempData;
                // Prepend the directory to the entry
                source[i] = dirString + "\n" + source[i];
                // Add the entry to fileDiffList
                fileDiffList.append(source[i]);
            }
        }

        // Show progress

        percDone = (sourceNum == second)? 50 : 0;
        fPercDone = (sourceNum == second)? 0.5 : 0;

        fI = i;
        fLen = source.length();
        percDone += ((fI + 1) / fLen) * 50; // 50% max, this function runs twice
        ui->progBar->setValue(percDone);

        // Show and move the car
        ui->carLbl->show();
        fPercDone += ((fI + 1) / fLen) / 2; // Can't devide 0 (fI + 1), 50% max (function runs twice)
        carPos = 800 * fPercDone;
        ui->carLbl->setGeometry(carPos,
                                  ui->carLbl->geometry().y(),
                                  ui->carLbl->geometry().width(),
                                  ui->carLbl->geometry().height());
    } // End of for loop

    // Hide the car after comparing
    ui->carLbl->hide();

    // Remove duplicate entries from list (because funtion runs twice)
    fileDiffList.removeDuplicates();
}





void writeWindow::openFile() {
    QDesktopServices::openUrl(QUrl("file://" + QFileInfo(saveFile).absoluteFilePath()));
}
