/*
* Copyright (C) 2008-2010 J-P Nurmi jpnurmi@gmail.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
* $Id$
*/

#include "sessiontabwidget.h"
#include "commandengine.h"
#include "application.h"
#include "messageview.h"
#include "settings.h"
#include "session.h"
#include <ircutil.h>
#include <irc.h>
#include <QtGui>

SessionTabWidget::SessionTabWidget(Session* session, QWidget* parent) :
    TabWidget(parent)
{
    // take ownership of the session
    session->setParent(this);

    connect(this, SIGNAL(currentChanged(int)), this, SLOT(tabActivated(int)));
    connect(session, SIGNAL(connected()), this, SLOT(connected()));
    connect(session, SIGNAL(reconnecting()), this, SLOT(reconnecting()));
    connect(session, SIGNAL(disconnected()), this, SLOT(disconnected()));
    connect(session, SIGNAL(bufferAdded(Irc::Buffer*)), this, SLOT(bufferAdded(Irc::Buffer*)));
    connect(session, SIGNAL(bufferRemoved(Irc::Buffer*)), this, SLOT(bufferRemoved(Irc::Buffer*)));

    QShortcut* shortcut = new QShortcut(QKeySequence::Close, this);
    connect(shortcut, SIGNAL(activated()), this, SLOT(closeCurrentView()));

    applySettings();

#if QT_VERSION >= 0x040600
    registerSwipeGestures(Qt::Horizontal);
#endif

    d.connectCounter = 0;
    d.session = session;
    d.engine = new CommandEngine(this);
    d.engine->setScriptObject("app", qApp);
    d.engine->setScriptObject("window", window());
    d.engine->setScriptObject("session", session);
    d.engine->setScriptObject("tabWidget", this);
}

Session* SessionTabWidget::session() const
{
    return d.session;
}

void SessionTabWidget::openView(const QString& receiver)
{
    if (d.views.contains(receiver.toLower()))
        setCurrentWidget(d.views.value(receiver.toLower()));
    else
        d.session->addBuffer(receiver);
}

void SessionTabWidget::closeCurrentView(const QString& message)
{
    int index = currentIndex();
    if (index != -1)
    {
        QString receiver = tabText(index);
        closeView(receiver, message);
    }
}

void SessionTabWidget::closeView(const QString& receiver, const QString& message)
{
    QString tmp = receiver;
    if (!Session::isChannel(tmp) && !d.views.contains(tmp.toLower()))
        tmp.prepend("#");

    if (Session::isChannel(tmp))
        d.session->part(tmp, message);

    MessageView* view = d.views.take(tmp.toLower());
    if (view)
    {
        if (indexOf(view) == 0)
        {
            // closing a server tab
            emit disconnectFrom(QString());
            deleteLater();
        }
        view->deleteLater();
    }
}

void SessionTabWidget::connected()
{
    MessageView* view = static_cast<MessageView*>(widget(0));
    if (view)
    {
        QStringList params;
        params << d.session->host() << tr("Connected");
        view->receiveMessage(tr("[%1] %2"), params);
        highlightTab(view, true);
    }
    d.connectCounter = 0;
}

void SessionTabWidget::reconnecting()
{
    MessageView* view = static_cast<MessageView*>(widget(0));
    if (view)
    {
        QStringList params;
        params << d.session->host() << tr("Connecting... #%1").arg(++d.connectCounter);
        view->receiveMessage(tr("[%1] %2"), params);
        highlightTab(view, true);
    }
}

void SessionTabWidget::disconnected()
{
    MessageView* view = static_cast<MessageView*>(widget(0));
    if (view)
    {
        QStringList params;
        params << d.session->host() << tr("Disconnected");
        view->receiveMessage(tr("[%1] %2"), params);
        highlightTab(view, true);
    }
}

void SessionTabWidget::bufferAdded(Irc::Buffer* buffer)
{
    MessageView* view = new MessageView(buffer, this);
    connect(view, SIGNAL(send(QString, QString)), this, SLOT(send(QString, QString)));
    connect(view, SIGNAL(rename(MessageView*)), this, SLOT(nameTab(MessageView*)));
    connect(view, SIGNAL(alert(MessageView*, bool)), this, SLOT(alertTab(MessageView*, bool)));
    connect(view, SIGNAL(highlight(MessageView*, bool)), this, SLOT(highlightTab(MessageView*, bool)));
    connect(view, SIGNAL(query(QString)), this, SLOT(openView(QString)));

    d.views.insert(buffer->receiver().toLower(), view);
    int index = addTab(view, buffer->receiver());
    setCurrentIndex(index);
}

void SessionTabWidget::bufferRemoved(Irc::Buffer* buffer)
{
    MessageView* view = d.views.take(buffer->receiver().toLower());
    if (view)
        view->deleteLater();
}

void SessionTabWidget::send(const QString& receiver, const QString& message)
{
    if (message.trimmed().isEmpty())
        return;

    MessageView* view = static_cast<MessageView*>(currentWidget());
    if (message.startsWith('/'))
    {
        if (!d.engine->evaluateCommand(message.mid(1), receiver, d.views.keys()))
        {
            if (view)
                view->showHelp(message.mid(1));
        }
    }
    else if (indexOf(view) > 0)
    {
        d.session->message(receiver, message);
    }
    else
    {
        if (view)
            view->showHelp();
    }
}

void SessionTabWidget::tabActivated(int index)
{
    MessageView* view = static_cast<MessageView*>(currentWidget());
    if (view)
    {
        d.engine->setScriptObject("messageView", view);
        d.session->setDefaultBuffer(view->buffer());
    }

    setTabAlert(index, false);
    setTabHighlight(index, false);
    if (isVisible())
    {
        window()->setWindowFilePath(tabText(index));
        if (view)
            view->setFocus();
    }
}

void SessionTabWidget::delayedTabReset()
{
    d.delayedIndexes += currentIndex();
    QTimer::singleShot(500, this, SLOT(delayedTabResetTimeout()));
}

void SessionTabWidget::delayedTabResetTimeout()
{
    if (d.delayedIndexes.isEmpty())
        return;

    int index = d.delayedIndexes.takeFirst();
    tabActivated(index);
}

void SessionTabWidget::nameTab(MessageView* view)
{
    int index = indexOf(view);
    if (index != -1)
    {
        d.views.remove(tabText(index));
        d.views.insert(view->buffer()->receiver(), view);

        setTabText(index, view->buffer()->receiver());
        if (index == 0)
            emit titleChanged(view->buffer()->receiver());
    }
}

void SessionTabWidget::alertTab(MessageView* view, bool on)
{
    int index = indexOf(view);
    if (index != -1)
    {
        if (!isVisible() || index != currentIndex())
            setTabAlert(index, on);
        emit vibraRequested(on);
    }
}

void SessionTabWidget::highlightTab(MessageView* view, bool on)
{
    int index = indexOf(view);
    if (index != -1)
    {
        if (!isVisible() || index != currentIndex())
            setTabHighlight(index, on);
    }
}

void SessionTabWidget::applySettings()
{
    Settings settings = Application::settings();
    setAlertColor(QColor(settings.colors.value(Settings::Highlight)));
    setHighlightColor(QColor(settings.colors.value(Settings::Highlight)));
}
