/*
* Copyright (C) 2008-2010 J-P Nurmi jpnurmi@gmail.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
* $Id$
*/

#include "application.h"
#include "commandxmlreader.h"
#include "settingswizard.h"
#include "sharedtimer.h"
#include <QDesktopServices>
#include <QMessageBox>
#include <QSettings>
#include <QDebug>
#include <QIcon>
#include <QFile>
#include <QDir>

Settings Application::ApplicationData::settings;
Connections Application::ApplicationData::connections;
Commands Application::ApplicationData::commands;

Application::Application(int& argc, char* argv[]) : QApplication(argc, argv)
{
    setApplicationName("Communi");
    setApplicationVersion(VERSION);
    setOrganizationName("communi.bitbucket.org");
    setOrganizationDomain("http://communi.bitbucket.org");

    QIcon icon;
    icon.addFile(":/resources/icon-16.png");
    icon.addFile(":/resources/icon-32.png");
    icon.addFile(":/resources/icon-64.png");
    icon.addFile(":/resources/icon-128.png");
    setWindowIcon(icon);

    qRegisterMetaTypeStreamOperators<Settings>("Settings");
    qRegisterMetaTypeStreamOperators<Connection>("Connection");
    qRegisterMetaTypeStreamOperators<Connections>("Connections");

    QSettings settings;
    ApplicationData::settings = settings.value("settings").value<Settings>();
    ApplicationData::connections = settings.value("connections").value<Connections>();

    QDir dataDir(QDesktopServices::storageLocation(QDesktopServices::DataLocation));
    if (!dataDir.exists())
        dataDir.mkpath(dataDir.path());
    if (!dataDir.exists("commands.xml"))
        QFile::copy(":/resources/commands.xml", dataDir.filePath("commands.xml"));

    //QFile commandFile(dataDir.filePath("commands.xml"));
    QFile commandFile(":/resources/commands.xml");
    commandFile.open(QIODevice::ReadOnly);

    CommandXmlReader commandReader;
    ApplicationData::commands = commandReader.read(&commandFile);

    SharedTimer::instance()->setInterval(500);
}

Application::~Application()
{
    QSettings settings;
    settings.setValue("settings", ApplicationData::settings);
    settings.setValue("connections", QVariant::fromValue(ApplicationData::connections));

    //CommandXmlWriter commandWriter;
    //...
}

QString Application::applicationSlogan()
{
    return tr("%1 is a simple and light-weight cross-platform IRC client.").arg(applicationName());
}

Settings Application::settings()
{
    return ApplicationData::settings;
}

void Application::setSettings(const Settings& settings)
{
    if (ApplicationData::settings != settings)
    {
        ApplicationData::settings = settings;
        QMetaObject::invokeMethod(qApp, "settingsChanged", Q_ARG(Settings, settings));
    }
}

Connections Application::connections()
{
    return ApplicationData::connections;
}

void Application::setConnections(const Connections& connections)
{
    if (ApplicationData::connections != connections)
    {
        ApplicationData::connections = connections;
        QMetaObject::invokeMethod(qApp, "connectionsChanged", Q_ARG(Connections, connections));
    }
}

void Application::setConnection(int index, const Connection& connection)
{
    Q_ASSERT(index >= 0 && index < ApplicationData::connections.count());
    ApplicationData::connections[index] = connection;
    QMetaObject::invokeMethod(qApp, "connectionChanged", Q_ARG(int, index), Q_ARG(Connection, connection));
    QMetaObject::invokeMethod(qApp, "connectionsChanged", Q_ARG(Connections, ApplicationData::connections));
}

Commands Application::commands()
{
    return ApplicationData::commands;
}

void Application::setCommands(const Commands& commands)
{
    if (ApplicationData::commands != commands)
    {
        ApplicationData::commands = commands;
        QMetaObject::invokeMethod(qApp, "commandsChanged", Q_ARG(Commands, commands));
    }
}

void Application::aboutApplication()
{
    QMessageBox::about(
#ifdef Q_WS_MAC
        0,
#else
        activeWindow(),
#endif // Q_WS_MAC
        applicationName(), QString(
        "<h3>%1</h3>"
        "<p>%2</p>"
        "<p><a href='%3'>%3</a></p>"
        "<p><small>Copyright (C) 2008-2010 J-P Nurmi &lt;"
        "<a href='mailto:jpnurmi@gmail.com'>jpnurmi@gmail.com</a>&gt;</small></p>")
            .arg(tr("About %1").arg(applicationName()))
            .arg(applicationSlogan())
            .arg(organizationDomain())
        );
}

void Application::aboutOxygen()
{
    QMessageBox::about(
#ifdef Q_WS_MAC
        0,
#else
        activeWindow(),
#endif // Q_WS_MAC
        "Oxygen", QString(
        "<h3>%1</h3>"
        "<p><a href='http://www.oxygen-icons.org/'>http://www.oxygen-icons.org/</a></p>"
        ).arg(tr("About %1").arg("Oxygen")));
}

void Application::showSettings()
{
    SettingsWizard wizard(activeWindow());
    wizard.setSettings(settings());
    if (wizard.exec())
        setSettings(wizard.settings());
}
