#include "ColumbusTrackView.h"

#include <QtGui>
#include <rx/widgets/LCDIndicator.h>

#include "ui/widgets/TrackView.h"
#include "ui/widgets/CompassDial.h"

class ColumbusTrackViewPrivate
{
public:
    ApplicationController *controller;

    QPropertyAnimation *zoomAnimation;
    QPropertyAnimation *rotationAnimation;

    QHBoxLayout *hbox;

    TrackView *trackView;
    LCDIndicator *speedIndicator;
    LCDIndicator *headingIndicator;
    AbstractCompass *compass;

    QLabel *latitude;
    QLabel *longitude;
    QLabel *altitude;
    QLabel *tripDistance;

    QLabel *speedMode;
};

ColumbusTrackView::ColumbusTrackView(QWidget *parent)
    : ApplicationView(parent)
{
    this->d = new ColumbusTrackViewPrivate;

    QVBoxLayout *layout = new QVBoxLayout(this);
    QHBoxLayout *hboxt  = new QHBoxLayout();
    QHBoxLayout *hboxb  = new QHBoxLayout();

    QFormLayout *posl   = new QFormLayout();

    d->controller = NULL;

    posl->setLabelAlignment(Qt::AlignRight | Qt::AlignVCenter);

    layout->setContentsMargins(0, 0, 0, 0);
    hboxb->setContentsMargins(0, 0, 0, 0);

    d->trackView = new TrackView(this);
    d->trackView->setGeometry(this->rect());

    d->speedIndicator    = new LCDIndicator(this);
    d->headingIndicator  = new LCDIndicator(this);
    d->speedMode         = new QLabel("", this);
    d->speedMode->setFont(QFont(d->speedMode->font().family(), 12));

    d->speedIndicator->setFixedWidth(170);

    d->headingIndicator->setFixedWidth(170);
    d->headingIndicator->setUnitIndicator("deg");

    hboxb->addWidget(d->headingIndicator, 0, Qt::AlignLeft | Qt::AlignBottom);
    hboxb->addWidget(d->speedMode, 1, Qt::AlignRight | Qt::AlignTop);
    hboxb->addWidget(d->speedIndicator, 0, Qt::AlignRight | Qt::AlignBottom);

    d->latitude      = ValueLabel("-");
    d->longitude     = ValueLabel("-");
    d->altitude      = ValueLabel("-");
    d->tripDistance  = ValueLabel("-");

    d->altitude->hide();
    d->tripDistance->hide();

    posl->addRow(tr("Latitude"), d->latitude);
    posl->addRow(tr("Longitude"), d->longitude);
    //posl->addRow(tr("Altitude"), d->altitude);
    //posl->addRow(tr("Distance"), d->tripDistance);

    d->compass = new CompassDial(this);
    d->compass->setFixedWidth(164);

    d->latitude->raise();
    d->longitude->raise();

    hboxt->addLayout(posl, 0);
    hboxt->addWidget(d->compass, 0, Qt::AlignRight | Qt::AlignTop);

    layout->addSpacing(10);
    layout->addLayout(hboxt);
    layout->addSpacerItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding));
    layout->addLayout(hboxb);

    d->hbox = hboxb;

    d->zoomAnimation = new QPropertyAnimation(d->trackView, "zoom", this);
    d->rotationAnimation = new QPropertyAnimation(d->compass, "heading", this);

    QObject::connect(d->compass, SIGNAL(headingChanged(double)), d->trackView, SLOT(setRotation(double)));

    d->zoomAnimation->setDuration(500);
    d->zoomAnimation->setEasingCurve(QEasingCurve::InOutQuad);

    d->rotationAnimation->setDuration(750);
    d->rotationAnimation->setEasingCurve(QEasingCurve::InOutQuad);

    this->setFocusPolicy(Qt::StrongFocus);
    d->speedIndicator->installEventFilter(this);
}

ColumbusTrackView::~ColumbusTrackView()
{
    delete this->d;
}


QString ColumbusTrackView::friendlyName() const
{
    static const QString friendlyName = QString("Track");
    return friendlyName;
}

void ColumbusTrackView::setController(ApplicationController *controller)
{
    if(d->controller != NULL)
    {
        QObject::disconnect(d->controller->model(), SIGNAL(distanceUpdated(qreal)), this, SLOT(onDistanceUpdated(qreal)));
        QObject::disconnect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)), this, SLOT(onPositionUpdated(QGeoPositionInfo)));
    }

    d->controller = controller;

    if(d->controller != NULL)
    {
        if(d->controller->model() != NULL)
        {
            d->trackView->setModel(d->controller->model());
        }

        if(d->controller->waypoints() != NULL)
        {
            d->trackView->setWaypoints(d->controller->waypoints());
        }

        d->speedIndicator->setUnitIndicator(d->controller->model()->speedUnits());

        QObject::connect(d->controller->model(), SIGNAL(distanceUpdated(qreal)),
                         this, SLOT(onDistanceUpdated(qreal)));
        QObject::connect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                         this, SLOT(onPositionUpdated(QGeoPositionInfo)));
    }
}

void ColumbusTrackView::cycleSpeedIndicator()
{
    static const QStringList modeOpts = (QStringList() << "" << "Avg." << "Max.");
    static int nextMode = 1;

    if(nextMode >= 3) nextMode = 0;
    d->speedMode->setText(modeOpts.value(nextMode));

    switch(nextMode)
    {
    case 0:
        d->speedIndicator->setValue(d->controller->model()->currentSpeed());
        break;
    case 1:
        d->speedIndicator->setValue(d->controller->model()->averageSpeed());
        break;
    case 2:
        d->speedIndicator->setValue(d->controller->model()->maximumSpeed());
        break;
    }

    nextMode++;
}

void ColumbusTrackView::onDistanceUpdated(qreal distance)
{
    this->d->tripDistance->setText(QString::number(distance, 'f', 2) + " <sub>" + d->controller->model()->distanceUnits() + "</sub>");
}

void ColumbusTrackView::onPositionUpdated(const QGeoPositionInfo &posinfo)
{
    const ColumbusModel *model = d->controller->model();

    QString latitude = model->currentLatitude();
    QString longitude = model->currentLongitude();
    qreal direction = model->currentHeading();

    d->latitude->setText(latitude);
    d->longitude->setText(longitude);

    d->altitude->setText(QString::number(model->currentAltitude(), 'f', 2)
                        + " <sub>" + model->altitudeUnits() + "</sub>");

    d->headingIndicator->setValue(direction);

    d->speedIndicator->setUnitIndicator(model->speedUnits());
    if(d->speedMode->text() == "")
    {
        d->speedIndicator->setValue(model->currentSpeed());
    }
    else if(d->speedMode->text() == "Avg.")
    {
        d->speedIndicator->setValue(model->averageSpeed());
    }
    else if(d->speedMode->text() == "Max.")
    {
        d->speedIndicator->setValue(model->maximumSpeed());
    }

    if(d->controller->waypoints()->destination().isValid())
    {
        ColumbusWaypoint waypoint = d->controller->waypoints()->waypoint(d->controller->waypoints()->destination());
        d->compass->setBearing(posinfo.coordinate().azimuthTo(waypoint.coordinate()));

        if(!d->compass->isBearingShown()) d->compass->setBearingShown(true);
    }
    else
    {
        if(d->compass->isBearingShown()) d->compass->setBearingShown(false);
    }

    qreal nrot = direction;
    qreal crot = d->rotationAnimation->targetObject()->property(d->rotationAnimation->propertyName()).toReal();

    if(nrot != crot)
    {
        if(d->rotationAnimation->state() == QPropertyAnimation::Running) d->rotationAnimation->stop();

        d->rotationAnimation->setStartValue(crot);
        d->rotationAnimation->setEndValue(nrot);
        d->rotationAnimation->start();
    }

    d->trackView->update();
}

QLabel* ColumbusTrackView::ValueLabel(const QString &text) const
{
    QLabel *result = new QLabel(text, (QWidget*)this);
    QPalette palette = result->palette();
    QFont font;
    font.setPixelSize(24);

    palette.setColor(result->foregroundRole(), QColor(0xff, 0xff, 0xff));

    result->setAlignment(Qt::AlignLeft);
    result->setFont(font);
    result->setPalette(palette);

    return result;
}

void ColumbusTrackView::keyPressEvent(QKeyEvent *event)
{
    qreal cZoom = d->zoomAnimation->targetObject()->property(d->zoomAnimation->propertyName()).toReal();

    switch(event->key())
    {
    case Qt::Key_F7:
        if(d->zoomAnimation->state() == QPropertyAnimation::Running) d->zoomAnimation->stop();

        d->zoomAnimation->setStartValue(cZoom);
        d->zoomAnimation->setEndValue(cZoom + 5);
        d->zoomAnimation->start();
        break;

    case Qt::Key_F8:
        if(d->zoomAnimation->state() == QPropertyAnimation::Running) d->zoomAnimation->stop();

        d->zoomAnimation->setStartValue(cZoom);
        d->zoomAnimation->setEndValue(cZoom - 5);
        d->zoomAnimation->start();
        break;

    default:
        break;
    }

    QWidget::keyPressEvent(event);
}

void ColumbusTrackView::resizeEvent(QResizeEvent *event)
{
    d->trackView->setGeometry(this->rect());

    ApplicationView::resizeEvent(event);
}

bool ColumbusTrackView::eventFilter(QObject *object, QEvent *event)
{
    if(object == d->speedIndicator && event->type() == QEvent::MouseButtonPress)
    {
        this->cycleSpeedIndicator();
    }

    return QWidget::eventFilter(object, event);
}
