#include <QtGui>
#include <QtDBus>

#include "config.h"
#include <rx/columbus/config.h>

#include <rx/widgets/Maemo5TaskButton.h>
#include <rx/widgets/Maemo5MenuButton.h>
#include <rx/widgets/Maemo5FullScreenButton.h>

#include "ColumbusStatusView.h"
#include "ColumbusStatisticsView.h"
#include "ColumbusTrackView.h"

#include "ColumbusDialogWaypoints.h"
#include "ColumbusDialogLogging.h"
#include "ColumbusDialogOptions.h"
#include "ColumbusDialogTools.h"

#include "MainWindow.h"

#include <X11/Xlib.h>
#include <X11/Xatom.h>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    m_ScreenSaver               = NULL;
    m_Controller                = NULL;
    m_ControllerSuspended       = NULL;
    m_ViewsActionGroup          = NULL;

    m_ScreenLockState           = "unlocked";

    m_WaypointsDialog           = new ColumbusDialogWaypoints(this);
    m_LoggingDialog             = new ColumbusDialogLogging(this);
    m_OptionsDialog             = new ColumbusDialogOptions(this);
    m_ToolsDialog               = new ColumbusDialogTools(this);

    m_ControlsAnimationGroup    = new QParallelAnimationGroup(this);

    QToolButton *tBtn           = new Maemo5TaskButton(this);
    QToolButton *mBtn           = new Maemo5MenuButton(this);
    QToolButton *fBtn           = new Maemo5FullScreenButton(this);

    QPropertyAnimation *tAnim   = new QPropertyAnimation(tBtn, "geometry", this);
    QPropertyAnimation *mAnim   = new QPropertyAnimation(mBtn, "geometry", this);
    QPropertyAnimation *fAnim   = new QPropertyAnimation(fBtn, "geometry", this);

    tAnim->setDuration(750);
    mAnim->setDuration(tAnim->duration());
    fAnim->setDuration(tAnim->duration());

    m_ControlsAnimationGroup->addAnimation(tAnim);
    m_ControlsAnimationGroup->addAnimation(mAnim);
    m_ControlsAnimationGroup->addAnimation(fAnim);

    this->setWindowTitle(tr("Columbus"));
    this->setAutoFillBackground(true);

    this->setupPalettes();

    this->installEventFilter(this);

    QTimer::singleShot(1000, this, SLOT(hideFullScreenControls()));

    tBtn->raise();
    mBtn->raise();
    fBtn->raise();

    QObject::connect(m_ControlsAnimationGroup, SIGNAL(finished()), this, SLOT(onAnimationFinished()));

#if defined(Q_WS_MAEMO_5)
    QObject::connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(resized()));
#endif

    this->setupX11Keys();

    QDBusConnection::systemBus().connect("",
                                         "/com/nokia/mce/signal",
                                         "com.nokia.mce.signal",
                                         "tklock_mode_ind",
                                         this,
                                         SLOT(onScreenLockSignal(QString)));
}

void MainWindow::setScreenSaverInhibit(bool on)
{
    if(m_ScreenSaver == NULL && on)
    {
        qDebug() << "MainWindow: Enabling screen saver inhibit.";
        m_ScreenSaver = new QSystemScreenSaver(this);
        m_ScreenSaver->setScreenSaverInhibit();
    }
    else if(m_ScreenSaver != NULL && !on)
    {
        qDebug() << "MainWindow: Disabling screen saver inhibit.";
        delete m_ScreenSaver;
        m_ScreenSaver = NULL;
    }
}

void MainWindow::setOrientationAuto(bool on)
{
#if defined(Q_WS_MAEMO_5)
    this->setAttribute(Qt::WA_Maemo5AutoOrientation, on);
#endif
}

void MainWindow::setOrientationLandscape(bool on)
{
#if defined(Q_WS_MAEMO_5)
    this->setAttribute(Qt::WA_Maemo5LandscapeOrientation, on);
#endif
}

void MainWindow::setOrientationPortrait(bool on)
{
#if defined(Q_WS_MAEMO_5)
    this->setAttribute(Qt::WA_Maemo5PortraitOrientation, on);
#endif
}

void MainWindow::setupX11Keys()
{
#if defined(Q_WS_MAEMO_5)
    unsigned char value = 1;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", false);

    XChangeProperty(QX11Info::display(),
                    this->winId(),
                    atom,
                    XA_INTEGER,
                    32,
                    PropModeReplace,
                    (unsigned char*) &value,
                    1);
#endif
}

void MainWindow::resume()
{
    // Don't resume the UI if the screen is locked or we're not in focus.
    if(m_ScreenLockState != "locked" && this->isActiveWindow())
    {
        qDebug() << "MainWindow: Resuming UI updates.";
        this->initialize(m_ControllerSuspended);
        emit this->uiResumed();
    }
}

void MainWindow::suspend()
{
    qDebug() << "MainWindow: Suspending UI updates.";
    this->initialize(NULL);
    emit this->uiSuspended();
}

void MainWindow::showFullScreenControls()
{
    QPropertyAnimation *tAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(0));
    QPropertyAnimation *mAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(1));
    QPropertyAnimation *fAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(2));

    QWidget *tBtn = qobject_cast<QWidget*>(tAnim->targetObject());
    QWidget *mBtn = qobject_cast<QWidget*>(mAnim->targetObject());
    QWidget *fBtn = qobject_cast<QWidget*>(fAnim->targetObject());

    if(m_ControlsAnimationGroup->state() == QPropertyAnimation::Running) return;

    tAnim->setEasingCurve(QEasingCurve::OutBounce);
    mAnim->setEasingCurve(tAnim->easingCurve());
    fAnim->setEasingCurve(tAnim->easingCurve());

    tAnim->setStartValue(tBtn->geometry());
    tAnim->setEndValue(QRect(0, 0,
                             tBtn->width(), tBtn->height()));

    mAnim->setStartValue(mBtn->geometry());
    mAnim->setEndValue(QRect(this->width() - mBtn->width(), 0,
                             mBtn->width(), mBtn->height()));

    fAnim->setStartValue(fBtn->geometry());
    fAnim->setEndValue(QRect(this->width() - mBtn->width(),
                             this->height() - mBtn->height(),
                             fBtn->width(), fBtn->height()));

    m_ControlsAnimationGroup->start();
}

void MainWindow::hideFullScreenControls()
{
    QPropertyAnimation *tAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(0));
    QPropertyAnimation *mAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(1));
    QPropertyAnimation *fAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(2));

    QWidget *tBtn = qobject_cast<QWidget*>(tAnim->targetObject());
    QWidget *mBtn = qobject_cast<QWidget*>(mAnim->targetObject());
    QWidget *fBtn = qobject_cast<QWidget*>(fAnim->targetObject());

    if(m_ControlsAnimationGroup->state() == QPropertyAnimation::Running) return;

    tAnim->setEasingCurve(QEasingCurve::InQuad);
    mAnim->setEasingCurve(tAnim->easingCurve());
    fAnim->setEasingCurve(tAnim->easingCurve());

    tAnim->setStartValue(tBtn->geometry());
    tAnim->setEndValue(QRect(0, -tBtn->height(),
                             tBtn->width(), tBtn->height()));

    mAnim->setStartValue(mBtn->geometry());
    mAnim->setEndValue(QRect(this->width() - mBtn->width(), -mBtn->height(),
                             mBtn->width(), mBtn->height()));

    fAnim->setStartValue(fBtn->geometry());
    fAnim->setEndValue(QRect(this->width() - mBtn->width(),
                             this->height(),
                             mBtn->width(), mBtn->height()));

    m_ControlsAnimationGroup->start();
}

void MainWindow::onAnimationFinished()
{
}

void MainWindow::resized()
{
    QPropertyAnimation *tAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(0));
    QPropertyAnimation *mAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(1));
    QPropertyAnimation *fAnim = qobject_cast<QPropertyAnimation*>(m_ControlsAnimationGroup->animationAt(2));

    QWidget *tBtn = qobject_cast<QWidget*>(tAnim->targetObject());
    QWidget *mBtn = qobject_cast<QWidget*>(mAnim->targetObject());
    QWidget *fBtn = qobject_cast<QWidget*>(fAnim->targetObject());

    const QDesktopWidget *desktop = QApplication::desktop();

    QPalette palette = this->palette();
    if(desktop->width() > desktop->height())
    {
        palette.setBrush(this->backgroundRole(), QBrush(QImage(":/MainWindowBG.png")));

#if defined(Q_WS_MAEMO_5)
        tBtn->move(0, -tBtn->height());
        mBtn->move(800 - mBtn->width(), -mBtn->height());
        fBtn->move(800 - fBtn->width(), 480);
#endif

        emit this->onLandscape();
    }
    else
    {
        palette.setBrush(this->backgroundRole(), QBrush(QImage(":/MainWindowBG_Portrait.png")));

#if defined(Q_WS_MAEMO_5)
        tBtn->move(0, -tBtn->height());
        mBtn->move(480 - mBtn->width(), -mBtn->height());
        fBtn->move(480 - fBtn->width(), 800);
#endif

        emit this->onPortrait();
    }

    this->setPalette(palette);
    this->hideFullScreenControls();
}

bool MainWindow::eventFilter(QObject *object, QEvent *event)
{
    switch(event->type())
    {
    case QEvent::MouseButtonPress:
        break;

    case QEvent::MouseButtonRelease:
        this->showFullScreenControls();
        QTimer::singleShot(4000, this, SLOT(hideFullScreenControls()));
        break;

    case QEvent::WindowActivate:
        qDebug() << "MainWindow: Window activate event.";
        if(this->m_ControllerSuspended != NULL) this->resume();
        break;

    case QEvent::WindowDeactivate:
        qDebug() << "MainWindow: Window deactivate event.";
        if(this->m_ControllerSuspended != NULL)
        {
            this->setScreenSaverInhibit(false);
            this->suspend();
        }
        break;

    default:
        break;
    }

    return QMainWindow::eventFilter(object, event);
}

void MainWindow::onScreenLockSignal(const QString &state)
{
    if(m_ScreenLockState != state)
    {
        m_ScreenLockState = state;

        if(state == "unlocked")
        {
            qDebug() << "MainWindow: Screen unlocked";
            this->resume();
        }
        else if(state == "locked")
        {
            qDebug() << "MainWindow: Screen locked.";
            this->setScreenSaverInhibit(false);
            this->suspend();
        }
    }
}

void MainWindow::onConfigUpdated(const QString &key, QVariant value)
{
    if(key == CONFIG_KEY_DISPLAY_ORIENTATION)
    {
        DisplayOrientation orientation = (DisplayOrientation)m_ControllerSuspended->config()->valueIndex(key);

        this->setOrientationAuto(orientation        == DISPLAY_ORIENTATION_AUTOMATIC);
        this->setOrientationLandscape(orientation   == DISPLAY_ORIENTATION_LANDSCAPE);
        this->setOrientationPortrait(orientation    == DISPLAY_ORIENTATION_PORTRAIT);
    }
    else if(key == CONFIG_KEY_DISPLAY_KEEP_ALIVE)
    {
        this->setScreenSaverInhibit(value.toBool());
    }
}

void MainWindow::setupMenu()
{
    QAction *gsAction = NULL;

    m_ViewsActionGroup = new QActionGroup(this);
    m_ViewsActionGroup->setExclusive(true);

    foreach(ApplicationView* view, m_ApplicationViewSwitcher->views())
    {
        gsAction = new QAction(view->friendlyName(), m_ViewsActionGroup);
        gsAction->setData(m_ApplicationViewSwitcher->views().indexOf(view));
        gsAction->setCheckable(true);
    }

    m_ViewsActionGroup->actions().at(0)->setChecked(true);
    this->menuBar()->addActions(m_ViewsActionGroup->actions());

    this->menuBar()->addAction(tr("Waypoints"), m_WaypointsDialog, SLOT(showMaximized()));
    this->menuBar()->addAction(tr("Logging"), m_LoggingDialog, SLOT(show()));
    this->menuBar()->addAction(tr("Options"), m_OptionsDialog, SLOT(show()));
    this->menuBar()->addAction(tr("Tools"), m_ToolsDialog, SLOT(show()));

    QObject::connect(m_ViewsActionGroup, SIGNAL(selected(QAction*)),
                     this, SLOT(onViewsActionSelected(QAction*)));
}

void MainWindow::setupViews()
{
    m_ApplicationViewSwitcher = new ApplicationViewSwitcher(this);

    m_ApplicationViewSwitcher->addView(new ColumbusStatusView(this));
    m_ApplicationViewSwitcher->addView(new ColumbusTrackView(this));
    m_ApplicationViewSwitcher->addView(new ColumbusStatisticsView(this));

    this->setCentralWidget(m_ApplicationViewSwitcher);
}

void MainWindow::setupPalettes()
{
    m_Theme = this->palette();
    m_Theme.setColor(QPalette::Highlight, QColor(0xff, 0xff, 0xff));
    m_Theme.setColor(this->foregroundRole(), QColor(0x6f, 0xff, 0xff));
    m_Theme.setBrush(this->backgroundRole(), QBrush(QPixmap(":/MainWindowBG.png")));

    this->setAutoFillBackground(true);
}

void MainWindow::onViewChanged(int view)
{
    m_ViewsActionGroup->actions().at(view)->setChecked(true);
    this->hideFullScreenControls();
}

void MainWindow::onViewsActionSelected(QAction *action)
{
    m_ApplicationViewSwitcher->slideTo(action->data().toInt());
}

void MainWindow::initialize(ApplicationController *controller)
{
    if(controller != NULL && m_ControllerSuspended == NULL) // On first initialization.
    {
        if(m_ViewsActionGroup == NULL)
        {
            this->setupViews();
            this->setupMenu();

            this->setPalette(m_Theme);
        }

        m_ControllerSuspended = controller;

        m_WaypointsDialog->setController(controller);
        m_LoggingDialog->setController(controller);
        m_OptionsDialog->setController(controller);
        m_ToolsDialog->setController(controller);

        QObject::connect(m_ApplicationViewSwitcher, SIGNAL(viewChanged(int)), this, SLOT(onViewChanged(int)));
        QObject::connect(controller->config(), SIGNAL(updated(QString,QVariant)), this, SLOT(onConfigUpdated(QString,QVariant)));
    }

    m_Controller = controller;

    if(controller != NULL)
    {
        this->onConfigUpdated(CONFIG_KEY_DISPLAY_ORIENTATION,
                              controller->config()->value(CONFIG_KEY_DISPLAY_ORIENTATION));
        this->onConfigUpdated(CONFIG_KEY_DISPLAY_KEEP_ALIVE,
                              controller->config()->value(CONFIG_KEY_DISPLAY_KEEP_ALIVE));
    }

    foreach(ApplicationView *view, m_ApplicationViewSwitcher->views())
    {
        view->setController(controller);
    }
}
