#include "ColumbusStatisticsView.h"

#include "widgets/SpeedGraph.h"
#include "widgets/AltitudeProfileGraph.h"
#include "widgets/AccuracyGraph.h"

#include <QtGui>

class ColumbusStatisticsViewPrivate
{
public:
    ApplicationController *controller;

    QFormLayout *details;

    QLabel *distance;
    QLabel *averageSpeed;
    QLabel *maximumSpeed;
    QLabel *minimumAltitude;
    QLabel *maximumAltitude;
    QLabel *travelTime;
    QLabel *elapsedTime;

    SpeedGraph *speedGraph;
    AltitudeProfileGraph *altitudeProfile;
    AccuracyGraph *accuracyGraph;

    QWidget *buttons;
};

ColumbusStatisticsView::ColumbusStatisticsView(QWidget *parent)
    : ApplicationView(parent)
{
    this->d = new ColumbusStatisticsViewPrivate;

    d->controller = NULL;
    d->details = NULL;
    d->distance = NULL;
    d->averageSpeed = NULL;
    d->maximumSpeed = NULL;
    d->minimumAltitude = NULL;
    d->maximumAltitude = NULL;
    d->travelTime = NULL;
    d->elapsedTime = NULL;
    d->speedGraph = NULL;
    d->altitudeProfile = NULL;
    d->accuracyGraph = NULL;

    this->setupUi();

    QObject::connect(parent, SIGNAL(onLandscape()), this, SLOT(onLandscape()));
    QObject::connect(parent, SIGNAL(onPortrait()), this, SLOT(onPortrait()));
}

QString ColumbusStatisticsView::friendlyName() const
{
    static const QString friendlyName = "Statistics";
    return friendlyName;
}

void ColumbusStatisticsView::setController(ApplicationController *controller)
{
    if(d->controller != NULL)
    {
        QObject::disconnect(d->controller->model(), SIGNAL(updated()), this, SLOT(updateUi()));
    }

    if(controller != NULL)
    {
        d->speedGraph->setModel(controller->model());
        d->speedGraph->update();

        d->altitudeProfile->setColumbusModel(controller->model());
        d->altitudeProfile->update();

        d->accuracyGraph->setModel(controller->model());
        d->accuracyGraph->update();

        QObject::connect(controller->model(), SIGNAL(updated()), this, SLOT(updateUi()));
    }

    d->controller = controller;
}

void ColumbusStatisticsView::setupUi()
{
    QBoxLayout  *mbox = new QBoxLayout(QBoxLayout::LeftToRight, this);
    QVBoxLayout *vbox1 = new QVBoxLayout();
    QVBoxLayout *vbox2 = new QVBoxLayout();

    d->details = new QFormLayout();

    d->details->setFieldGrowthPolicy(QFormLayout::FieldsStayAtSizeHint);
    d->details->setLabelAlignment(Qt::AlignRight | Qt::AlignVCenter);
    d->details->setContentsMargins(0, 0, 0, 0);

    d->distance = ValueLabel("-");
    d->averageSpeed = ValueLabel("-");
    d->maximumSpeed = ValueLabel("-");
    d->minimumAltitude = ValueLabel("-");
    d->maximumAltitude = ValueLabel("-");
    d->travelTime = ValueLabel("-");
    d->elapsedTime = ValueLabel("-");

    d->details->addRow(tr("Distance"), d->distance);
    d->details->addRow(tr("Average Speed"), d->averageSpeed);
    d->details->addRow(tr("Maximum Speed"), d->maximumSpeed);
    d->details->addRow(tr("Min. Altitude"), d->minimumAltitude);
    d->details->addRow(tr("Max. Altitude"), d->maximumAltitude);
    d->details->addRow(tr("Travel Time"), d->travelTime);
    d->details->addRow(tr("Elapsed Time"), d->elapsedTime);

    vbox1->addSpacerItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding));
    vbox1->addLayout(d->details);
    vbox1->addSpacerItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding));

    d->speedGraph = new SpeedGraph(this);
    d->altitudeProfile = new AltitudeProfileGraph(this);
    d->accuracyGraph = new AccuracyGraph(this);

    vbox2->addSpacing(10);
    vbox2->addWidget(d->speedGraph);
    vbox2->addWidget(d->altitudeProfile);
    vbox2->addWidget(d->accuracyGraph);
    vbox2->addSpacing(10);

    mbox->addLayout(vbox1);
    mbox->addLayout(vbox2);
}

void ColumbusStatisticsView::updateUi()
{
    if(d->controller == NULL) return;

    ColumbusModel *model = d->controller->model();

    d->distance->setText(QString::number(model->distance(), 'f', 2) + " <sub>" + model->distanceUnits() + "</sub>");
    d->averageSpeed->setText(QString::number(model->averageSpeed(), 'f', 2) + " <sub>" + model->speedUnits() + "</sub>");
    d->maximumSpeed->setText(QString::number(model->maximumSpeed(), 'f', 2) + " <sub>" + model->speedUnits() + "</sub>");
    d->maximumAltitude->setText(QString::number(model->maximumAltitude(), 'f', 2) + " <sub>" + model->altitudeUnits() + "</sub>");
    d->minimumAltitude->setText(QString::number(model->minimumAltitude(), 'f', 2) + " <sub>" + model->altitudeUnits() + "</sub>");
    d->travelTime->setText(QTime(0, 0, 0).addSecs(model->travelTime()).toString());
    d->elapsedTime->setText(QTime(0, 0, 0).addSecs(model->elapsedTime()).toString());

    d->speedGraph->setLabel(tr("SpeedGraph") + " (" + model->speedUnits() + ")");
    d->speedGraph->update();
    d->altitudeProfile->setLabel(tr("Elevation Profile") + " (" + model->altitudeUnits() + ")");
    d->altitudeProfile->update();
    d->accuracyGraph->update();
}

void ColumbusStatisticsView::onLandscape()
{
    QBoxLayout  *mbox = qobject_cast<QBoxLayout*>(this->layout());
    mbox->setDirection(QBoxLayout::LeftToRight);
}

void ColumbusStatisticsView::onPortrait()
{
    QBoxLayout  *mbox = qobject_cast<QBoxLayout*>(this->layout());
    mbox->setDirection(QBoxLayout::TopToBottom);
}

QLabel* ColumbusStatisticsView::ValueLabel(const QString &text)
{
    QLabel  *result = new QLabel(text, this);
    QPalette palette = result->palette();

    palette.setColor(result->foregroundRole(), QColor(0xff, 0xff, 0xff));

    result->setAlignment(Qt::AlignLeft);
    result->setPalette(palette);

    return result;
}
