#include "ConfigManager.h"

#include <QSettings>

class ConfigManagerPrivate
{
public:
    QList<QString>              keys;       // Ordered keys for presentation.
    QHash<QString,QString>      labels;     // Friendly names of options.
    QHash<QString,QVariant>     defaults;   // Default values for options.
    QHash<QString,QStringList>  options;    // Options for option options.
};

ConfigManager::ConfigManager(QObject *parent)
    : QObject(parent)
{
    this->d = new ConfigManagerPrivate;
}

ConfigManager::~ConfigManager()
{
    delete this->d;
}

QList<QString> ConfigManager::keys() const
{
    return d->keys;
}

QString ConfigManager::label(const QString &key) const
{
    return d->keys.contains(key) ? d->labels.value(key) : QString();
}

QVariant ConfigManager::value(const QString &key) const
{
    if(!d->keys.contains(key)) return QVariant(NULL);

    QVariant value = QSettings().value(key, d->defaults.value(key));

    if(this->isOption(key))
    {
        return d->options.value(key).at(value.toInt());
    }

    return value;
}

int ConfigManager::valueIndex(const QString &key) const
{
    if(!d->keys.contains(key) || !this->isOption(key)) return -1;
    QVariant value = QSettings().value(key, d->defaults.value(key));
    return value.toInt();
}

void ConfigManager::setValue(const QString &key, QVariant value)
{
    if(this->isOption(key))
    {
        value = this->options(key).indexOf(value.toString());
    }

    QSettings().setValue(key, value);

    emit this->updated(key, value);
}

void ConfigManager::addToggle(const QString &key, const QString &label, bool initial)
{
    d->keys.append(key);
    d->labels.insert(key, label);
    d->defaults.insert(key, initial);
}

void ConfigManager::addOption(const QString &key, const QString &label, const QStringList &options, int initial)
{
    d->keys.append(key);
    d->labels.insert(key, label);
    d->options.insert(key, options);
    d->defaults.insert(key, initial);
}

bool ConfigManager::isToggle(const QString &key) const
{
    return !this->isOption(key);
}

bool ConfigManager::isOption(const QString &key) const
{
    return d->options.contains(key);
}

QStringList ConfigManager::options(const QString &key) const
{
    return d->options.value(key);
}
