#include "USBSerialPlugin.h"

#include "USBSerialConfig.h"
#include "USBSerialOptionsDialog.h"

#include "SerialPort.h"
#include "plugins/generic-server/ServicePerformerFactory.h"

#include <QDebug>
#include <QPalette>
#include <QtMaemo5>

class USBSerialPluginPrivate
{
public:
    USBSerialConfig *config;
    PluginManager *manager;

    QMaemo5ValueButton *control;
    USBSerialOptionsDialog *dialog;

    SerialPort *port;
    ServicePerformer *performer;
    ServicePerformerFactory *factory;
};

USBSerialPlugin::USBSerialPlugin(QObject *parent)
    : ColumbusPlugin(parent)
{
    qDebug() << "USBSerialPlugin: ctor()";
    this->d = new USBSerialPluginPrivate;
    d->config = new USBSerialConfig(this);
    d->manager = NULL;
    d->control = NULL;
    d->dialog  = NULL;
    d->port = NULL;
    d->performer = NULL;
    d->factory = NULL;
}

USBSerialPlugin::~USBSerialPlugin()
{
    qDebug() << "USBSerialPlugin: dtor()";
    this->shutdown();
    delete this->d;
}

QString USBSerialPlugin::id() const
{
    static const QString id = "usb-serial";
    return id;
}

QString USBSerialPlugin::version() const
{
    static const QString version = "0.1.0";
    return version;
}

QString USBSerialPlugin::friendlyName() const
{
    static const QString friendlyName = tr("USB Serial Profile");
    return friendlyName;
}

QWidget* USBSerialPlugin::control()
{
    return d->control;
}

bool USBSerialPlugin::initialize(PluginManager *manager)
{
    qDebug() << "USBSerialPlugin: Initializing plugin.";
    d->manager = manager;

    d->control = new QMaemo5ValueButton(this->friendlyName());
    d->control->setValueText(tr("Disabled"));

    d->dialog = new USBSerialOptionsDialog(d->config, d->manager->controller()->logger(), d->control);

    QObject::connect(d->control, SIGNAL(clicked()), d->dialog, SLOT(show()));
    QObject::connect(d->dialog, SIGNAL(finished(int)), this, SLOT(onDialogFinished()));

    if(d->config->isEnabled() && d->config->isRunOnStartup())
    {
        qDebug() << "BluetoothServerPlugin: Automatically starting up service.";
        this->startup();
    }

    return true;
}

void USBSerialPlugin::enablePhonetBridge()
{
    qDebug() << "USBSerialPlugin: Enabling Phonet AT bridge.";
    ::system("sudo /sbin/start phonet-at");
}

void USBSerialPlugin::disablePhonetBridge()
{
    qDebug() << "USBSerialPlugin: Disabling Phonet AT bridge.";
    ::system("sudo /sbin/stop phonet-at");
}

bool USBSerialPlugin::startup()
{
    qDebug() << "BluetoothServerPlugin: Starting up services.";

#if defined(Q_WS_MAEMO_5)
    // Port 2 is the same port that phonet-at uses on the N900, so we need to disasble it.
    if(d->config->portNumber() == 2)
    {
        this->disablePhonetBridge();
    }
#endif

    qDebug() << "USBSerialPlugin: Creating performer.";
    d->port = new SerialPort(this);

    if(!d->port->open(QString("/dev/ttyGS%1").arg(d->config->portNumber()), QIODevice::WriteOnly))
    {
        d->control->setValueText("Could not open serial device");
        return false;
    }

    AbstractDataFormatter *formatter = d->manager->controller()->logger()->formatters().value(d->config->formatId());
    if(formatter == NULL)
    {
        qWarning() << "USBSerialPlugin: Unknown logging format selected!" << d->config->formatId();
        return false;
    }

    qDebug() << "USBSerialPlugin: Constructing performer factory for format" << formatter->id() << "data.";
    d->factory = new ServicePerformerFactory(formatter, d->manager->controller()->logger(), this);

    d->performer = d->factory->createInstance(d->port, this);
    d->control->setValueText("Streaming to port " + d->port->deviceName());

    return true;
}

void USBSerialPlugin::shutdown()
{
    qDebug() << "USBSerialPlugin: Shutting down services.";

    if(d->factory != NULL)
    {
        qDebug() << "USBSerialPlugin: Destroying redundant performer factory.";
        delete d->factory;
        d->factory = NULL;
    }

    if(d->performer != NULL)
    {
        qDebug() << "USBSerialPlugin: Destroying redunent performer.";
        delete d->performer;
        d->performer = NULL;
    }

    if(d->port != NULL)
    {
        qDebug() << "USBSerialPlugin: Destroying redundant transport.";
        delete d->port;
        d->port = NULL;

#if defined(Q_WS_MAEMO_5)
        if(d->config->portNumber() == 2)
        {
            qDebug() << "USBSerialPlugin: Enabling Phonet AT bridge.";
            this->enablePhonetBridge();
        }
#endif
    }
}

void USBSerialPlugin::onDialogFinished()
{
    qDebug() << "USBSerialPlugin: Processing configuration.";

    this->shutdown();

    if(!d->config->isEnabled())
    {
        qDebug() << "USBSerialPlugin: Doing nothing, service is not enabled.";
        d->control->setValueText(tr("Disabled"));
        return;
    }

    this->startup();
}
