#include "NetworkOptionsDialog.h"

#include <QSettings>

#include <QtGui>
#include <QtMaemo5>

class NetworkOptionsDialogPrivate
{
public:
    LoggingManager *logger;
    NetworkServerConfig *config;

    QScrollArea *scrollarea;
    QBoxLayout *ibox;

    QCheckBox *isEnabled;
    QCheckBox *isRunOnStartup;

    QMaemo5ValueButton *transport;
    QMaemo5ListPickSelector *transportSelector;

    QMaemo5ValueButton *connectionType;
    QMaemo5ListPickSelector *connectionTypeSelector;

    QStringListModel *udpConnectionTypeModel;
    QStringListModel *tcpConnectionTypeModel;

    QLineEdit *address;
    QLineEdit *port;

    QMaemo5ValueButton *format;
    QMaemo5ListPickSelector *formatSelector;
};

NetworkOptionsDialog::NetworkOptionsDialog(NetworkServerConfig *config, LoggingManager *manager, QWidget *parent)
    : Maemo5Dialog(parent)
{
    new QVBoxLayout(this);

    this->d = new NetworkOptionsDialogPrivate;

    d->config = config;
    d->logger = manager;

    this->setupUi(manager);
    this->restoreConfig();
    this->updateUi();

    QObject::connect(this, SIGNAL(finished(int)), this, SLOT(updateConfig()));
}

NetworkOptionsDialog::~NetworkOptionsDialog()
{
    delete this->d;
}

void NetworkOptionsDialog::setupUi(LoggingManager *manager)
{
    QStringList       fStrings = QStringList();

    foreach(AbstractDataFormatter *formatter, manager->formatters())
    {
        fStrings.append(formatter->friendlyName());
    }

    qSort(fStrings);

    QStringListModel *transportModel = new QStringListModel(QStringList() << "UDP" << "TCP", this);
    QStringListModel *formatModel = new QStringListModel(fStrings, this);

    d->scrollarea = new QScrollArea(this);
    d->scrollarea->setWidgetResizable(true);
    d->scrollarea->setWidget(new QWidget(this));

    QVBoxLayout *vbox = new QVBoxLayout(d->scrollarea->widget());
    d->ibox = new QBoxLayout(QBoxLayout::LeftToRight);

    d->transportSelector = new QMaemo5ListPickSelector(this);
    d->transportSelector->setModel(transportModel);

    d->udpConnectionTypeModel = new QStringListModel(QStringList() << tr("Unicast") << tr("Multicast"), this);
    d->tcpConnectionTypeModel = new QStringListModel(QStringList() << tr("Client") << tr("Server"), this);

    d->connectionTypeSelector = new QMaemo5ListPickSelector(this);
    d->connectionTypeSelector->setModel(d->udpConnectionTypeModel);

    d->formatSelector = new QMaemo5ListPickSelector(this);
    d->formatSelector->setModel(formatModel);

    d->isEnabled = new QCheckBox(tr("Enabled"), this);
    d->isRunOnStartup = new QCheckBox(tr("On Startup"), this);

    d->transport = new QMaemo5ValueButton(tr("Transport"), this);
    //d->transport->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->transport->setPickSelector(d->transportSelector);

    d->connectionType = new QMaemo5ValueButton(tr("Connection Type"), this);
    //d->connectionType->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->connectionType->setPickSelector(d->connectionTypeSelector);

    d->address = new QLineEdit(this);
    d->address->setInputMethodHints(Qt::ImhFormattedNumbersOnly);
    d->address->setPlaceholderText("IP Address");

    d->port = new QLineEdit(this);
    d->port->setInputMethodHints(Qt::ImhDigitsOnly);

    d->format = new QMaemo5ValueButton(tr("Format"), this);
    //d->format->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->format->setPickSelector(d->formatSelector);

    d->ibox->addWidget(d->address, 1);
    d->ibox->addWidget(d->port, 0);

    vbox->addWidget(d->isEnabled);
    vbox->addWidget(d->isRunOnStartup);
    vbox->addWidget(d->transport);
    vbox->addWidget(d->connectionType);
    vbox->addLayout(d->ibox);
    vbox->addWidget(d->format);

    this->layout()->addWidget(d->scrollarea);
    this->setWindowTitle(tr("Network"));

    QObject::connect(d->isEnabled, SIGNAL(toggled(bool)), this, SLOT(updateUi()));
    QObject::connect(d->transportSelector, SIGNAL(selected(QString)), this, SLOT(updateUi()));
    QObject::connect(d->connectionTypeSelector, SIGNAL(selected(QString)), this, SLOT(updateUi()));
}

void NetworkOptionsDialog::updateUi()
{
    qDebug() << "NetworkOptionsDialog: Performing UI update.";
    d->isRunOnStartup->setEnabled(d->isEnabled->isChecked());
    d->transport->setEnabled(d->isEnabled->isChecked());
    d->address->setEnabled(d->isEnabled->isChecked());
    d->port->setEnabled(d->isEnabled->isChecked());
    d->format->setEnabled(d->isEnabled->isChecked());

    // TODO: REMOVE WHEN MULTICAST IS IMPLEMENTED
    d->connectionType->setEnabled(d->isEnabled->isChecked() && d->transport->valueText().toLower() == "tcp");

    if(d->transport->valueText().toLower() == "udp")
    {
        if(QObject::sender() == d->transportSelector)
        {
            d->connectionTypeSelector->setModel(d->udpConnectionTypeModel);
            d->connectionTypeSelector->setCurrentIndex((int)d->config->udpConnectionType());
        }

        switch((NetworkServerConfig::UdpConnectionType)d->transportSelector->currentIndex())
        {
        case NetworkServerConfig::UdpUnicast:
            d->address->setText(d->config->udpUnicastAddress());
            d->port->setText(QString::number(d->config->udpUnicastPort()));
            break;

        case NetworkServerConfig::UdpMulticast:
            d->address->setText(d->config->udpMulticastAddress());
            d->port->setText(QString::number(d->config->udpMulticastPort()));
            break;
        }
    }
    else
    {
        if(QObject::sender() == d->transportSelector)
        {
            d->connectionTypeSelector->setModel(d->tcpConnectionTypeModel);
            d->connectionTypeSelector->setCurrentIndex((int)d->config->tcpConnectionType());
        }

        switch((NetworkServerConfig::TcpConnectionType)d->transportSelector->currentIndex())
        {
        case NetworkServerConfig::TcpClient:
            d->address->setText(d->config->tcpClientAddress());
            d->port->setText(QString::number(d->config->tcpClientPort()));
            break;

        case NetworkServerConfig::TcpServer:
            d->address->setText(d->config->tcpServerAddress());
            d->port->setText(QString::number(d->config->tcpServerPort()));
            break;
        }
    }
}

void NetworkOptionsDialog::restoreConfig()
{
    qDebug() << "NetworkOptionsDialog: Restoring configuration.";
    d->isEnabled->setChecked(d->config->isEnabled() && d->config->isRunOnStartup());
    d->isRunOnStartup->setChecked(d->config->isRunOnStartup());
    d->transportSelector->setCurrentIndex(d->config->transport());
    d->formatSelector->setCurrentIndex(0);
}

void NetworkOptionsDialog::updateConfig()
{
    qDebug() << "NetworkOptionsDialog: Updating configuration.";

    d->config->setEnabled(d->isEnabled->isChecked());
    d->config->setRunOnStartup(d->isRunOnStartup->isChecked());
    d->config->setTransport((NetworkServerConfig::Transport)d->transportSelector->currentIndex());

    if(d->config->transport() == NetworkServerConfig::TransportUdp)
    {
        d->config->setUdpConnectionType((NetworkServerConfig::UdpConnectionType)d->connectionTypeSelector->currentIndex());

        if(d->config->udpConnectionType() == NetworkServerConfig::UdpUnicast)
        {
            d->config->setUdpUnicastAddress(d->address->text());
            d->config->setUdpUnicastPort(d->port->text().toInt());
        }
        else if(d->config->udpConnectionType() == NetworkServerConfig::UdpMulticast)
        {
            d->config->setUdpMulticastAddress(d->address->text());
            d->config->setUdpMulticastPort(d->port->text().toInt());
        }
    }
    else
    {
        d->config->setTcpConnectionType((NetworkServerConfig::TcpConnectionType)d->connectionTypeSelector->currentIndex());

        if(d->config->tcpConnectionType() == NetworkServerConfig::TcpClient)
        {
            d->config->setTcpClientAddress(d->address->text());
            d->config->setTcpClientPort(d->port->text().toInt());
        }
        else if(d->config->tcpConnectionType() == NetworkServerConfig::TcpServer)
        {
            d->config->setTcpServerAddress(d->address->text());
            d->config->setTcpServerPort(d->port->text().toInt());
        }
    }

    foreach(AbstractDataFormatter *format, d->logger->formatters().values())
    {
        if(format->friendlyName() == d->format->valueText())
        {
            d->config->setFormatId(format->id());
            break;
        }
    }
}
