#include "BluetoothConnector.h"
#include "BluetoothSocket.h"

#include <QDebug>
#include <QSocketNotifier>

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>

#include <sys/socket.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>

class BluetoothConnectorPrivate
{
public:
    QString bdaddr;
    quint16 channel;

    BluetoothSocket *socket;
    LoggingServicePerformerFactory *factory;
    LoggingServicePerformer *performer;
};

BluetoothConnector::BluetoothConnector(const QString &bdaddr, quint16 channel, LoggingServicePerformerFactory *factory, QObject *parent)
    : BluetoothServiceProvider(parent)
{
    qDebug() << "BluetoothConnector: ctor()";
    this->d = new BluetoothConnectorPrivate;

    d->bdaddr  = bdaddr;
    d->channel = channel;
    d->socket = NULL;
    d->factory = factory;
    d->performer = NULL;
}

BluetoothConnector::~BluetoothConnector()
{
    qDebug() << "BluetoothConnector: dtor()";

    if(d->socket != NULL)
    {
        qDebug() << "BluetoothConnector: Disconnecting from remote.";
        d->socket->close();
    }

    delete this->d;
}

bool BluetoothConnector::initialize()
{
    qDebug() << "BluetoothConnector: Initializing connection.";

    d->socket = new BluetoothSocket(this);

    QObject::connect(d->socket, SIGNAL(connected()), this, SLOT(onConnect()));
    QObject::connect(d->socket, SIGNAL(disconnected()), this, SLOT(onDisconnect()));

    if(!d->socket->connectToHost(d->bdaddr, d->channel))
    {
        emit this->statusChanged("Failed to connect");
        return false;
    }

    emit this->statusChanged("Attempting to connect.");
    return true;
}

void BluetoothConnector::onConnect()
{
    qDebug() << "BluetoothConnector: Connected to remote, creating service performer.";
    d->performer = d->factory->createInstance(d->socket, this);
    emit this->statusChanged(tr("Connected to %1 (Channel %2)").arg(d->bdaddr, QString::number(d->channel)));
}

void BluetoothConnector::onDisconnect()
{
    qDebug() << "BluetoothConnector: Disconnected from remote, destroying service performer.";
    d->factory->destroyInstance(d->performer);
    d->performer = NULL;
    emit this->statusChanged(tr("Disconnected"));
}
